# frontend/components/ventalibre/roi_tracker_card.py
"""
ROI Tracker Widget Component (Issue #514)

Componente que muestra el ROI del mes actual en el dashboard de venta libre.

Estructura de 3 secciones:
1. Headline: ROI % grande + net EUR
2. Breakdown: Grafico de barras por tipo de accion
3. Actions: Acciones pendientes + CTA

Incluye:
- Indicador de tendencia vs mes anterior
- Tasa de ejecucion
- Link a lista de acciones pendientes
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html

from components.base import BaseCard
from styles.design_tokens import BORDER_RADIUS, COLORS, SHADOWS, SPACING
from utils.helpers import format_currency, format_percentage


def create_roi_tracker_card(
    widget_data: Optional[Dict[str, Any]] = None,
    loading: bool = False,
    error: Optional[str] = None,
) -> dbc.Card:
    """
    Crea la tarjeta de ROI Tracker.

    Args:
        widget_data: Respuesta de /roi-tracker/widget endpoint
        loading: True mientras carga desde API
        error: Mensaje de error si falla

    Returns:
        dbc.Card con el widget ROI
    """
    if loading:
        return _create_loading_card()

    if error:
        return _create_error_card(error)

    if not widget_data:
        return _create_empty_card()

    return dbc.Card(
        [
            # Header
            dbc.CardHeader(
                html.Div(
                    [
                        html.Div(
                            [
                                html.I(
                                    className="fas fa-chart-line me-2",
                                    style={"color": COLORS["success"]},
                                ),
                                html.Span(
                                    "ROI del Mes",
                                    style={"fontWeight": "600"},
                                ),
                            ],
                            className="d-flex align-items-center",
                        ),
                        # Badge mes actual
                        dbc.Badge(
                            widget_data.get("current_month", ""),
                            color="light",
                            text_color="dark",
                            className="ms-auto",
                        ),
                    ],
                    className="d-flex align-items-center",
                ),
            ),
            dbc.CardBody(
                [
                    # Seccion 1: ROI Headline
                    _create_roi_headline(widget_data),
                    html.Hr(className="my-3"),
                    # Seccion 2: Breakdown por tipo
                    _create_breakdown_section(widget_data),
                    html.Hr(className="my-3"),
                    # Seccion 3: Acciones pendientes + CTA
                    _create_actions_section(widget_data),
                ],
            ),
        ],
        className="mb-4 shadow-sm",
        style={"borderRadius": BORDER_RADIUS["lg"]},
    )


def _create_roi_headline(widget_data: Dict[str, Any]) -> html.Div:
    """Seccion principal con ROI % y tendencia."""
    roi_pct = widget_data.get("roi_percentage")
    net_roi = widget_data.get("net_roi_eur", 0)
    trend = widget_data.get("roi_trend")
    trend_direction = widget_data.get("trend_direction")
    execution_rate = widget_data.get("execution_rate", 0)

    # Color segun ROI
    if roi_pct is None:
        roi_color = COLORS["secondary"]
        roi_text = "Sin datos"
    elif roi_pct >= 500:
        roi_color = COLORS["success"]
        roi_text = f"{roi_pct:.0f}%"
    elif roi_pct >= 100:
        roi_color = COLORS["info"]
        roi_text = f"{roi_pct:.0f}%"
    else:
        roi_color = COLORS["warning"]
        roi_text = f"{roi_pct:.0f}%"

    # Icono de tendencia
    if trend_direction == "up":
        trend_icon = "fas fa-arrow-up"
        trend_color = COLORS["success"]
        trend_text = f"+{trend:.0f}%"
    elif trend_direction == "down":
        trend_icon = "fas fa-arrow-down"
        trend_color = COLORS["danger"]
        trend_text = f"{trend:.0f}%"
    else:
        trend_icon = "fas fa-minus"
        trend_color = COLORS["secondary"]
        trend_text = "estable"

    return html.Div(
        [
            # ROI grande
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                roi_text,
                                style={
                                    "fontSize": "2.5rem",
                                    "fontWeight": "700",
                                    "color": roi_color,
                                    "lineHeight": "1",
                                },
                            ),
                            html.Small(
                                "Retorno sobre suscripcion",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            # ROI neto
                            html.Div(
                                format_currency(net_roi),
                                style={
                                    "fontSize": "1.5rem",
                                    "fontWeight": "600",
                                    "color": COLORS["dark"],
                                },
                            ),
                            html.Small(
                                "Beneficio neto",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                        className="text-end",
                    ),
                ],
                className="mb-3",
            ),
            # Tendencia y tasa de ejecucion
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.I(
                                className=f"{trend_icon} me-1",
                                style={"color": trend_color},
                            ),
                            html.Span(
                                f"{trend_text} vs mes anterior",
                                style={"fontSize": "0.85rem"},
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            html.I(
                                className="fas fa-check-circle me-1",
                                style={"color": COLORS["info"]},
                            ),
                            html.Span(
                                f"{execution_rate:.0f}% ejecutadas",
                                style={"fontSize": "0.85rem"},
                            ),
                        ],
                        width=6,
                        className="text-end",
                    ),
                ],
            ),
        ],
    )


def _create_breakdown_section(widget_data: Dict[str, Any]) -> html.Div:
    """Seccion con breakdown por tipo de accion."""
    top_types = widget_data.get("top_action_types", [])

    if not top_types:
        return html.Div(
            html.Small(
                "Sin acciones ejecutadas este mes",
                className="text-muted",
            ),
            className="text-center py-2",
        )

    # Crear mini grafico de barras horizontal
    labels = []
    values = []
    colors = {
        "liquidation": COLORS["warning"],
        "restock": COLORS["success"],
        "pricing": COLORS["info"],
        "diversify": COLORS["primary"],
    }

    type_names = {
        "liquidation": "Liquidacion",
        "restock": "Reposicion",
        "pricing": "Precios",
        "diversify": "Diversificar",
    }

    for item in top_types:
        action_type = item.get("type", "")
        labels.append(type_names.get(action_type, action_type))
        values.append(item.get("impact", 0))

    fig = go.Figure(
        go.Bar(
            x=values,
            y=labels,
            orientation="h",
            marker_color=[
                colors.get(t.get("type", ""), COLORS["secondary"])
                for t in top_types
            ],
            text=[format_currency(v) for v in values],
            textposition="outside",
            textfont={"size": 11},
        )
    )

    fig.update_layout(
        margin=dict(l=0, r=0, t=10, b=0),
        height=120,
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
        showlegend=False,
        xaxis=dict(showgrid=False, showticklabels=False, zeroline=False),
        yaxis=dict(showgrid=False, tickfont=dict(size=11)),
    )

    return html.Div(
        [
            html.Small(
                "Impacto por tipo de accion",
                className="text-muted mb-2 d-block",
            ),
            dcc.Graph(
                figure=fig,
                config={"displayModeBar": False, "responsive": True},
                style={"height": "120px"},
            ),
        ],
    )


def _create_actions_section(widget_data: Dict[str, Any]) -> html.Div:
    """Seccion con acciones pendientes y CTA."""
    pending_count = widget_data.get("pending_actions", 0)
    opportunity = widget_data.get("total_opportunity_eur", 0)

    if pending_count == 0:
        return html.Div(
            [
                html.I(
                    className="fas fa-check-circle me-2",
                    style={"color": COLORS["success"]},
                ),
                html.Span(
                    "Todas las acciones han sido procesadas",
                    style={"fontSize": "0.9rem"},
                ),
            ],
            className="text-center py-2",
        )

    return html.Div(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Badge(
                                f"{pending_count}",
                                color="warning",
                                className="me-2",
                                style={"fontSize": "1rem"},
                            ),
                            html.Span(
                                "acciones pendientes",
                                style={"fontSize": "0.9rem"},
                            ),
                        ],
                        width=7,
                    ),
                    dbc.Col(
                        [
                            html.Small(
                                f"Oportunidad: {format_currency(opportunity)}",
                                className="text-muted",
                            ),
                        ],
                        width=5,
                        className="text-end",
                    ),
                ],
                className="mb-2",
            ),
            dbc.Button(
                [
                    html.I(className="fas fa-tasks me-2"),
                    "Ver Acciones",
                ],
                id="roi-tracker-view-actions-btn",
                color="primary",
                outline=True,
                size="sm",
                className="w-100",
            ),
        ],
    )


def _create_loading_card() -> dbc.Card:
    """Tarjeta en estado de carga."""
    return dbc.Card(
        [
            dbc.CardHeader(
                html.Div(
                    [
                        html.I(
                            className="fas fa-chart-line me-2",
                            style={"color": COLORS["secondary"]},
                        ),
                        html.Span("ROI del Mes", style={"fontWeight": "600"}),
                    ],
                    className="d-flex align-items-center",
                ),
            ),
            dbc.CardBody(
                [
                    dbc.Spinner(
                        color="primary",
                        type="border",
                        size="md",
                    ),
                    html.P(
                        "Calculando ROI...",
                        className="text-muted mt-2 mb-0 text-center",
                    ),
                ],
                className="text-center py-4",
            ),
        ],
        className="mb-4 shadow-sm",
        style={"borderRadius": BORDER_RADIUS["lg"]},
    )


def _create_error_card(error: str) -> dbc.Card:
    """Tarjeta en estado de error."""
    return dbc.Card(
        [
            dbc.CardHeader(
                html.Div(
                    [
                        html.I(
                            className="fas fa-chart-line me-2",
                            style={"color": COLORS["danger"]},
                        ),
                        html.Span("ROI del Mes", style={"fontWeight": "600"}),
                    ],
                    className="d-flex align-items-center",
                ),
            ),
            dbc.CardBody(
                [
                    html.I(
                        className="fas fa-exclamation-triangle fa-2x mb-2",
                        style={"color": COLORS["warning"]},
                    ),
                    html.P(
                        error,
                        className="text-muted mb-0",
                    ),
                ],
                className="text-center py-3",
            ),
        ],
        className="mb-4 shadow-sm",
        style={"borderRadius": BORDER_RADIUS["lg"]},
    )


def _create_empty_card() -> dbc.Card:
    """Tarjeta cuando no hay datos."""
    return dbc.Card(
        [
            dbc.CardHeader(
                html.Div(
                    [
                        html.I(
                            className="fas fa-chart-line me-2",
                            style={"color": COLORS["secondary"]},
                        ),
                        html.Span("ROI del Mes", style={"fontWeight": "600"}),
                    ],
                    className="d-flex align-items-center",
                ),
            ),
            dbc.CardBody(
                [
                    html.I(
                        className="fas fa-info-circle fa-2x mb-2",
                        style={"color": COLORS["info"]},
                    ),
                    html.P(
                        "Ejecuta acciones sugeridas para comenzar a medir tu ROI",
                        className="text-muted mb-0",
                    ),
                ],
                className="text-center py-3",
            ),
        ],
        className="mb-4 shadow-sm",
        style={"borderRadius": BORDER_RADIUS["lg"]},
    )


def create_roi_actions_modal() -> dbc.Modal:
    """
    Modal para ver y gestionar acciones pendientes.

    Muestra lista de acciones con opciones para:
    - Marcar como ejecutada
    - Descartar
    - Posponer
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(
                    [
                        html.I(
                            className="fas fa-tasks me-2",
                            style={"color": COLORS["primary"]},
                        ),
                        "Acciones Pendientes",
                    ]
                ),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    # Container para la lista de acciones (se llena dinámicamente)
                    dcc.Loading(
                        id="roi-tracker-actions-loading",
                        type="default",
                        children=html.Div(
                            id="roi-tracker-actions-list",
                            children=html.P(
                                "Cargando acciones...",
                                className="text-muted text-center py-3",
                            ),
                        ),
                    ),
                ],
            ),
            dbc.ModalFooter(
                [
                    html.Small(
                        [
                            html.I(className="fas fa-info-circle me-1"),
                            "Ejecuta acciones para mejorar tu ROI",
                        ],
                        className="text-muted me-auto",
                    ),
                    dbc.Button(
                        "Cerrar",
                        id="roi-tracker-modal-close-btn",
                        color="secondary",
                        outline=True,
                    ),
                ],
            ),
        ],
        id="roi-tracker-actions-modal",
        is_open=False,
        centered=True,
        size="lg",
    )
