"""
Tabla de productos venta libre con acciones de feedback (Issue #461, #523)

Muestra productos OTC con clasificación NECESIDAD (L1 + L2), confianza ML,
marca detectada y totales de ventas. Incluye botón de feedback.

Issue #523: Añadida columna ml_subcategory para mostrar subcategoría L2.
"""

from typing import List, Dict

import dash_bootstrap_components as dbc
from dash import html

from components.ventalibre.categories import get_l2_info, has_l2_subcategories
from styles.design_tokens import COLORS
from utils.helpers import format_currency


def create_productos_table(products: List[Dict]) -> html.Div:
    """
    Crear tabla de productos con botón de feedback por fila.

    Args:
        products: Lista de productos con campos:
            - id: ID del producto (para feedback)
            - product_code: Código del producto
            - product_name: Nombre del producto
            - ml_category: Categoría NECESIDAD asignada
            - ml_confidence: Confianza de la clasificación (0-1)
            - detected_brand: Marca detectada
            - total_sales: Ventas totales en euros
            - total_units: Unidades vendidas

    Returns:
        html.Div con tabla Bootstrap
    """
    if not products:
        return html.Div([
            html.Div([
                html.I(className="fas fa-box-open fa-3x text-muted mb-3"),
                html.P("No hay productos para mostrar", className="text-muted"),
                html.Small(
                    "Selecciona una categoría o ajusta los filtros",
                    className="text-muted"
                ),
            ], className="text-center py-4")
        ])

    # Header de la tabla (Issue #523: añadida Subcategoría L2)
    header = html.Thead(
        html.Tr([
            html.Th("Producto", style={"width": "30%"}),
            html.Th("Categoría", style={"width": "15%"}),
            html.Th("Subcategoría", style={"width": "15%"}, className="d-none d-lg-table-cell"),  # L2
            html.Th("Confianza", style={"width": "12%"}),
            html.Th("Ventas", style={"width": "12%"}, className="text-end"),
            html.Th("", style={"width": "6%"}),  # Acciones
        ]),
        className="table-light"
    )

    # Filas de productos
    rows = []
    for product in products:
        # Obtener valores con fallbacks
        product_id = product.get("id", product.get("product_code", ""))
        product_name = product.get("product_name", "?")
        detected_brand = product.get("detected_brand", "")
        ml_category = product.get("ml_category", "sin_clasificar")
        ml_subcategory = product.get("ml_subcategory", "")  # Issue #523: L2 subcategory
        ml_confidence = float(product.get("ml_confidence", 0) or 0)
        total_sales = float(product.get("total_sales", 0) or 0)

        # Determinar color de confianza
        if ml_confidence >= 0.8:
            confidence_color = "success"
        elif ml_confidence >= 0.6:
            confidence_color = "warning"
        else:
            confidence_color = "danger"

        # Formatear categoría para display
        category_label = (ml_category or "sin clasificar").replace("_", " ").title()

        # Issue #523: Formatear subcategoría L2 con info de arquetipos
        if ml_subcategory:
            l2_info = get_l2_info(ml_subcategory)
            subcategory_label = l2_info.get("name", ml_subcategory.replace("_", " ").title())
            subcategory_archetype = l2_info.get("archetype", "")
        else:
            subcategory_label = "-"
            subcategory_archetype = ""

        # Crear fila
        row = html.Tr([
            # Columna: Producto
            html.Td([
                html.Div(
                    product_name[:60] + ("..." if len(product_name) > 60 else ""),
                    className="fw-medium text-truncate",
                    style={"maxWidth": "300px"},
                    title=product_name,
                ),
                html.Small(
                    detected_brand or "-",
                    className="text-muted d-block",
                ),
            ]),

            # Columna: Categoría (L1)
            html.Td(
                dbc.Badge(
                    category_label,
                    color="info",
                    className="text-truncate",
                    style={"maxWidth": "120px"},
                )
            ),

            # Columna: Subcategoría (L2) - Issue #523
            html.Td(
                html.Div([
                    html.Span(
                        subcategory_label,
                        className="small",
                        style={"fontWeight": "500"},
                    ),
                    html.Small(
                        f" {subcategory_archetype}" if subcategory_archetype else "",
                        className="text-muted",
                    ),
                ]) if subcategory_label != "-" else html.Span("-", className="text-muted"),
                className="d-none d-lg-table-cell",  # Oculto en móvil/tablet pequeño
            ),

            # Columna: Confianza
            html.Td([
                dbc.Progress(
                    value=ml_confidence * 100,
                    color=confidence_color,
                    style={"height": "8px"},
                    className="mb-1",
                ),
                html.Small(
                    f"{ml_confidence:.0%}",
                    className="text-muted",
                ),
            ]),

            # Columna: Ventas
            html.Td(
                format_currency(total_sales),
                className="text-end fw-medium",
            ),

            # Columna: Acciones
            html.Td(
                dbc.Button(
                    html.I(className="fas fa-flag"),
                    id={"type": "ventalibre-feedback-btn", "id": str(product_id)},
                    color="link",
                    size="sm",
                    title="Reportar error de clasificación",
                    className="text-warning p-0",
                ),
                className="text-end",
            ),
        ])

        rows.append(row)

    # Construir tabla completa
    table = dbc.Table(
        [header, html.Tbody(rows)],
        striped=True,
        hover=True,
        responsive=True,
        className="mb-0",
        size="sm",
    )

    return html.Div([table])
