# frontend/components/ventalibre/product_search.py
"""
Buscador de productos con autocompletado para Tab 3 (Issue #494).

Componente que permite buscar productos venta libre por CN, nombre o marca.
Prioriza: CN exacto > Nombre starts_with > Nombre contains.
"""

from dash import dcc, html
import dash_bootstrap_components as dbc


def create_product_search() -> html.Div:
    """
    Crea el buscador de productos con autocompletado.

    Returns:
        html.Div con el dropdown searchable y ayuda contextual
    """
    return html.Div([
        dbc.Row([
            dbc.Col([
                html.Label(
                    "Buscar Producto",
                    className="form-label fw-bold mb-2",
                ),
                dcc.Dropdown(
                    id="ventalibre-product-search",
                    placeholder="Escribe CN, nombre o marca (mín. 3 caracteres)...",
                    searchable=True,
                    clearable=True,
                    options=[],
                    value=None,
                    className="mb-2",
                    style={"fontSize": "14px"},
                ),
                html.Small(
                    "Busca por código nacional, nombre del producto o marca.",
                    className="text-muted",
                ),
            ], width=12, lg=8),
            dbc.Col([
                html.Div(
                    id="ventalibre-product-search-status",
                    className="text-end pt-4",
                ),
            ], width=12, lg=4),
        ]),
    ], className="mb-4")


def create_product_search_skeleton() -> html.Div:
    """
    Skeleton loader para el buscador.

    Returns:
        html.Div con placeholder de carga
    """
    return html.Div([
        html.Div(
            className="skeleton-text mb-2",
            style={"width": "120px", "height": "20px"},
        ),
        html.Div(
            className="skeleton-input",
            style={"width": "100%", "height": "38px", "borderRadius": "4px"},
        ),
    ], className="mb-4")


def format_search_option(product: dict) -> dict:
    """
    Formatea un producto para mostrarlo como opción del dropdown.

    Args:
        product: Dict con codigo_nacional, product_name, detected_brand, total_sales

    Returns:
        dict con label y value para dcc.Dropdown
    """
    cn = product.get("codigo_nacional", "")
    name = product.get("product_name", "Desconocido")
    brand = product.get("detected_brand")
    sales = product.get("total_sales", 0)

    # Label con formato: "CN - Nombre (Marca) - €ventas"
    label_parts = [cn, name[:40]]
    if brand:
        label_parts.append(f"({brand})")
    label_parts.append(f"€{sales:,.0f}")

    return {
        "label": " - ".join(label_parts),
        "value": cn,
    }
