# frontend/components/ventalibre/product_ficha.py
"""
Ficha de producto con KPIs para Tab 3 (Issue #494).

Muestra información del producto seleccionado:
- Datos básicos (nombre, marca, categoría)
- KPIs: MAT, GMROI, días cobertura, margen vs categoría
"""

from typing import Optional

from dash import html
import dash_bootstrap_components as dbc


def create_product_ficha(ficha_data: Optional[dict] = None) -> html.Div:
    """
    Crea la ficha de producto con información básica.

    Args:
        ficha_data: Dict con product_code, product_name, detected_brand, ml_category, etc.

    Returns:
        html.Div con la card de información del producto
    """
    if not ficha_data:
        return create_product_ficha_placeholder()

    product_name = ficha_data.get("product_name", "Producto no encontrado")
    product_code = ficha_data.get("product_code", "")
    brand = ficha_data.get("detected_brand") or "Sin marca"
    brand_line = ficha_data.get("brand_line")
    category = ficha_data.get("ml_category") or "Sin categoría"
    subcategory = ficha_data.get("ml_subcategory")

    return dbc.Card([
        dbc.CardHeader([
            html.H5(product_name, className="mb-0 text-truncate"),
        ]),
        dbc.CardBody([
            dbc.Row([
                dbc.Col([
                    html.Div([
                        html.I(className="fas fa-barcode me-2 text-primary"),
                        html.Span("CN: ", className="fw-bold"),
                        html.Span(product_code),
                    ], className="mb-2"),
                    html.Div([
                        html.I(className="fas fa-tag me-2 text-info"),
                        html.Span("Marca: ", className="fw-bold"),
                        dbc.Badge(brand.upper(), color="info", className="ms-1"),
                        html.Span(f" / {brand_line}" if brand_line else "", className="text-muted ms-1"),
                    ], className="mb-2"),
                ], width=12, lg=6),
                dbc.Col([
                    html.Div([
                        html.I(className="fas fa-folder-open me-2 text-success"),
                        html.Span("Necesidad: ", className="fw-bold"),
                        dbc.Badge(
                            category.replace("_", " ").title(),
                            color="success",
                            className="ms-1",
                        ),
                    ], className="mb-2"),
                    html.Div([
                        html.I(className="fas fa-layer-group me-2 text-secondary"),
                        html.Span("Sub: ", className="fw-bold"),
                        html.Span(
                            subcategory.replace("_", " ").title() if subcategory else "—",
                            className="text-muted",
                        ),
                    ], className="mb-2") if subcategory else html.Div(),
                ], width=12, lg=6),
            ]),
        ]),
    ], className="shadow-sm h-100")


def create_product_ficha_placeholder() -> html.Div:
    """
    Placeholder cuando no hay producto seleccionado.

    Returns:
        html.Div con mensaje de selección
    """
    return dbc.Card([
        dbc.CardBody([
            html.Div([
                html.I(className="fas fa-search fa-3x text-muted mb-3"),
                html.P(
                    "Selecciona un producto para ver su ficha",
                    className="text-muted mb-0",
                ),
            ], className="text-center py-4"),
        ]),
    ], className="shadow-sm h-100 bg-light")


def create_product_kpis(kpis_data: Optional[dict] = None) -> html.Div:
    """
    Crea las 4 tarjetas de KPIs del producto.

    Args:
        kpis_data: Dict con mat, gmroi, coverage_days, margin_vs_category, etc.

    Returns:
        html.Div con grid de 4 KPI cards
    """
    if not kpis_data:
        return create_kpis_placeholder()

    mat = kpis_data.get("mat", 0)
    mat_yoy = kpis_data.get("mat_yoy_change")
    gmroi = kpis_data.get("gmroi", 0)
    gmroi_interp = kpis_data.get("gmroi_interpretation", "Sin datos")
    coverage = kpis_data.get("coverage_days")
    coverage_alert = kpis_data.get("coverage_alert_level", "info")
    margin_vs = kpis_data.get("margin_vs_category", 0)

    return dbc.Row([
        # KPI 1: MAT (Moving Annual Total)
        dbc.Col([
            _create_kpi_card(
                title="MAT",
                value=f"€{mat:,.0f}",
                subtitle=_format_yoy_change(mat_yoy),
                icon="fa-chart-line",
                color="primary",
            ),
        ], width=6, lg=3, className="mb-3"),

        # KPI 2: GMROI
        dbc.Col([
            _create_kpi_card(
                title="GMROI",
                value=f"{gmroi:.1f}x",
                subtitle=gmroi_interp,
                icon="fa-percentage",
                color=_gmroi_color(gmroi),
            ),
        ], width=6, lg=3, className="mb-3"),

        # KPI 3: Días Cobertura
        dbc.Col([
            _create_kpi_card(
                title="Cobertura",
                value=f"{coverage:.0f} días" if coverage else "N/A",
                subtitle=_coverage_label(coverage_alert),
                icon="fa-boxes",
                color=_alert_to_color(coverage_alert),
            ),
        ], width=6, lg=3, className="mb-3"),

        # KPI 4: Margen vs Categoría
        dbc.Col([
            _create_kpi_card(
                title="Margen vs Cat.",
                value=f"{margin_vs:+.1f} pp",
                subtitle="vs promedio categoría",
                icon="fa-balance-scale",
                color="success" if margin_vs > 0 else "warning" if margin_vs > -3 else "danger",
            ),
        ], width=6, lg=3, className="mb-3"),
    ])


def _create_kpi_card(
    title: str,
    value: str,
    subtitle: str,
    icon: str,
    color: str,
) -> dbc.Card:
    """Crea una tarjeta de KPI individual."""
    return dbc.Card([
        dbc.CardBody([
            html.Div([
                html.I(className=f"fas {icon} fa-2x text-{color} mb-2"),
            ], className="text-center"),
            html.H4(value, className="text-center mb-1 fw-bold"),
            html.P(title, className="text-center text-muted mb-0 small"),
            html.Small(subtitle, className="d-block text-center text-muted"),
        ], className="py-2"),
    ], className="shadow-sm h-100")


def _format_yoy_change(yoy: Optional[float]) -> str:
    """Formatea cambio YoY con flecha."""
    if yoy is None:
        return "Sin datos año anterior"
    arrow = "↑" if yoy > 0 else "↓" if yoy < 0 else "→"
    return f"{arrow} {abs(yoy):.1f}% vs año ant."


def _gmroi_color(gmroi: float) -> str:
    """Color según valor GMROI."""
    if gmroi >= 3.0:
        return "success"
    if gmroi >= 2.0:
        return "primary"
    if gmroi >= 1.0:
        return "warning"
    return "danger"


def _coverage_label(alert: str) -> str:
    """Etiqueta para nivel de cobertura."""
    labels = {
        "danger": "Stock crítico",
        "warning": "Stock bajo",
        "success": "Stock óptimo",
        "info": "Stock alto",
    }
    return labels.get(alert, "")


def _alert_to_color(alert: str) -> str:
    """Convierte alert level a color Bootstrap."""
    return alert if alert in ("danger", "warning", "success", "info") else "secondary"


def create_kpis_placeholder() -> html.Div:
    """Placeholder para KPIs cuando no hay producto."""
    return dbc.Row([
        dbc.Col([
            dbc.Card([
                dbc.CardBody([
                    html.Div(
                        className="skeleton-circle mx-auto mb-2",
                        style={"width": "40px", "height": "40px"},
                    ),
                    html.Div(
                        className="skeleton-text mx-auto mb-1",
                        style={"width": "60px", "height": "24px"},
                    ),
                    html.Div(
                        className="skeleton-text mx-auto",
                        style={"width": "80px", "height": "16px"},
                    ),
                ], className="py-3"),
            ], className="shadow-sm h-100 bg-light"),
        ], width=6, lg=3, className="mb-3")
        for _ in range(4)
    ])
