"""
Treemap y Barras de ventas por NECESIDAD (Issue #461, #492)

Visualización interactiva de ventas de productos OTC por categoría.
Implementa lógica 80/20: categorías <2% se agregan en "Otras Necesidades".
Toggle entre treemap y barras horizontales (Issue #492).
Referencia: components/prescription/atc_distribution_treemap.py
"""

from typing import Dict, Literal, Optional

import dash_bootstrap_components as dbc
import pandas as pd
import plotly.express as px
import plotly.graph_objects as go
from dash import dcc, html

# Usar colores de categorías principales
from components.ventalibre.categories import PRINCIPAL_CATEGORIES

# Color especial para nodo agregado "Otras Necesidades"
OTRAS_NECESIDADES_COLOR = "#95a5a6"  # Gris distintivo


def create_necesidad_chart(
    sales_data: Dict,
    chart_type: Literal["treemap", "bars"] = "treemap",
    height: int = 400
) -> html.Div:
    """
    Crear visualización de ventas por NECESIDAD (treemap o barras).

    Args:
        sales_data: Dict con datos para el gráfico
        chart_type: "treemap" o "bars"
        height: Altura del gráfico en pixels

    Returns:
        html.Div con el gráfico correspondiente
    """
    if chart_type == "bars":
        return create_necesidad_barchart(sales_data, height)
    return create_necesidad_treemap(sales_data, height)


def create_necesidad_treemap(sales_data: Dict, height: int = 400) -> html.Div:
    """
    Crear treemap interactivo de ventas por NECESIDAD.

    Implementa lógica 80/20: muestra solo categorías principales (≥2% ventas).
    Categorías minoritarias se agregan en "Otras Necesidades" (W14a).

    Args:
        sales_data: Dict con datos para treemap:
            {
                "nodes": [
                    {"category": "proteccion_solar", "sales": 15000, "count": 234, "percentage": 25.3},
                    {"category": "otras_necesidades", "is_aggregated": True, ...},  # Nodo agregado
                ],
                "other_nodes": [...],  # Categorías minoritarias (para tabla colapsible)
                "total_sales": 60000
            }
        height: Altura del gráfico en pixels

    Returns:
        html.Div con dcc.Graph del treemap Plotly
    """
    nodes = sales_data.get("nodes", [])

    if not nodes:
        return html.Div([
            html.Div([
                html.I(className="fas fa-chart-pie fa-3x text-muted mb-3"),
                html.P("No hay datos de ventas disponibles", className="text-muted"),
                html.Small(
                    "Sube archivos de ventas para ver el análisis por NECESIDAD",
                    className="text-muted"
                ),
            ], className="text-center py-5")
        ])

    # Preparar datos para treemap
    df_data = []
    for node in nodes:
        category = node["category"]
        is_aggregated = node.get("is_aggregated", False)
        pending_corrections = node.get("pending_corrections", 0)

        # Formatear label para mostrar
        if is_aggregated:
            label = "Otras Necesidades"
            color = OTRAS_NECESIDADES_COLOR
        else:
            # Usar display_name si existe (para categorías principales)
            label = node.get("display_name") or category.replace("_", " ").title()
            # Usar color de categoría principal
            principal_info = PRINCIPAL_CATEGORIES.get(category, {})
            color = principal_info.get("color", "#6c757d")

        # W14b: Indicador visual de "salud del dato"
        needs_review = pending_corrections > 0

        df_data.append({
            "category": label,
            "category_id": category,
            "sales": node.get("sales", 0),
            "count": node.get("count", 0),
            "percentage": node.get("percentage", 0),
            "color": color,
            "is_aggregated": is_aggregated,
            "pending_corrections": pending_corrections,
            "needs_review": needs_review,
        })

    df = pd.DataFrame(df_data)

    # Crear color map para treemap
    color_map = {row["category"]: row["color"] for _, row in df.iterrows()}

    fig = px.treemap(
        df,
        path=["category"],
        values="sales",
        color="category",
        color_discrete_map=color_map,
        custom_data=["category_id", "count", "percentage", "is_aggregated", "pending_corrections"],
    )

    # Configurar hover y texto
    # W14b: Mostrar indicador de correcciones pendientes en hover
    fig.update_traces(
        hovertemplate=(
            "<b>%{label}</b><br>"
            "Ventas: €%{value:,.2f}<br>"
            "Productos: %{customdata[1]:,}<br>"
            "Porcentaje: %{customdata[2]:.1f}%<br>"
            "Pendientes revisión: %{customdata[4]}"
            "<extra></extra>"
        ),
        texttemplate="<b>%{label}</b><br>€%{value:,.0f}",
        textfont={"size": 12},
    )

    # W14b: Aplicar borde naranja a categorías con correcciones pendientes
    # Crear arrays de colores y anchos de borde basados en needs_review
    line_widths = [3 if row["needs_review"] else 0 for _, row in df.iterrows()]
    line_colors = ["#ff8c00" if row["needs_review"] else "white" for _, row in df.iterrows()]

    # Nota: Plotly treemaps usan marker.line para los bordes
    # El primer elemento del treemap es el root (total), los demás son las categorías
    # Añadir elemento para el root (sin borde)
    line_widths = [0] + line_widths
    line_colors = ["white"] + line_colors

    fig.update_traces(
        marker=dict(
            line=dict(
                width=line_widths,
                color=line_colors
            )
        )
    )

    # Layout limpio
    fig.update_layout(
        margin=dict(t=10, l=10, r=10, b=10),
        height=height,
        showlegend=False,
    )

    return html.Div([
        dcc.Graph(
            id="ventalibre-treemap",
            figure=fig,
            config={
                "displayModeBar": False,
                "responsive": True,
            },
            style={"height": f"{height}px"},
        ),
        # Leyenda de interacción y W14b
        html.Div([
            html.Small([
                html.I(className="fas fa-mouse-pointer me-1"),
                "Clic para filtrar",
            ], className="text-muted me-3"),
            html.Small([
                html.Span(
                    "█",
                    style={"color": "#ff8c00", "marginRight": "4px"}
                ),
                html.Span(
                    [
                        "Requiere revisión",
                        html.I(className="fas fa-info-circle ms-1", style={"fontSize": "0.8em"}),
                    ],
                    id="ventalibre-requiere-revision-help",
                    style={"cursor": "help"},
                ),
            ], className="text-muted"),
        ], className="d-flex justify-content-center mt-2"),
        # Tooltip explicativo para "Requiere revisión" (Issue #490)
        dbc.Tooltip(
            html.Div([
                html.Strong("Productos pendientes de clasificación"),
                html.Br(),
                html.Span("Estos productos no pudieron ser clasificados automáticamente "),
                html.Span("por categoría NECESIDAD y requieren revisión manual."),
                html.Br(),
                html.Small(
                    "El borde naranja indica categorías con productos pendientes.",
                    className="text-warning",
                ),
            ]),
            target="ventalibre-requiere-revision-help",
            placement="top",
        ),
    ])


def create_necesidad_barchart(sales_data: Dict, height: int = 400) -> html.Div:
    """
    Crear gráfico de barras horizontales de ventas por NECESIDAD (Issue #492).

    Alternativa al treemap para comparación exacta entre categorías.
    Las barras están ordenadas de mayor a menor ventas.

    Args:
        sales_data: Dict con datos (mismo formato que treemap)
        height: Altura del gráfico en pixels

    Returns:
        html.Div con dcc.Graph de barras horizontales
    """
    nodes = sales_data.get("nodes", [])

    if not nodes:
        return html.Div([
            html.Div([
                html.I(className="fas fa-chart-bar fa-3x text-muted mb-3"),
                html.P("No hay datos de ventas disponibles", className="text-muted"),
                html.Small(
                    "Sube archivos de ventas para ver el análisis por NECESIDAD",
                    className="text-muted"
                ),
            ], className="text-center py-5")
        ])

    # Preparar datos para barras
    df_data = []
    for node in nodes:
        category = node["category"]
        is_aggregated = node.get("is_aggregated", False)
        pending_corrections = node.get("pending_corrections", 0)

        # Formatear label para mostrar
        if is_aggregated:
            label = "Otras Necesidades"
            color = OTRAS_NECESIDADES_COLOR
        else:
            label = node.get("display_name") or category.replace("_", " ").title()
            principal_info = PRINCIPAL_CATEGORIES.get(category, {})
            color = principal_info.get("color", "#6c757d")

        needs_review = pending_corrections > 0

        df_data.append({
            "category": label,
            "category_id": category,
            "sales": node.get("sales", 0),
            "count": node.get("count", 0),
            "percentage": node.get("percentage", 0),
            "color": color,
            "is_aggregated": is_aggregated,
            "pending_corrections": pending_corrections,
            "needs_review": needs_review,
        })

    df = pd.DataFrame(df_data)
    # Ordenar de mayor a menor
    df = df.sort_values("sales", ascending=True)  # ascending=True para barras horizontales

    # Crear gráfico de barras horizontales
    fig = go.Figure()

    # Añadir barras con colores personalizados
    fig.add_trace(go.Bar(
        y=df["category"],
        x=df["sales"],
        orientation="h",
        marker=dict(
            color=df["color"].tolist(),
            line=dict(
                color=["#ff8c00" if nr else "white" for nr in df["needs_review"]],
                width=[2 if nr else 0 for nr in df["needs_review"]]
            )
        ),
        customdata=df[["category_id", "count", "percentage", "is_aggregated", "pending_corrections"]].values,
        hovertemplate=(
            "<b>%{y}</b><br>"
            "Ventas: €%{x:,.2f}<br>"
            "Productos: %{customdata[1]:,}<br>"
            "Porcentaje: %{customdata[2]:.1f}%<br>"
            "Pendientes revisión: %{customdata[4]}"
            "<extra></extra>"
        ),
        text=[f"€{s:,.0f}" for s in df["sales"]],
        textposition="outside",
        textfont=dict(size=11),
    ))

    # Layout
    fig.update_layout(
        margin=dict(t=10, l=10, r=60, b=10),
        height=height,
        showlegend=False,
        xaxis=dict(
            title="",
            showgrid=True,
            gridcolor="rgba(0,0,0,0.1)",
            tickformat="€,.0f",
        ),
        yaxis=dict(
            title="",
            showgrid=False,
            automargin=True,
        ),
        plot_bgcolor="white",
        bargap=0.2,
    )

    return html.Div([
        dcc.Graph(
            id="ventalibre-treemap",  # Mismo ID para mantener callbacks
            figure=fig,
            config={
                "displayModeBar": False,
                "responsive": True,
            },
            style={"height": f"{height}px"},
        ),
        # Leyenda de interacción y W14b
        html.Div([
            html.Small([
                html.I(className="fas fa-mouse-pointer me-1"),
                "Clic para filtrar",
            ], className="text-muted me-3"),
            html.Small([
                html.Span(
                    "█",
                    style={"color": "#ff8c00", "marginRight": "4px"}
                ),
                html.Span(
                    [
                        "Requiere revisión",
                        html.I(className="fas fa-info-circle ms-1", style={"fontSize": "0.8em"}),
                    ],
                    id="ventalibre-requiere-revision-help-bars",
                    style={"cursor": "help"},
                ),
            ], className="text-muted"),
        ], className="d-flex justify-content-center mt-2"),
        # Tooltip explicativo (Issue #490)
        dbc.Tooltip(
            html.Div([
                html.Strong("Productos pendientes de clasificación"),
                html.Br(),
                html.Span("Estos productos no pudieron ser clasificados automáticamente "),
                html.Span("por categoría NECESIDAD y requieren revisión manual."),
                html.Br(),
                html.Small(
                    "El borde naranja indica categorías con productos pendientes.",
                    className="text-warning",
                ),
            ]),
            target="ventalibre-requiere-revision-help-bars",
            placement="top",
        ),
    ])
