# frontend/components/ventalibre/l2_value_quadrant.py
"""
L2 Value Quadrant Component (Issue #505)

Muestra subcategorias L2 en un cuadrante de valor:
- Eje X: Ventas totales
- Eje Y: Margen porcentual
- Color: Por cuadrante (star, opportunity, cash_cow, dog)
- Archetype: Info del negocio
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html

from styles.design_tokens import COLORS
from utils.chart_benchmarks import add_benchmark_lines  # Issue #510

# Colores por cuadrante
QUADRANT_COLORS = {
    "star": "#28a745",       # Verde - Estrellas
    "opportunity": "#ffc107",  # Amarillo - Oportunidades (interrogantes)
    "cash_cow": "#17a2b8",   # Azul - Vacas lecheras
    "dog": "#dc3545",        # Rojo - Perros
}

QUADRANT_LABELS = {
    "star": "Estrellas",
    "opportunity": "Oportunidades",
    "cash_cow": "Vacas Lecheras",
    "dog": "Perros",
}

QUADRANT_ICONS = {
    "star": "star",
    "opportunity": "question-circle",
    "cash_cow": "dollar-sign",
    "dog": "paw",
}


def create_l2_value_quadrant(
    quadrant_data: Optional[Dict] = None,
    l1_category: Optional[str] = None,
    height: int = 350,
    targets: Optional[List[Dict[str, Any]]] = None,  # Issue #510: Benchmark lines
) -> html.Div:
    """
    Create L2 Value Quadrant scatter chart.

    Args:
        quadrant_data: Dict from API with structure:
            {
                "subcategories": [
                    {
                        "l2": "solar_facial",
                        "display_name": "Solar Facial",
                        "sales": 5000,
                        "margin_pct": 35.5,
                        "transactions": 100,
                        "quadrant": "star",
                        "quadrant_label": "Estrellas",
                        "archetype": "Estrategico"
                    },
                    ...
                ],
                "thresholds": {"median_sales": 2500, "median_margin": 30.0},
                "l1_category": "dermocosmetica"
            }
        l1_category: L1 category code
        height: Chart height in pixels
        targets: List of benchmark targets for margin lines (Issue #510)
            [{value, color, line_style, label}, ...]

    Returns:
        html.Div with quadrant chart
    """
    if not quadrant_data or not quadrant_data.get("subcategories"):
        return _create_quadrant_placeholder(l1_category)

    subcategories = quadrant_data["subcategories"]
    thresholds = quadrant_data.get("thresholds", {})

    fig = _create_quadrant_figure(subcategories, thresholds, targets)

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            html.I(className="fas fa-th-large me-2"),
                            html.Span("Cuadrante de Valor L2", className="fw-bold"),
                        ],
                        className="d-flex align-items-center",
                    ),
                    html.Small(
                        "Posicionamiento de subcategorias por ventas y margen",
                        className="text-muted d-block mt-1",
                    ),
                ]
            ),
            dbc.CardBody(
                [
                    dcc.Graph(
                        id="ventalibre-l2-value-quadrant",
                        figure=fig,
                        config={"responsive": True, "displayModeBar": False},
                        style={"height": f"{height}px"},
                    ),
                    _create_quadrant_legend(),
                ]
            ),
        ],
        className="shadow-sm",
    )


def _create_quadrant_figure(
    subcategories: List[Dict],
    thresholds: Dict,
    targets: Optional[List[Dict[str, Any]]] = None,
) -> go.Figure:
    """Create the quadrant scatter figure."""
    fig = go.Figure()

    # Group by quadrant for different colors
    for quadrant in ["star", "opportunity", "cash_cow", "dog"]:
        items = [s for s in subcategories if s.get("quadrant") == quadrant]

        if not items:
            continue

        fig.add_trace(
            go.Scatter(
                x=[item["sales"] for item in items],
                y=[item["margin_pct"] for item in items],
                mode="markers+text",
                marker=dict(
                    size=14,
                    color=QUADRANT_COLORS.get(quadrant, "#6c757d"),
                    line=dict(width=1, color="white"),
                    opacity=0.85,
                ),
                text=[item["display_name"][:12] for item in items],
                textposition="top center",
                textfont=dict(size=9),
                name=QUADRANT_LABELS.get(quadrant, quadrant),
                hovertemplate=(
                    "<b>%{customdata[0]}</b><br>"
                    "Arquetipo: %{customdata[1]}<br>"
                    "Ventas: %{x:,.0f}<br>"
                    "Margen: %{y:.1f}%<br>"
                    "Transacciones: %{customdata[2]:,}<br>"
                    "<extra></extra>"
                ),
                customdata=[
                    [
                        item.get("display_name", item.get("l2", "")),
                        item.get("archetype", "N/A"),
                        item.get("transactions", 0),
                    ]
                    for item in items
                ],
            )
        )

    # Add median reference lines
    median_sales = thresholds.get("median_sales", 0)
    median_margin = thresholds.get("median_margin", 0)

    if median_sales > 0:
        fig.add_vline(
            x=median_sales,
            line_dash="dash",
            line_color="rgba(0, 0, 0, 0.3)",
            annotation_text="Mediana Ventas",
            annotation_position="top right",
            annotation_font_size=9,
        )

    if median_margin > 0:
        fig.add_hline(
            y=median_margin,
            line_dash="dash",
            line_color="rgba(0, 0, 0, 0.3)",
            annotation_text="Mediana Margen",
            annotation_position="top right",
            annotation_font_size=9,
        )

    # Issue #510: Add target benchmark lines
    if targets:
        add_benchmark_lines(fig, targets, axis="y", annotation_position="left")

    # Layout configuration
    fig.update_layout(
        xaxis_title="Ventas Totales",
        yaxis_title="Margen (%)",
        showlegend=True,
        legend=dict(
            orientation="h",
            yanchor="bottom",
            y=1.02,
            xanchor="center",
            x=0.5,
            font=dict(size=10),
        ),
        margin=dict(t=50, b=40, l=50, r=20),
        plot_bgcolor="rgba(248, 249, 250, 1)",
        hovermode="closest",
    )

    # Grid styling
    fig.update_xaxes(
        showgrid=True,
        gridcolor="rgba(0, 0, 0, 0.08)",
        zeroline=False,
    )
    fig.update_yaxes(
        showgrid=True,
        gridcolor="rgba(0, 0, 0, 0.08)",
        zeroline=False,
    )

    return fig


def _create_quadrant_placeholder(l1_category: Optional[str] = None) -> html.Div:
    """Placeholder when no quadrant data available."""
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-th-large me-2"),
                    html.Span("Cuadrante de Valor L2", className="fw-bold"),
                ]
            ),
            dbc.CardBody(
                [
                    html.Div(
                        [
                            html.I(
                                className="fas fa-th-large fa-3x text-muted mb-2"
                            ),
                            html.P(
                                "No hay datos de subcategorias L2 disponibles",
                                className="text-muted mb-0",
                            ),
                            html.Small(
                                "Asegurate de tener productos clasificados con L2",
                                className="text-muted",
                            ),
                        ],
                        className="text-center py-4",
                    )
                ]
            ),
        ],
        className="shadow-sm bg-light",
    )


def _create_quadrant_legend() -> html.Div:
    """Legend explaining each quadrant."""
    return html.Div(
        [
            html.Hr(className="my-2"),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                [
                                    html.I(
                                        className=f"fas fa-{QUADRANT_ICONS['star']} me-1",
                                        style={"color": QUADRANT_COLORS["star"]},
                                    ),
                                    html.Strong(
                                        "Estrellas",
                                        style={"color": QUADRANT_COLORS["star"]},
                                    ),
                                ]
                            ),
                            html.Small(
                                " Alto margen + Altas ventas",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                        className="mb-1",
                    ),
                    dbc.Col(
                        [
                            html.Span(
                                [
                                    html.I(
                                        className=f"fas fa-{QUADRANT_ICONS['cash_cow']} me-1",
                                        style={"color": QUADRANT_COLORS["cash_cow"]},
                                    ),
                                    html.Strong(
                                        "Vacas Lecheras",
                                        style={"color": QUADRANT_COLORS["cash_cow"]},
                                    ),
                                ]
                            ),
                            html.Small(
                                " Alto margen + Bajas ventas",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                        className="mb-1",
                    ),
                ],
            ),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Span(
                                [
                                    html.I(
                                        className=f"fas fa-{QUADRANT_ICONS['opportunity']} me-1",
                                        style={"color": QUADRANT_COLORS["opportunity"]},
                                    ),
                                    html.Strong(
                                        "Oportunidades",
                                        style={"color": QUADRANT_COLORS["opportunity"]},
                                    ),
                                ]
                            ),
                            html.Small(
                                " Bajo margen + Altas ventas",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            html.Span(
                                [
                                    html.I(
                                        className=f"fas fa-{QUADRANT_ICONS['dog']} me-1",
                                        style={"color": QUADRANT_COLORS["dog"]},
                                    ),
                                    html.Strong(
                                        "Perros",
                                        style={"color": QUADRANT_COLORS["dog"]},
                                    ),
                                ]
                            ),
                            html.Small(
                                " Bajo margen + Bajas ventas",
                                className="text-muted",
                            ),
                        ],
                        width=6,
                    ),
                ],
            ),
        ],
        className="small",
    )
