"""
L2 Drill-Down Modal Component (Issue #505)

Modal para análisis granular de subcategorías L2 dentro de categorías L1.
Solo disponible para: dermocosmetica, suplementos, higiene_bucal.

Flujo:
1. Click en categoría L1 con badge "↘ L2" en treemap
2. Modal abre con treemap L2 de ventas por subcategoría
3. Click en L2 filtra tabla de productos
"""

from typing import Dict, List

import dash_bootstrap_components as dbc
import pandas as pd
import plotly.express as px
import plotly.graph_objects as go
from dash import dcc, html

from components.ventalibre.categories import L2_CATEGORIES, get_l2_info, get_principal_info
from styles.design_tokens import COLORS, SPACING


def create_l2_drilldown_modal() -> dbc.Modal:
    """
    Create the L2 drill-down modal.

    Returns:
        dbc.Modal: Modal component with L2 treemap, value quadrant, and details
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(
                    [
                        html.I(className="fas fa-layer-group me-2"),
                        html.Span(id="ventalibre-l2-modal-title", children="Subcategorías L2"),
                    ]
                ),
                close_button=True,
            ),
            dbc.ModalBody(
                [
                    # L2 coverage indicator
                    html.Div(
                        id="ventalibre-l2-coverage-badge",
                        className="mb-3",
                    ),
                    # Two-column layout: Treemap + Value Quadrant
                    dbc.Row(
                        [
                            # L2 treemap container (left)
                            dbc.Col(
                                [
                                    html.H6(
                                        [
                                            html.I(className="fas fa-sitemap me-2"),
                                            "Distribución de Ventas",
                                        ],
                                        className="text-muted mb-2",
                                    ),
                                    dcc.Loading(
                                        id="ventalibre-l2-treemap-loading",
                                        type="default",
                                        children=html.Div(id="ventalibre-l2-treemap-container"),
                                    ),
                                ],
                                width=12,
                                lg=6,
                                className="mb-3 mb-lg-0",
                            ),
                            # L2 value quadrant (right)
                            dbc.Col(
                                [
                                    dcc.Loading(
                                        id="ventalibre-l2-quadrant-loading",
                                        type="default",
                                        children=html.Div(id="ventalibre-l2-quadrant-container"),
                                    ),
                                ],
                                width=12,
                                lg=6,
                            ),
                        ],
                        className="g-3",
                    ),
                    # L2 archetype legend
                    html.Div(
                        id="ventalibre-l2-archetype-legend",
                        className="mt-3",
                    ),
                ],
            ),
            dbc.ModalFooter(
                [
                    dbc.Button(
                        html.Span([
                            html.I(className="fas fa-arrow-left me-2"),
                            "Volver a L1",
                        ]),
                        id="ventalibre-l2-modal-close",
                        color="secondary",
                        outline=True,
                    ),
                ],
            ),
        ],
        id="ventalibre-l2-modal",
        is_open=False,
        centered=True,
        size="xl",  # Larger modal for two-column layout
        backdrop="static",
    )


def create_l2_treemap(sales_data: Dict, l1_category: str, height: int = 350) -> html.Div:
    """
    Create treemap for L2 subcategories.

    Args:
        sales_data: Dict with L2 sales data from API
            {
                "nodes": [
                    {"category": "solar_facial", "display_name": "Solar Facial",
                     "archetype": "...", "sales": 5000, "count": 100, "percentage": 25.0},
                    ...
                ],
                "total_sales": 20000,
                "l1_category": "dermocosmetica"
            }
        l1_category: Parent L1 category code
        height: Chart height in pixels

    Returns:
        html.Div with Plotly treemap
    """
    nodes = sales_data.get("nodes", [])

    if not nodes:
        l1_info = get_principal_info(l1_category)
        return html.Div(
            [
                html.Div(
                    [
                        html.I(className="fas fa-chart-pie fa-3x text-muted mb-3"),
                        html.P("No hay datos L2 disponibles", className="text-muted"),
                        html.Small(
                            f"No hay productos clasificados a nivel L2 en {l1_info.get('name', l1_category)}",
                            className="text-muted",
                        ),
                    ],
                    className="text-center py-5",
                )
            ]
        )

    # Prepare data for treemap
    df_data = []
    for node in nodes:
        # Backend returns "category" as the L2 category code (e.g., "solar_facial")
        l2_category = node.get("category", "")
        l2_info = get_l2_info(l2_category)

        df_data.append(
            {
                "category": l2_info.get("name", l2_category),
                "category_id": l2_category,
                "sales": node.get("sales", 0),
                "count": node.get("count", 0),
                "percentage": node.get("percentage", 0),
                "color": l2_info.get("color", "#6c757d"),
                "archetype": l2_info.get("archetype", ""),
            }
        )

    df = pd.DataFrame(df_data)

    # Create color map (using dict(zip()) for efficiency)
    color_map = dict(zip(df["category"], df["color"]))

    fig = px.treemap(
        df,
        path=["category"],
        values="sales",
        color="category",
        color_discrete_map=color_map,
        custom_data=["category_id", "count", "percentage", "archetype"],
    )

    # Configure hover and text
    fig.update_traces(
        hovertemplate=(
            "<b>%{label}</b><br>"
            "Ventas: €%{value:,.2f}<br>"
            "Productos: %{customdata[1]:,}<br>"
            "Porcentaje: %{customdata[2]:.1f}%<br>"
            "Arquetipo: %{customdata[3]}"
            "<extra></extra>"
        ),
        texttemplate="<b>%{label}</b><br>€%{value:,.0f}",
        textfont={"size": 11},
    )

    # Clean layout
    fig.update_layout(
        margin=dict(t=10, l=10, r=10, b=10),
        height=height,
        showlegend=False,
    )

    return html.Div(
        [
            dcc.Graph(
                id="ventalibre-l2-treemap",
                figure=fig,
                config={
                    "displayModeBar": False,
                    "responsive": True,
                },
                style={"height": f"{height}px"},
            ),
            # Interaction legend
            html.Div(
                [
                    html.Small(
                        [
                            html.I(className="fas fa-mouse-pointer me-1"),
                            "Clic para filtrar productos",
                        ],
                        className="text-muted",
                    ),
                ],
                className="text-center mt-2",
            ),
        ]
    )


def create_l2_coverage_badge(coverage_data: Dict, l1_category: str) -> html.Div:
    """
    Create coverage badge for L2 classification.

    When coverage < 80%, includes a tooltip explaining implications.

    Args:
        coverage_data: Dict with coverage stats
            {"total": 1000, "classified": 750, "percentage": 75.0}
        l1_category: L1 category code

    Returns:
        html.Div with coverage badge (and tooltip if < 80%)
    """
    total = coverage_data.get("total", 0)
    classified = coverage_data.get("classified", 0)
    percentage = coverage_data.get("percentage", 0)

    l1_info = get_principal_info(l1_category)

    # Color based on coverage
    if percentage >= 80:
        badge_color = "success"
        icon = "check-circle"
        tooltip_message = None
    elif percentage >= 50:
        badge_color = "warning"
        icon = "exclamation-circle"
        tooltip_message = (
            f"⚠️ Cobertura L2 mejorable ({percentage:.0f}%)\n\n"
            f"• {total - classified:,} productos sin clasificar L2\n"
            f"• Los datos agregados pueden no reflejar el 100%\n"
            f"• Considera ejecutar el backfill de clasificación L2"
        )
    else:
        badge_color = "danger"
        icon = "times-circle"
        tooltip_message = (
            f"⚠️ Cobertura L2 baja ({percentage:.0f}%)\n\n"
            f"• {total - classified:,} productos sin clasificar L2\n"
            f"• Los análisis por subcategoría serán incompletos\n"
            f"• Recomendado: ejecutar backfill de clasificación L2"
        )

    # Create badge with unique ID for tooltip
    badge_id = f"l2-coverage-badge-{l1_category}"
    badge = dbc.Badge(
        [
            html.I(className=f"fas fa-{icon} me-2"),
            f"Cobertura L2: {percentage:.1f}% ({classified:,}/{total:,} productos)",
        ],
        id=badge_id,
        color=badge_color,
        className="px-3 py-2",
        style={"cursor": "help"} if tooltip_message else {},
    )

    # Build components list
    components = [badge]

    # Add tooltip if coverage < 80%
    if tooltip_message:
        components.append(
            dbc.Tooltip(
                tooltip_message,
                target=badge_id,
                placement="bottom",
                style={"whiteSpace": "pre-line", "textAlign": "left"},
            )
        )

    components.append(
        html.Small(
            f" en {l1_info.get('name', l1_category)}",
            className="text-muted ms-2",
        )
    )

    return html.Div(
        components,
        className="d-flex align-items-center",
    )


def create_l2_archetype_legend(l1_category: str) -> html.Div:
    """
    Create archetype legend for L2 subcategories.

    Args:
        l1_category: L1 category code

    Returns:
        html.Div with archetype legend
    """
    # Get L2 categories for this L1
    l2_list = []
    for l2_code, l2_data in L2_CATEGORIES.items():
        if l2_data.get("l1") == l1_category:
            l2_list.append(
                {
                    "name": l2_data.get("name", l2_code),
                    "archetype": l2_data.get("archetype", ""),
                    "color": l2_data.get("color", "#6c757d"),
                }
            )

    if not l2_list:
        return html.Div()

    # Create legend items
    legend_items = []
    for l2 in l2_list:
        legend_items.append(
            html.Div(
                [
                    html.Span(
                        l2["archetype"].split(" ")[0] if l2["archetype"] else "●",
                        style={"marginRight": "4px"},
                    ),
                    html.Small(
                        l2["name"],
                        style={"color": l2["color"], "fontWeight": "500"},
                    ),
                ],
                className="d-inline-flex align-items-center me-3 mb-1",
            )
        )

    return html.Div(
        [
            html.Hr(className="my-2"),
            html.H6(
                [
                    html.I(className="fas fa-info-circle me-2", style={"color": COLORS.get("info", "#17a2b8")}),
                    "Arquetipos de Venta",
                ],
                className="mb-2",
                style={"fontSize": "0.9rem"},
            ),
            html.Div(
                legend_items,
                className="d-flex flex-wrap",
            ),
        ]
    )


def create_l2_badge_for_l1(l1_category: str) -> dbc.Badge:
    """
    Create a small badge to indicate L2 drill-down is available.

    This badge should be shown next to L1 categories that have L2 subcategories.

    Args:
        l1_category: L1 category code

    Returns:
        dbc.Badge if L2 available, otherwise None
    """
    from components.ventalibre.categories import has_l2_subcategories

    if not has_l2_subcategories(l1_category):
        return None

    return dbc.Badge(
        html.Span([
            html.I(className="fas fa-chevron-down", style={"fontSize": "0.7em"}),
            " L2",
        ]),
        color="info",
        pill=True,
        className="ms-1",
        style={"fontSize": "0.65rem", "verticalAlign": "middle", "cursor": "pointer"},
        title="Click para ver subcategorías L2",
    )
