"""
Executive Summary Card Component (Issue #509)

Componente que muestra el resumen ejecutivo generado por NarrativeService.

Estructura de 4 secciones:
1. Titular (headline) - EL QUE
2. Analisis (analysis) - EL PORQUE
3. Alerta (alert) - Si hay problema critico
4. Accion (action) - EL COMO

Incluye:
- Botones de feedback (thumbs up/down)
- Badge de validacion de grounding
- Boton de accion con navegacion real
"""

from typing import Any, Dict, Optional

import dash_bootstrap_components as dbc
from dash import dcc, html

from components.base import BaseCard
from styles.design_tokens import COLORS, SHADOWS, SPACING


def create_executive_summary_card(
    summary_data: Optional[Dict[str, Any]] = None,
    loading: bool = False,
    error: Optional[str] = None,
) -> dbc.Card:
    """
    Crea la tarjeta de resumen ejecutivo.

    Args:
        summary_data: Respuesta de NarrativeService (NarrativeResponse)
        loading: True mientras carga desde API
        error: Mensaje de error si falla

    Returns:
        dbc.Card con el resumen ejecutivo
    """
    if loading:
        return _create_loading_card()

    if error:
        return _create_error_card(error)

    if not summary_data:
        return _create_empty_card()

    summary = summary_data.get("summary", {})

    return dbc.Card(
        [
            # Header con titulo y botones de feedback
            dbc.CardHeader(
                html.Div(
                    [
                        html.Div(
                            [
                                html.I(
                                    className="fas fa-brain me-2",
                                    style={"color": COLORS["primary"]},
                                ),
                                html.Span(
                                    "Resumen Ejecutivo",
                                    style={"fontWeight": "600"},
                                ),
                            ],
                            className="d-flex align-items-center",
                        ),
                        # Feedback buttons
                        html.Div(
                            [
                                dbc.Button(
                                    html.I(className="fas fa-thumbs-up"),
                                    id="narrative-feedback-up",
                                    color="success",
                                    outline=True,
                                    size="sm",
                                    className="me-1",
                                    title="Util",
                                ),
                                dbc.Button(
                                    html.I(className="fas fa-thumbs-down"),
                                    id="narrative-feedback-down",
                                    color="danger",
                                    outline=True,
                                    size="sm",
                                    title="No util",
                                ),
                            ],
                            className="ms-auto d-flex",
                        ),
                    ],
                    className="d-flex align-items-center",
                ),
            ),
            dbc.CardBody(
                [
                    # 1. Headline (EL QUE)
                    html.H5(
                        summary.get("headline", "Analizando datos..."),
                        className="mb-3",
                        style={
                            "color": COLORS["text_primary"],
                            "fontWeight": "600",
                            "lineHeight": "1.4",
                        },
                    ),
                    # 2. Analysis (EL PORQUE) - con Markdown para **negrita**
                    dcc.Markdown(
                        summary.get("analysis", ""),
                        className="mb-3",
                        style={
                            "color": COLORS["text_secondary"],
                            "lineHeight": "1.6",
                        },
                    ),
                    # 3. Alert (si existe)
                    _create_alert_section(summary)
                    if summary.get("alert")
                    else None,
                    # 4. Action (EL COMO)
                    _create_action_section(summary),
                ],
            ),
            # Footer con metadata
            dbc.CardFooter(
                html.Div(
                    [
                        html.Small(
                            [
                                html.I(className="fas fa-clock me-1"),
                                f"Generado: {_format_timestamp(summary_data.get('generated_at', ''))}",
                            ],
                            className="text-muted",
                        ),
                        html.Small(
                            [
                                html.I(className="fas fa-shield-alt me-1"),
                                "Datos verificados"
                                if summary_data.get("grounding_valid", True)
                                else "Revisar datos",
                            ],
                            className="text-success ms-3"
                            if summary_data.get("grounding_valid", True)
                            else "text-warning ms-3",
                        ),
                    ],
                    className="d-flex align-items-center",
                ),
                style={"fontSize": "0.8rem"},
            ),
        ],
        className="mb-4 shadow-sm",
        style={
            "borderRadius": "8px",
            "border": f"1px solid {COLORS['border_light']}",
        },
    )


def _create_alert_section(summary: Dict[str, Any]) -> html.Div:
    """Crea la seccion de alerta con fondo coloreado."""
    severity = summary.get("alert_severity", "warning")

    # Mapeo de severidad a colores
    bg_colors = {
        "critical": f"rgba(220, 53, 69, 0.1)",  # Rojo claro
        "warning": f"rgba(255, 193, 7, 0.15)",  # Ambar claro
        "info": f"rgba(23, 162, 184, 0.1)",  # Azul claro
    }

    border_colors = {
        "critical": COLORS["danger"],
        "warning": COLORS["warning"],
        "info": COLORS["info"],
    }

    icons = {
        "critical": "fas fa-exclamation-triangle",
        "warning": "fas fa-exclamation-circle",
        "info": "fas fa-info-circle",
    }

    icon_colors = {
        "critical": COLORS["danger"],
        "warning": COLORS["warning_dark"],
        "info": COLORS["info"],
    }

    return html.Div(
        [
            html.I(
                className=f"{icons.get(severity, 'fas fa-info-circle')} me-2",
                style={"color": icon_colors.get(severity, COLORS["info"])},
            ),
            html.Span(summary.get("alert", ""), style={"fontWeight": "500"}),
        ],
        className="p-3 mb-3 rounded",
        style={
            "backgroundColor": bg_colors.get(severity, bg_colors["warning"]),
            "borderLeft": f"4px solid {border_colors.get(severity, COLORS['warning'])}",
        },
    )


def _create_action_section(summary: Dict[str, Any]) -> html.Div:
    """Crea la seccion de accion con boton navegable."""
    action_value = summary.get("action_value")
    action_target_type = summary.get("action_target_type")
    action_target_id = summary.get("action_target_id")

    return html.Div(
        [
            # Texto de recomendacion
            html.Div(
                [
                    html.I(
                        className="fas fa-rocket me-2",
                        style={"color": COLORS["success"]},
                    ),
                    html.Strong("Accion recomendada: "),
                    html.Span(summary.get("action", "Revisar el panel de insights")),
                ],
                className="mb-2",
            ),
            # Boton de accion + impacto economico
            html.Div(
                [
                    dbc.Button(
                        [
                            html.I(className="fas fa-arrow-right me-2"),
                            summary.get("action_button_text", "Ver detalles"),
                        ],
                        id="narrative-action-button",
                        color="primary",
                        size="sm",
                        # Guardar datos de navegacion en data attributes
                        # Se leera en el callback para navegar
                    ),
                    html.Span(
                        f" Impacto potencial: {_format_currency(action_value)}"
                        if action_value
                        else "",
                        className="ms-3 text-success fw-bold",
                    )
                    if action_value
                    else None,
                ],
            ),
            # Store oculto para datos de navegacion
            dcc.Store(
                id="narrative-action-target",
                data={
                    "type": action_target_type,
                    "id": action_target_id,
                },
            ),
        ],
        className="p-3 rounded",
        style={"backgroundColor": f"rgba(40, 167, 69, 0.05)"},
    )


def _create_loading_card():
    """Tarjeta en estado de carga (REGLA #5: usa BaseCard)."""
    return BaseCard(
        children=[
            dbc.Spinner(color="primary", size="lg"),
            html.P(
                "Generando resumen ejecutivo...",
                className="text-center mt-3 text-muted",
            ),
        ],
        className="mb-4 text-center py-4",
        role="status",
        aria_label="Cargando resumen ejecutivo",
    )


def _create_error_card(error: str):
    """Tarjeta en estado de error (REGLA #5: usa BaseCard)."""
    return BaseCard(
        children=[
            html.I(className="fas fa-exclamation-triangle fa-2x text-warning"),
            html.P(f"Error: {error}", className="text-center mt-3 text-muted"),
            dbc.Button(
                "Reintentar",
                id="narrative-retry-button",
                color="primary",
                size="sm",
                className="mt-2",
            ),
        ],
        className="mb-4 text-center py-4",
        role="alert",
        aria_label="Error al cargar resumen ejecutivo",
    )


def _create_empty_card():
    """Tarjeta vacia (sin datos) (REGLA #5: usa BaseCard)."""
    return BaseCard(
        children=[
            html.I(className="fas fa-chart-line fa-2x text-muted"),
            html.P(
                "Selecciona un rango de fechas para generar el resumen",
                className="text-center mt-3 text-muted",
            ),
        ],
        className="mb-4 text-center py-4",
        aria_label="Esperando seleccion de fechas",
    )


def _format_timestamp(iso_timestamp: str) -> str:
    """Formatea timestamp ISO a formato legible."""
    if not iso_timestamp:
        return "N/A"
    try:
        # Extraer solo la fecha (YYYY-MM-DD)
        return iso_timestamp[:10]
    except (ValueError, TypeError):
        return "N/A"


def _format_currency(value: Optional[float]) -> str:
    """Formatea valor como moneda."""
    if value is None:
        return ""
    return f"{value:,.0f}"
