"""
Componente de visualización: Evolución temporal de ventas venta libre (Issue #491)

Gráfico de líneas con series por categoría NECESIDAD.
Top 5-8 categorías visibles por defecto, resto en legendonly.
"""

import pandas as pd
import plotly.graph_objects as go
from dash import dcc, html
from typing import List, Dict

from utils.helpers import generate_plotly_spanish_month_ticks
from components.ventalibre.categories import PRINCIPAL_CATEGORIES, format_necesidad_name


def get_category_color(category: str) -> str:
    """
    Obtiene el color asignado a una categoría NECESIDAD.

    Args:
        category: Nombre interno de categoría

    Returns:
        Código hex del color
    """
    principal_info = PRINCIPAL_CATEGORIES.get(category, {})
    return principal_info.get("color", "#95a5a6")  # Gris por defecto


def create_evolution_chart(
    time_series_data: List[Dict],
    category_summary: List[Dict],
    chart_type: str = "line",
    top_n: int = 6,
    height: int = 450,
) -> html.Div:
    """
    Crea gráfico de evolución temporal de ventas venta libre por NECESIDAD.

    Top N categorías visibles por defecto, resto en legendonly.
    Formato español para números y fechas.

    Args:
        time_series_data: Lista en formato LONG con {period, category, sales, units}
        category_summary: Resumen por categoría para ordenar top N
        chart_type: Tipo de visualización ("line" o "area")
        top_n: Número de categorías visibles por defecto
        height: Altura del gráfico en pixels

    Returns:
        html.Div con dcc.Graph
    """
    # Validación 1: Verificar que hay datos
    if not time_series_data or not category_summary:
        return html.Div([
            html.Div([
                html.I(className="fas fa-chart-line fa-3x text-muted mb-3"),
                html.P("No hay datos disponibles", className="text-muted mb-1"),
                html.Small(
                    "Sube archivos de ventas para ver la evolución temporal",
                    className="text-muted"
                ),
            ], className="text-center py-5")
        ])

    # Convertir a DataFrame
    df = pd.DataFrame(time_series_data)

    # Validación 2: Verificar columnas esperadas
    required_columns = ['period', 'category', 'sales', 'units']
    missing_columns = [col for col in required_columns if col not in df.columns]

    if missing_columns or df.empty:
        error_msg = f"Columnas faltantes: {missing_columns}" if missing_columns else "Sin datos"
        return html.Div([
            html.Div([
                html.I(className="fas fa-exclamation-triangle fa-2x text-warning mb-2"),
                html.P(f"Error: {error_msg}", className="text-muted"),
            ], className="text-center py-4")
        ])

    # Convertir período a datetime para ordenar
    df['period_dt'] = pd.to_datetime(df['period'] + '-01')  # 'YYYY-MM' -> 'YYYY-MM-01'
    df = df.sort_values(['period_dt', 'category'])

    # Ordenar categorías por ventas totales (top N)
    category_order = sorted(
        category_summary,
        key=lambda x: x.get('total_sales', 0),
        reverse=True
    )
    top_categories = [cat['category'] for cat in category_order[:top_n]]

    # Crear figura
    fig = go.Figure()

    # Añadir serie por categoría
    unique_categories = sorted(df['category'].unique())

    for category in unique_categories:
        category_df = df[df['category'] == category].sort_values('period_dt')

        # Determinar si categoría debe estar visible por defecto
        visible = True if category in top_categories else 'legendonly'

        # Obtener color y nombre
        color = get_category_color(category)
        display_name = format_necesidad_name(category)

        if chart_type == "area":
            # Área apilada
            fig.add_trace(go.Scatter(
                x=category_df['period_dt'],
                y=category_df['sales'],
                mode='lines',
                name=display_name,
                line={'width': 0.5, 'color': color},
                fillcolor=color,
                stackgroup='one',
                groupnorm='',
                visible=visible,
                customdata=category_df['units'],
                hovertemplate=(
                    '<b>%{fullData.name}</b><br>' +
                    'Período: %{x|%b %Y}<br>' +
                    'Ventas: €%{y:,.2f}<br>' +
                    'Unidades: %{customdata:,.0f}<br>' +
                    '<extra></extra>'
                )
            ))
        else:
            # Líneas + marcadores (default)
            fig.add_trace(go.Scatter(
                x=category_df['period_dt'],
                y=category_df['sales'],
                mode='lines+markers',
                name=display_name,
                line={'color': color, 'width': 2},
                marker={'size': 6},
                visible=visible,
                customdata=category_df['units'],
                hovertemplate=(
                    '<b>%{fullData.name}</b><br>' +
                    'Período: %{x|%b %Y}<br>' +
                    'Ventas: €%{y:,.2f}<br>' +
                    'Unidades: %{customdata:,.0f}<br>' +
                    '<extra></extra>'
                )
            ))

    # Generar ticks de meses en español
    all_dates = df['period_dt'].tolist()
    tickvals, ticktext = generate_plotly_spanish_month_ticks(all_dates)

    # Configurar eje X
    xaxis_config = {
        'title': '',
        'showgrid': True,
        'gridcolor': '#ecf0f1',
        'linecolor': '#bdc3c7',
    }

    if tickvals and ticktext:
        xaxis_config['tickvals'] = tickvals
        xaxis_config['ticktext'] = ticktext
    else:
        xaxis_config['tickformat'] = '%b %Y'
        xaxis_config['dtick'] = 'M1'

    # Layout
    fig.update_layout(
        title={
            'text': 'Evolución de Ventas por Categoría',
            'x': 0.5,
            'xanchor': 'center',
            'font': {'size': 16, 'color': '#2c3e50'}
        },
        xaxis=xaxis_config,
        yaxis={
            'title': 'Ventas (€)',
            'showgrid': True,
            'gridcolor': '#ecf0f1',
            'linecolor': '#bdc3c7',
            'tickformat': ',.0f',
            'separatethousands': True
        },
        hovermode='x unified',
        legend={
            'title': {'text': 'Categorías'},
            'orientation': 'h',
            'yanchor': 'top',
            'y': -0.15,
            'xanchor': 'center',
            'x': 0.5,
            'font': {'size': 9}
        },
        height=height,
        margin={'l': 70, 'r': 30, 't': 50, 'b': 100},
        plot_bgcolor='white',
        paper_bgcolor='white'
    )

    return html.Div([
        dcc.Graph(
            id="ventalibre-evolution-chart",
            figure=fig,
            config={
                'displayModeBar': True,
                'displaylogo': False,
                'modeBarButtonsToRemove': ['lasso2d', 'select2d'],
                'locale': 'es'
            }
        ),
        html.Small([
            html.I(className="fas fa-info-circle me-1"),
            f"Mostrando top {min(top_n, len(unique_categories))} categorías. ",
            "Clic en leyenda para mostrar/ocultar."
        ], className="text-muted d-block text-center mt-1")
    ])
