# frontend/components/ventalibre/decision_box.py
"""
Decision Box (Semáforo) para Tab 3 (Issue #494).

Muestra recomendación algorítmica para el producto:
- Semáforo: Verde (comprar), Amarillo (mantener), Rojo (reducir/eliminar)
- Toggle para ver detalle con razones y métricas
- Badge de estacionalidad (Issue #507)
"""

import uuid
from typing import Optional, Tuple

from dash import html
import dash_bootstrap_components as dbc


def create_decision_box(
    recommendation: Optional[dict] = None,
    factors: Optional[dict] = None,
    product_name: Optional[str] = None,
) -> html.Div:
    """
    Crea el Decision Box con semáforo y recomendación.

    Args:
        recommendation: Dict con action, alert_level, confidence, reasons
        factors: Dict con métricas consideradas
        product_name: Nombre del producto

    Returns:
        html.Div con el semáforo y detalle colapsable
    """
    if not recommendation:
        return create_decision_placeholder()

    action = recommendation.get("action", "mantener")
    alert_level = recommendation.get("alert_level", "warning")
    confidence = recommendation.get("confidence", 0)
    reasons = recommendation.get("reasons", [])
    suggested_qty = recommendation.get("suggested_quantity")

    # Mapeo de colores y iconos
    semaphore_config = {
        "success": {"color": "success", "icon": "fa-check-circle", "label": "Oportunidad"},
        "warning": {"color": "warning", "icon": "fa-exclamation-triangle", "label": "Atención"},
        "danger": {"color": "danger", "icon": "fa-times-circle", "label": "Acción requerida"},
    }
    config = semaphore_config.get(alert_level, semaphore_config["warning"])

    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-traffic-light me-2"),
                html.Span("Recomendación", className="fw-bold"),
            ], className="d-flex align-items-center"),
        ]),
        dbc.CardBody([
            # Semáforo principal
            html.Div([
                # Círculo de semáforo
                html.Div([
                    html.I(className=f"fas {config['icon']} fa-4x text-{config['color']}"),
                ], className="text-center mb-3"),

                # Acción recomendada
                html.H4(
                    action.upper(),
                    className=f"text-center text-{config['color']} fw-bold mb-2",
                ),

                # Badge estacionalidad (Issue #507)
                _create_seasonality_badge(factors),

                # Confianza
                html.Div([
                    html.Small("Confianza: ", className="text-muted"),
                    dbc.Progress(
                        value=confidence,
                        color=config["color"],
                        className="flex-grow-1 ms-2",
                        style={"height": "8px"},
                    ),
                    html.Small(f" {confidence:.0f}%", className="ms-2 fw-bold"),
                ], className="d-flex align-items-center mb-3"),

                # Cantidad sugerida (si aplica)
                html.Div([
                    dbc.Alert([
                        html.I(className="fas fa-box me-2"),
                        f"Cantidad sugerida próximo pedido: {suggested_qty} uds",
                    ], color=config["color"], className="mb-0 py-2"),
                ]) if suggested_qty else html.Div(),

            ]),

            # Toggle para ver detalle
            dbc.Button(
                html.Span([
                    html.I(className="fas fa-chevron-down me-2"),
                    "Ver detalle",
                ]),
                id="ventalibre-recommendation-toggle",
                color="link",
                size="sm",
                className="w-100 mt-2",
            ),

            # Detalle colapsable
            dbc.Collapse(
                id="ventalibre-recommendation-collapse",
                is_open=False,
                children=html.Div([
                    html.Hr(),
                    # Razones
                    html.H6("Razones:", className="fw-bold mb-2"),
                    html.Ul([
                        html.Li(reason, className="small")
                        for reason in reasons
                    ], className="mb-3"),

                    # Factores considerados
                    html.H6("Factores analizados:", className="fw-bold mb-2"),
                    _create_factors_display(factors),
                ]),
            ),
        ]),
    ], className="shadow-sm")


def _create_seasonality_badge(factors: Optional[dict]) -> html.Div:
    """
    Crea badge de estacionalidad con tooltip (Issue #507).

    Muestra contexto estacional cuando el índice es significativo:
    - Temporada baja: index < 0.8
    - Temporada alta: index > 1.2
    - Normal: 0.8 <= index <= 1.2 (no mostrar badge)

    Args:
        factors: Dict con seasonality_index y momentum_seasonal

    Returns:
        html.Div con badge y tooltip, o html.Div vacío si no aplica
    """
    if not factors:
        return html.Div()

    index = factors.get("seasonality_index")
    momentum = factors.get("momentum_seasonal")
    context = factors.get("seasonality_context")

    # No mostrar badge si no hay índice o es cercano a 1.0
    if index is None or (0.8 <= index <= 1.2):
        return html.Div()

    # Determinar tipo de temporada
    if index < 0.8:
        badge_color = "info"
        badge_icon = "fa-snowflake"
        badge_text = f"Temporada baja ({index:.1f}x)"
    else:  # index > 1.2
        badge_color = "warning"
        badge_icon = "fa-sun"
        badge_text = f"Temporada alta ({index:.1f}x)"

    # Construir mensaje del tooltip
    tooltip_lines = []
    tooltip_lines.append(f"Índice estacional: {index:.1f}")
    tooltip_lines.append("1.0 = promedio anual")

    if momentum is not None:
        if momentum >= 0:
            tooltip_lines.append(f"Rendimiento vs esperado: +{momentum:.1f}%")
        else:
            tooltip_lines.append(f"Rendimiento vs esperado: {momentum:.1f}%")

    if context:
        tooltip_lines.append(context)

    tooltip_text = " | ".join(tooltip_lines)

    # ID dinámico para evitar conflictos si se renderiza múltiples veces
    badge_id = f"seasonality-badge-{uuid.uuid4().hex[:8]}"

    return html.Div([
        dbc.Badge(
            [
                html.I(className=f"fas {badge_icon} me-1"),
                badge_text,
            ],
            id=badge_id,
            color=badge_color,
            className="mb-2",
            style={"cursor": "help"},
        ),
        dbc.Tooltip(
            tooltip_text,
            target=badge_id,
            placement="top",
        ),
    ], className="text-center")


def _create_factors_display(factors: Optional[dict]) -> html.Div:
    """Muestra los factores considerados en la recomendación."""
    if not factors:
        return html.Div()

    items = []

    coverage = factors.get("coverage_days")
    if coverage:
        items.append(("Días cobertura", f"{coverage:.0f} días"))

    trend = factors.get("trend_mat")
    if trend is not None:
        items.append(("Tendencia MAT", f"{trend:+.1f}%"))

    margin_vs = factors.get("margin_vs_category")
    if margin_vs is not None:
        items.append(("Margen vs Cat.", f"{margin_vs:+.1f} pp"))

    gmroi = factors.get("gmroi")
    if gmroi:
        items.append(("GMROI", f"{gmroi:.2f}x"))

    volume_rank = factors.get("sales_volume_rank")
    if volume_rank:
        items.append(("Ranking ventas", volume_rank.capitalize()))

    has_alternatives = factors.get("has_better_alternatives", False)
    items.append(("Alternativas mejores", "Sí" if has_alternatives else "No"))

    # Issue #507: Factores de estacionalidad
    seasonality_index = factors.get("seasonality_index")
    if seasonality_index is not None:
        items.append(("Índice estacional", f"{seasonality_index:.2f}x"))

    momentum_seasonal = factors.get("momentum_seasonal")
    if momentum_seasonal is not None:
        items.append(("Momentum vs esperado", f"{momentum_seasonal:+.1f}%"))

    return dbc.Table([
        html.Tbody([
            html.Tr([
                html.Td(label, className="text-muted small"),
                html.Td(value, className="fw-bold small text-end"),
            ])
            for label, value in items
        ]),
    ], size="sm", borderless=True, className="mb-0")


def create_decision_placeholder() -> html.Div:
    """Placeholder cuando no hay producto seleccionado."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-traffic-light me-2"),
            html.Span("Recomendación", className="fw-bold"),
        ]),
        dbc.CardBody([
            html.Div([
                # Semáforos apagados
                html.Div([
                    html.I(className="fas fa-circle fa-2x text-muted me-2"),
                    html.I(className="fas fa-circle fa-2x text-muted me-2"),
                    html.I(className="fas fa-circle fa-2x text-muted"),
                ], className="text-center mb-3"),
                html.P(
                    "Selecciona un producto para ver la recomendación",
                    className="text-muted mb-0 text-center",
                ),
            ], className="py-3"),
        ]),
    ], className="shadow-sm bg-light")


def create_semaphore_legend() -> html.Div:
    """Leyenda de interpretación del semáforo."""
    return html.Div([
        html.Small("Interpretación:", className="fw-bold d-block mb-2"),
        html.Div([
            html.Span([
                html.I(className="fas fa-circle text-success me-1"),
                "Comprar",
            ], className="me-3"),
            html.Span([
                html.I(className="fas fa-circle text-warning me-1"),
                "Mantener/Revisar",
            ], className="me-3"),
            html.Span([
                html.I(className="fas fa-circle text-danger me-1"),
                "Reducir/Eliminar",
            ]),
        ], className="small"),
    ])
