"""
Context Menu Component for VentaLibre Treemap (Issue #512)

Modal de navegación contextual que aparece al hacer click derecho
en una categoría del treemap de NECESIDAD.

Opciones disponibles:
1. Ver productos de esta categoría → Tab 4 (Producto) + filtro
2. Ver evolución temporal → Tab 3 (Tendencias)
3. Ver análisis de marcas → Tab 2 (Brands) + dropdown
4. Exportar lista de productos → CSV download

Flujo:
1. Right-click/long-press capturado por context_menu.js
2. JavaScript actualiza ventalibre-context-click-store
3. Callback detecta cambio y abre este modal
4. Usuario selecciona opción → navegación + filtro aplicado
"""

import dash_bootstrap_components as dbc
from dash import html

from styles.design_tokens import COLORS, SPACING


def create_context_menu_modal() -> dbc.Modal:
    """
    Create the context menu modal for treemap navigation.

    Returns:
        dbc.Modal: Context menu modal component
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                dbc.ModalTitle(
                    [
                        html.I(className="fas fa-crosshairs me-2"),
                        html.Span(
                            id="ventalibre-context-menu-title",
                            children="Acciones",
                        ),
                    ],
                    style={"fontSize": "1rem"},
                ),
                close_button=True,
                style={"padding": SPACING.get("md", "16px")},
            ),
            dbc.ModalBody(
                [
                    # Category info badge
                    html.Div(
                        id="ventalibre-context-category-badge",
                        className="mb-3",
                    ),
                    # Navigation options as ListGroup
                    dbc.ListGroup(
                        [
                            # Option 1: Ver productos
                            dbc.ListGroupItem(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-box-open me-3",
                                                style={
                                                    "color": COLORS.get("primary", "#2c3e50"),
                                                    "width": "20px",
                                                },
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Ver productos"),
                                                    html.Br(),
                                                    html.Small(
                                                        "Explorar productos de esta categoría",
                                                        className="text-muted",
                                                    ),
                                                ],
                                            ),
                                        ],
                                        className="d-flex align-items-center",
                                    ),
                                ],
                                id="ventalibre-context-option-products",
                                action=True,
                                style={"cursor": "pointer"},
                                className="py-3",
                            ),
                            # Option 2: Ver evolución temporal
                            dbc.ListGroupItem(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-chart-line me-3",
                                                style={
                                                    "color": COLORS.get("success", "#27ae60"),
                                                    "width": "20px",
                                                },
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Ver tendencias"),
                                                    html.Br(),
                                                    html.Small(
                                                        "Evolución temporal de ventas",
                                                        className="text-muted",
                                                    ),
                                                ],
                                            ),
                                        ],
                                        className="d-flex align-items-center",
                                    ),
                                ],
                                id="ventalibre-context-option-evolution",
                                action=True,
                                style={"cursor": "pointer"},
                                className="py-3",
                            ),
                            # Option 3: Ver análisis de marcas
                            dbc.ListGroupItem(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-tag me-3",
                                                style={
                                                    "color": COLORS.get("info", "#17a2b8"),
                                                    "width": "20px",
                                                },
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Análisis de marcas"),
                                                    html.Br(),
                                                    html.Small(
                                                        "Comparativa entre marcas",
                                                        className="text-muted",
                                                    ),
                                                ],
                                            ),
                                        ],
                                        className="d-flex align-items-center",
                                    ),
                                ],
                                id="ventalibre-context-option-brands",
                                action=True,
                                style={"cursor": "pointer"},
                                className="py-3",
                            ),
                            # Option 4: Exportar CSV
                            dbc.ListGroupItem(
                                [
                                    html.Div(
                                        [
                                            html.I(
                                                className="fas fa-file-csv me-3",
                                                style={
                                                    "color": COLORS.get("warning", "#f39c12"),
                                                    "width": "20px",
                                                },
                                            ),
                                            html.Div(
                                                [
                                                    html.Strong("Exportar CSV"),
                                                    html.Br(),
                                                    html.Small(
                                                        "Descargar lista de productos",
                                                        className="text-muted",
                                                    ),
                                                ],
                                            ),
                                        ],
                                        className="d-flex align-items-center",
                                    ),
                                ],
                                id="ventalibre-context-option-export",
                                action=True,
                                style={"cursor": "pointer"},
                                className="py-3",
                            ),
                        ],
                        flush=True,
                    ),
                ],
                style={"padding": SPACING.get("sm", "8px")},
            ),
        ],
        id="ventalibre-context-menu-modal",
        is_open=False,
        centered=True,
        size="sm",
        backdrop=True,
        keyboard=True,
    )


def create_context_category_badge(category_label: str, count: int, percentage: float) -> html.Div:
    """
    Create a badge displaying the selected category info.

    Args:
        category_label: Display name of the category
        count: Number of products
        percentage: Percentage of total sales

    Returns:
        html.Div with category info badge
    """
    return html.Div(
        [
            dbc.Badge(
                [
                    html.I(className="fas fa-layer-group me-2"),
                    category_label,
                ],
                color="primary",
                className="me-2 px-3 py-2",
            ),
            html.Small(
                [
                    html.Span(f"{count:,} productos", className="me-2"),
                    html.Span("•", className="text-muted mx-1"),
                    html.Span(f"{percentage:.1f}% ventas"),
                ],
                className="text-muted",
            ),
        ],
        className="d-flex align-items-center flex-wrap",
    )
