# frontend/components/ventalibre/complementary_table.py
"""
Tabla de productos complementarios (MBA) para Tab 3 (Issue #494).

Muestra productos que se compran frecuentemente junto al producto seleccionado.
Basado en Market Basket Analysis con métricas Support, Confidence, Lift.
"""

from typing import List, Optional

from dash import html
import dash_bootstrap_components as dbc


def create_complementary_table(
    complementary: Optional[List[dict]] = None,
    base_product_name: Optional[str] = None,
    basket_count: Optional[int] = None,
    analysis_period_days: int = 90,
) -> html.Div:
    """
    Crea la tabla de productos complementarios (MBA).

    Args:
        complementary: Lista de productos complementarios con métricas MBA
        base_product_name: Nombre del producto base
        basket_count: Número de cestas que contienen el producto base
        analysis_period_days: Período de análisis en días

    Returns:
        html.Div con la tabla de complementarios
    """
    if not complementary:
        return create_complementary_placeholder()

    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-shopping-basket me-2"),
                html.Span("Complementarios (Cross-selling)", className="fw-bold"),
                dbc.Badge(
                    f"{len(complementary)} productos",
                    color="success",
                    className="ms-2",
                ),
            ], className="d-flex align-items-center"),
            html.Small([
                f"Basado en {basket_count or 0} cestas | Últimos {analysis_period_days} días",
            ], className="text-muted d-block mt-1"),
        ]),
        dbc.CardBody([
            # Leyenda de métricas
            html.Div([
                html.Small([
                    html.Span("Support", className="fw-bold me-1"),
                    html.Span("= % cestas | ", className="text-muted"),
                    html.Span("Confidence", className="fw-bold me-1"),
                    html.Span("= P(B|A) | ", className="text-muted"),
                    html.Span("Lift", className="fw-bold me-1"),
                    html.Span("= fuerza asociación", className="text-muted"),
                ], className="text-muted"),
            ], className="mb-2"),
            # Tabla
            dbc.Table([
                html.Thead([
                    html.Tr([
                        html.Th("Producto", style={"width": "35%"}),
                        html.Th("Necesidad", style={"width": "20%"}),
                        html.Th("Support", className="text-center", style={"width": "15%"}),
                        html.Th("Confidence", className="text-center", style={"width": "15%"}),
                        html.Th("Lift", className="text-center", style={"width": "15%"}),
                    ]),
                ]),
                html.Tbody([
                    _create_complementary_row(comp)
                    for comp in complementary[:10]  # Limitar a 10
                ]),
            ], hover=True, striped=True, size="sm", className="mb-0"),
        ]),
    ], className="shadow-sm")


def _create_complementary_row(comp: dict) -> html.Tr:
    """Crea una fila de la tabla de complementarios."""
    cn = comp.get("codigo_nacional", "")
    name = comp.get("product_name", "")
    category = comp.get("ml_category") or "—"
    support = comp.get("support", 0)
    support_label = comp.get("support_label", f"{support:.1%}")
    confidence = comp.get("confidence", 0)
    confidence_label = comp.get("confidence_label", f"{confidence:.1%}")
    lift = comp.get("lift", 0)
    lift_level = comp.get("lift_level", "medium")

    # Color del Lift según nivel
    lift_colors = {
        "high": "success",
        "medium": "primary",
        "low": "secondary",
    }
    lift_color = lift_colors.get(lift_level, "secondary")

    return html.Tr([
        html.Td([
            html.Div(name[:30] + "..." if len(name) > 30 else name, className="text-truncate"),
            html.Small(cn, className="text-muted"),
        ]),
        html.Td(
            dbc.Badge(
                category.replace("_", " ").title()[:15],
                color="light",
                text_color="dark",
            ),
        ),
        html.Td(
            dbc.Badge(support_label, color="info", pill=True),
            className="text-center",
        ),
        html.Td(
            dbc.Badge(confidence_label, color="warning", pill=True),
            className="text-center",
        ),
        html.Td([
            dbc.Badge(
                f"{lift:.2f}x",
                color=lift_color,
                className="fw-bold",
            ),
        ], className="text-center"),
    ])


def create_complementary_placeholder() -> html.Div:
    """Placeholder cuando no hay producto seleccionado."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-shopping-basket me-2"),
            html.Span("Complementarios (Cross-selling)", className="fw-bold"),
        ]),
        dbc.CardBody([
            html.Div([
                html.I(className="fas fa-shopping-cart fa-2x text-muted mb-2"),
                html.P(
                    "Selecciona un producto para ver complementarios",
                    className="text-muted mb-0 small",
                ),
                html.Small(
                    "Basado en Market Basket Analysis (3 meses)",
                    className="text-muted",
                ),
            ], className="text-center py-3"),
        ]),
    ], className="shadow-sm bg-light")


def create_complementary_empty(basket_count: int) -> html.Div:
    """Mensaje cuando no hay complementarios significativos."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-shopping-basket me-2"),
            html.Span("Complementarios (Cross-selling)", className="fw-bold"),
        ]),
        dbc.CardBody([
            dbc.Alert([
                html.I(className="fas fa-info-circle me-2"),
                f"No se encontraron patrones de compra significativos ({basket_count} cestas analizadas).",
            ], color="info", className="mb-0"),
        ]),
    ], className="shadow-sm")


def create_lift_legend() -> html.Div:
    """Leyenda visual para interpretar el Lift."""
    return html.Div([
        html.Small("Interpretación Lift:", className="fw-bold me-2"),
        dbc.Badge("< 1.0 Negativo", color="secondary", className="me-1"),
        dbc.Badge("1.0-1.5 Moderado", color="primary", className="me-1"),
        dbc.Badge("> 1.5 Fuerte", color="success"),
    ], className="d-flex align-items-center flex-wrap")
