# frontend/components/ventalibre/competitive_scatter.py
"""
Scatter plot competitivo para Tab 3 (Issue #494).

Muestra posicionamiento del producto vs competidores en la categoría:
- Eje X: Ventas totales
- Eje Y: Margen porcentual
- Tamaño: Stock actual (opcional)
"""

from typing import Any, Dict, List, Optional

from dash import dcc, html
import dash_bootstrap_components as dbc
import plotly.graph_objects as go

from styles.design_tokens import COLORS
from utils.chart_benchmarks import add_benchmark_lines  # Issue #510


def create_competitive_scatter(
    data_points: Optional[List[dict]] = None,
    category: Optional[str] = None,
    avg_sales: Optional[float] = None,
    avg_margin: Optional[float] = None,
    targets: Optional[List[Dict[str, Any]]] = None,  # Issue #510: Benchmark lines
) -> html.Div:
    """
    Crea el scatter plot de posicionamiento competitivo.

    Args:
        data_points: Lista de productos con x (ventas), y (margen), etc.
        category: Categoría NECESIDAD
        avg_sales: Ventas promedio de la categoría
        avg_margin: Margen promedio de la categoría
        targets: List of benchmark targets for margin lines (Issue #510)
            [{value, color, line_style, label}, ...]

    Returns:
        html.Div con el gráfico scatter
    """
    if not data_points:
        return create_scatter_placeholder()

    fig = _create_scatter_figure(data_points, avg_sales, avg_margin, targets)

    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-chart-scatter me-2"),
                html.Span("Posicionamiento Competitivo", className="fw-bold"),
            ], className="d-flex align-items-center"),
            html.Small(
                f"Categoría: {category.replace('_', ' ').title() if category else 'N/A'}",
                className="text-muted d-block mt-1",
            ),
        ]),
        dbc.CardBody([
            dcc.Graph(
                id="ventalibre-competitive-scatter",
                figure=fig,
                config={"responsive": True, "displayModeBar": False},
                style={"height": "350px"},
            ),
        ]),
    ], className="shadow-sm")


def _create_scatter_figure(
    data_points: List[dict],
    avg_sales: Optional[float],
    avg_margin: Optional[float],
    targets: Optional[List[Dict[str, Any]]] = None,  # Issue #510
) -> go.Figure:
    """Crea la figura del scatter plot."""
    # Separar producto seleccionado de competidores
    selected = [p for p in data_points if p.get("is_selected")]
    competitors = [p for p in data_points if not p.get("is_selected")]

    fig = go.Figure()

    # Competidores (puntos grises)
    if competitors:
        fig.add_trace(go.Scatter(
            x=[p["x"] for p in competitors],
            y=[p["y"] for p in competitors],
            mode="markers",
            marker=dict(
                size=10,
                color="rgba(150, 150, 150, 0.6)",
                line=dict(width=1, color="rgba(100, 100, 100, 0.8)"),
            ),
            name="Competidores",
            hovertemplate=(
                "<b>%{customdata[0]}</b><br>"
                "CN: %{customdata[1]}<br>"
                "Ventas: €%{x:,.0f}<br>"
                "Margen: %{y:.1f}%<br>"
                "<extra></extra>"
            ),
            customdata=[
                [p.get("product_name", ""), p.get("codigo_nacional", "")]
                for p in competitors
            ],
        ))

    # Producto seleccionado (punto destacado)
    if selected:
        p = selected[0]
        fig.add_trace(go.Scatter(
            x=[p["x"]],
            y=[p["y"]],
            mode="markers+text",
            marker=dict(
                size=18,
                color=COLORS["primary"],
                line=dict(width=2, color=COLORS["primary_dark"]),
                symbol="star",
            ),
            text=["Seleccionado"],
            textposition="top center",
            name="Producto",
            hovertemplate=(
                "<b>%{customdata[0]}</b><br>"
                "CN: %{customdata[1]}<br>"
                "Ventas: €%{x:,.0f}<br>"
                "Margen: %{y:.1f}%<br>"
                "<extra></extra>"
            ),
            customdata=[[p.get("product_name", ""), p.get("codigo_nacional", "")]],
        ))

    # Líneas de referencia (promedios)
    if avg_sales:
        fig.add_vline(
            x=avg_sales,
            line_dash="dash",
            line_color="rgba(0, 0, 0, 0.3)",
            annotation_text="Ventas Prom.",
            annotation_position="top",
        )
    if avg_margin:
        fig.add_hline(
            y=avg_margin,
            line_dash="dash",
            line_color="rgba(0, 0, 0, 0.3)",
            annotation_text="Margen Prom.",
            annotation_position="right",
        )

    # Issue #510: Add target benchmark lines
    if targets:
        add_benchmark_lines(fig, targets, axis="y", annotation_position="left")

    # Layout
    fig.update_layout(
        xaxis_title="Ventas (€)",
        yaxis_title="Margen (%)",
        showlegend=True,
        legend=dict(orientation="h", yanchor="bottom", y=1.02, xanchor="right", x=1),
        margin=dict(t=40, b=40, l=50, r=20),
        plot_bgcolor="rgba(248, 249, 250, 1)",
        hovermode="closest",
    )

    # Cuadrantes con colores
    fig.update_xaxes(showgrid=True, gridcolor="rgba(0, 0, 0, 0.1)")
    fig.update_yaxes(showgrid=True, gridcolor="rgba(0, 0, 0, 0.1)")

    return fig


def create_scatter_placeholder() -> html.Div:
    """Placeholder cuando no hay producto seleccionado."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-chart-scatter me-2"),
            html.Span("Posicionamiento Competitivo", className="fw-bold"),
        ]),
        dbc.CardBody([
            html.Div([
                html.I(className="fas fa-bullseye fa-3x text-muted mb-2"),
                html.P(
                    "Selecciona un producto para ver su posicionamiento",
                    className="text-muted mb-0",
                ),
            ], className="text-center py-4"),
        ]),
    ], className="shadow-sm bg-light")


def create_quadrant_legend() -> html.Div:
    """Leyenda de cuadrantes para el scatter."""
    return html.Div([
        dbc.Row([
            dbc.Col([
                html.Small("⭐ Estrellas", className="text-success fw-bold"),
                html.Small(" (Alto margen + Altas ventas)", className="text-muted"),
            ], width=6),
            dbc.Col([
                html.Small("🐄 Vacas Lecheras", className="text-primary fw-bold"),
                html.Small(" (Alto margen + Bajas ventas)", className="text-muted"),
            ], width=6),
        ], className="mb-1"),
        dbc.Row([
            dbc.Col([
                html.Small("❓ Interrogantes", className="text-warning fw-bold"),
                html.Small(" (Bajo margen + Altas ventas)", className="text-muted"),
            ], width=6),
            dbc.Col([
                html.Small("🐕 Perros", className="text-danger fw-bold"),
                html.Small(" (Bajo margen + Bajas ventas)", className="text-muted"),
            ], width=6),
        ]),
    ], className="small mt-2")
