"""
Constantes de categorías NECESIDAD para visualizaciones (Issue #488).

Extraído de dendrogram.py para uso en necesidad_treemap.py y callbacks.
"""

from typing import Dict

# 21 Principal categories with display names and colors (Issue #467)
PRINCIPAL_CATEGORIES = {
    "dolor_fiebre": {"name": "Dolor y Fiebre", "color": "#e74c3c"},
    "respiratorio": {"name": "Respiratorio", "color": "#3498db"},
    "digestivo": {"name": "Digestivo", "color": "#2ecc71"},
    "piel": {"name": "Dermocosmética", "color": "#f39c12"},
    "bucal": {"name": "Bucal", "color": "#9b59b6"},
    "ocular": {"name": "Ocular", "color": "#1abc9c"},
    "oidos": {"name": "Oidos", "color": "#e67e22"},
    "capilar": {"name": "Capilar", "color": "#8e44ad"},
    "circulatorio": {"name": "Circulatorio", "color": "#c0392b"},
    "urinario": {"name": "Urinario", "color": "#16a085"},
    "sexual": {"name": "Sexual", "color": "#d35400"},
    "infantil": {"name": "Infantil", "color": "#f1c40f"},
    "nutricion": {"name": "Suplementos", "color": "#2980b9"},
    "movilidad": {"name": "Movilidad", "color": "#7f8c8d"},
    "incontinencia_cat": {"name": "Incontinencia", "color": "#95a5a6"},
    "diabetes_cat": {"name": "Diabetes", "color": "#34495e"},
    "botiquin": {"name": "Botiquin", "color": "#bdc3c7"},
    "veterinaria_cat": {"name": "Veterinaria", "color": "#795548"},
    "servicios_cat": {"name": "Servicios", "color": "#607d8b"},
    "no_salud": {"name": "No Salud", "color": "#9e9e9e"},
}

# Mapping: Especifica -> Principal (from TAXONOMY_HIERARCHY)
ESPECIFICA_TO_PRINCIPAL: Dict[str, str] = {
    # DOLOR_FIEBRE
    "dolor_muscular": "dolor_fiebre",
    "dolor_cabeza": "dolor_fiebre",
    "dolor_articular": "dolor_fiebre",
    "dolor_menstrual": "dolor_fiebre",
    "dolor_dental": "dolor_fiebre",
    "dolor_fiebre": "dolor_fiebre",
    "dolor_garganta": "dolor_fiebre",
    "hematomas": "dolor_fiebre",
    # RESPIRATORIO
    "tos_seca": "respiratorio",
    "mucosidad_respiratoria": "respiratorio",
    "congestion_nasal": "respiratorio",
    "higiene_nasal": "respiratorio",
    "gripe_resfriado": "respiratorio",
    "alergias": "respiratorio",
    "asma": "respiratorio",
    "asma_bronquitis": "respiratorio",
    "balsamos_respiratorios": "respiratorio",
    # DIGESTIVO
    "acidez_reflujo": "digestivo",
    "diarrea": "digestivo",
    "estrenimiento": "digestivo",
    "gases_flatulencia": "digestivo",
    "nauseas_vomitos": "digestivo",
    "digestion_pesada": "digestivo",
    "digestivo": "digestivo",
    "colon_irritable": "digestivo",
    "colicos": "digestivo",
    "probioticos": "digestivo",
    "parasitos": "digestivo",
    # PIEL
    "hidratacion_corporal": "piel",
    "higiene_corporal": "piel",
    "desodorante": "piel",
    "hidratacion_facial": "piel",
    "proteccion_solar": "piel",
    "acne": "piel",
    "dermatitis": "piel",
    "dermatitis_eccema": "piel",
    "piel_atopica": "piel",
    "psoriasis": "piel",
    "cicatrizacion": "piel",
    "picaduras": "piel",
    "quemaduras": "piel",
    "hongos_piel": "piel",
    "herpes": "piel",
    "sarna": "piel",
    "verrugas_callos": "piel",
    "manchas": "piel",
    "arrugas_antiedad": "piel",
    "contorno_ojos": "piel",
    "limpieza_facial": "piel",
    "exfoliacion": "piel",
    "celulitis": "piel",
    "estrias": "piel",
    "bronceado": "piel",
    "cuidado_labial": "piel",
    "accesorios_bano": "piel",
    "unas": "piel",
    "tatuajes": "piel",
    # BUCAL
    "cepillo_manual": "bucal",
    "cepillo_electrico": "bucal",
    "pasta_dental": "bucal",
    "enjuague_bucal": "bucal",
    "seda_dental": "bucal",
    "interdental": "bucal",
    "irrigador_dental": "bucal",
    "higiene_bucal_infantil": "bucal",
    "higiene_dental": "bucal",
    "encias": "bucal",
    "aftas_llagas": "bucal",
    "blanqueamiento": "bucal",
    "protesis_dental": "bucal",
    "halitosis": "bucal",
    "sensibilidad_dental": "bucal",
    # OCULAR
    "ojo_seco": "ocular",
    "conjuntivitis": "ocular",
    "alergia_ocular": "ocular",
    "fatiga_visual": "ocular",
    "lentillas": "ocular",
    "higiene_ocular": "ocular",
    "optica": "ocular",
    # OIDOS
    "higiene_oidos": "oidos",
    "cerumen": "oidos",
    "dolor_oidos": "oidos",
    "otitis": "oidos",
    # CAPILAR
    "caida_cabello": "capilar",
    "caspa": "capilar",
    "cabello_graso": "capilar",
    "piojos": "capilar",
    "capilar_cuero_sensible": "capilar",
    "capilar_hidratacion": "capilar",
    "capilar_color": "capilar",
    "capilar_densidad": "capilar",
    "capilar_frecuente": "capilar",
    "canas": "capilar",
    "cuidado_cabello": "capilar",
    # BIENESTAR (fusionado en nutricion)
    "sueno_insomnio": "nutricion",
    "melatonina": "nutricion",
    "estres_ansiedad": "nutricion",
    "energia_vitalidad": "nutricion",
    "concentracion_memoria": "nutricion",
    "estado_animo": "nutricion",
    "mareo_viaje": "nutricion",
    # CIRCULATORIO
    "piernas_cansadas": "circulatorio",
    "varices": "circulatorio",
    "hemorroides": "circulatorio",
    "tension_arterial": "circulatorio",
    "colesterol": "circulatorio",
    "retencion_liquidos": "circulatorio",
    # SEXUAL
    "salud_sexual": "sexual",
    "fertilidad": "sexual",
    "menopausia": "sexual",
    "higiene_intima": "sexual",
    "flora_vaginal": "sexual",
    "candidiasis_vaginal": "sexual",
    "vaginosis": "sexual",
    "ciclo_menstrual": "sexual",
    "embarazo_lactancia": "sexual",
    "disfuncion_erectil": "sexual",
    # INFANTIL
    "alimentacion_bebe": "infantil",
    "higiene_bebe": "infantil",
    "colicos_bebe": "infantil",
    "denticion": "infantil",
    "dermatitis_panal": "infantil",
    "fiebre_bebe": "infantil",
    "bebes_infantil": "infantil",
    # NUTRICION
    "vitaminas_minerales": "nutricion",
    "vitaminas_general": "nutricion",
    "vitaminas_infantil": "nutricion",
    "omega3": "nutricion",
    "hierro": "nutricion",
    "calcio_vitamina_d": "nutricion",
    "vitamina_d": "nutricion",
    "huesos": "nutricion",
    "magnesio": "nutricion",
    "proteinas_deporte": "nutricion",
    "nutricion_deportiva": "nutricion",
    "control_peso": "nutricion",
    "detox": "nutricion",
    "colageno": "nutricion",
    "nutricion_clinica": "nutricion",
    "hidratacion": "nutricion",
    "infusiones": "nutricion",
    # URINARIO
    "infeccion_urinaria": "urinario",
    "prostata": "urinario",
    # INCONTINENCIA
    "incontinencia": "incontinencia_cat",
    "cuidado_dependientes": "incontinencia_cat",
    # DIABETES
    "diabetes": "diabetes_cat",
    # MOVILIDAD
    "material_ortopedico": "movilidad",
    "ortopedia_pie": "movilidad",
    "rehabilitacion": "movilidad",
    "compresion_vendajes": "movilidad",
    "ayudas_tecnicas": "movilidad",
    # BOTIQUIN
    "desinfeccion_heridas": "botiquin",
    "apositos_curas": "botiquin",
    "termometros": "botiquin",
    "test_embarazo": "botiquin",
    "test_ovulacion": "botiquin",
    "test_covid_gripe": "botiquin",
    "test_diagnostico": "botiquin",
    "material_sanitario": "botiquin",
    # Reclasificados fuera de botiquín
    "servicios_farmacia": "servicios_cat",
    "regalos_bazar": "no_salud",
    "dulceria": "no_salud",
    "perfumeria": "no_salud",
    "joyeria": "no_salud",
    "veterinaria": "veterinaria_cat",
    "otros": "botiquin",
    # Aliases
    "dolor": "dolor_fiebre",
    "alergia": "respiratorio",
    "flora_intestinal": "digestivo",
    "hongos_pies": "piel",
    "circulacion_piernas": "circulatorio",
}


def get_principal_category(especifica: str) -> str:
    """Get the principal category for a specific category."""
    if not especifica:
        return "botiquin"
    return ESPECIFICA_TO_PRINCIPAL.get(especifica.lower(), "botiquin")


def get_principal_info(principal: str) -> Dict:
    """Get display info for a principal category."""
    return PRINCIPAL_CATEGORIES.get(principal, {"name": principal.title(), "color": "#95a5a6"})


def format_necesidad_name(category: str) -> str:
    """
    Convierte categoría NECESIDAD a nombre legible para visualizaciones.

    Args:
        category: Nombre interno de categoría (e.g., "dolor_fiebre", "piel")

    Returns:
        Nombre formateado (e.g., "Dolor y Fiebre", "Dermocosmética")
    """
    if not category:
        return "Sin categoría"

    principal_info = PRINCIPAL_CATEGORIES.get(category)
    if principal_info:
        return principal_info.get("name", category.replace("_", " ").title())

    return category.replace("_", " ").title()


# =============================================================================
# L2 SUBCATEGORIES (Issue #505, ADR-004)
# =============================================================================
# 28 subcategorías para 6 categorías L1 (expanded Q1 2026)
# Principio: "No clasificamos por lo que el producto ES, sino por CÓMO se vende"
# =============================================================================

# L1 categories that have L2 drill-down (synced with backend L1WithL2Category)
L1_WITH_L2 = {"dermocosmetica", "suplementos", "higiene_bucal", "infantil", "sexual", "control_peso"}

# L2 subcategories with display info
L2_CATEGORIES: Dict[str, Dict] = {
    # DERMOCOSMÉTICA (7 L2)
    "higiene_limpieza": {"name": "Higiene y Limpieza", "color": "#e74c3c", "archetype": "🪝 Gancho", "l1": "dermocosmetica"},
    "tratamiento_avanzado": {"name": "Tratamiento Avanzado", "color": "#9b59b6", "archetype": "💎 Joyas", "l1": "dermocosmetica"},
    "hidratacion_basica": {"name": "Hidratación Básica", "color": "#3498db", "archetype": "🔄 Mantenimiento", "l1": "dermocosmetica"},
    "solar_facial": {"name": "Solar Facial", "color": "#f39c12", "archetype": "🎯 Estratégico", "l1": "dermocosmetica"},
    "solar_corporal": {"name": "Solar Corporal", "color": "#e67e22", "archetype": "📅 Estacional", "l1": "dermocosmetica"},
    "capilar_tratante": {"name": "Capilar Tratante", "color": "#8e44ad", "archetype": "🔧 Específico", "l1": "dermocosmetica"},
    "acne_piel_grasa": {"name": "Acné y Piel Grasa", "color": "#1abc9c", "archetype": "🔒 Fidelización", "l1": "dermocosmetica"},
    # SUPLEMENTOS (6 L2)
    "descanso_estres": {"name": "Descanso y Estrés", "color": "#9b59b6", "archetype": "🔄 Recurrencia", "l1": "suplementos"},
    "energia_vitalidad": {"name": "Energía y Vitalidad", "color": "#f1c40f", "archetype": "📅 Estacional", "l1": "suplementos"},
    "articular_osea": {"name": "Articular y Ósea", "color": "#7f8c8d", "archetype": "👴 Cronicidad", "l1": "suplementos"},
    "defensas_inmunidad": {"name": "Defensas e Inmunidad", "color": "#3498db", "archetype": "❄️ Coyuntural", "l1": "suplementos"},
    "digestivo_probioticos": {"name": "Digestivo y Probióticos", "color": "#2ecc71", "archetype": "📈 Tendencia", "l1": "suplementos"},
    "control_peso_l2": {"name": "Control de Peso", "color": "#e74c3c", "archetype": "🏃 Zafarrancho", "l1": "suplementos"},
    # HIGIENE BUCAL (5 L2)
    "higiene_diaria_basica": {"name": "Higiene Diaria Básica", "color": "#95a5a6", "archetype": "🛒 Commodity", "l1": "higiene_bucal"},
    "sensibilidad_encias": {"name": "Sensibilidad y Encías", "color": "#e74c3c", "archetype": "🏥 Core Farmacia", "l1": "higiene_bucal"},
    "estetica_blanqueamiento": {"name": "Estética y Blanqueamiento", "color": "#f1c40f", "archetype": "✨ Deseo", "l1": "higiene_bucal"},
    "tratamiento_bucal": {"name": "Tratamiento Bucal", "color": "#9b59b6", "archetype": "💊 Nicho", "l1": "higiene_bucal"},
    "ortodoncia_interdentales": {"name": "Ortodoncia e Interdentales", "color": "#3498db", "archetype": "➕ Complementarios", "l1": "higiene_bucal"},
    # INFANTIL (4 L2) - ADR-004
    "nutricion_infantil": {"name": "Nutrición Infantil", "color": "#f39c12", "archetype": "🍼 Tráfico", "l1": "infantil"},
    "higiene_bebe": {"name": "Higiene Bebé", "color": "#3498db", "archetype": "🔄 Recurrencia", "l1": "infantil"},
    "cuidados_especificos": {"name": "Cuidados Específicos", "color": "#9b59b6", "archetype": "🏥 Consejo", "l1": "infantil"},
    "biberoneria_accesorios": {"name": "Biberonería y Accesorios", "color": "#1abc9c", "archetype": "➕ Complementarios", "l1": "infantil"},
    # SEXUAL (3 L2) - ADR-004
    "preservativos_lubricantes": {"name": "Preservativos y Lubricantes", "color": "#e74c3c", "archetype": "💰 Alto Margen", "l1": "sexual"},
    "test_fertilidad_embarazo": {"name": "Test Fertilidad/Embarazo", "color": "#9b59b6", "archetype": "🔬 Diagnóstico", "l1": "sexual"},
    "salud_intima_femenina": {"name": "Salud Íntima Femenina", "color": "#e67e22", "archetype": "🏥 Consejo", "l1": "sexual"},
    # CONTROL PESO (3 L2) - ADR-004
    "sustitutivos_comida": {"name": "Sustitutivos de Comida", "color": "#2ecc71", "archetype": "🥗 Sustitución", "l1": "control_peso"},
    "quemagrasas_drenantes": {"name": "Quemagrasas y Drenantes", "color": "#e74c3c", "archetype": "🔥 Acción Rápida", "l1": "control_peso"},
    "control_apetito": {"name": "Control de Apetito", "color": "#f39c12", "archetype": "🎯 Complementario", "l1": "control_peso"},
}

# Mapping L1 -> L2 list
L1_TO_L2: Dict[str, list] = {
    "dermocosmetica": [
        "higiene_limpieza", "tratamiento_avanzado", "hidratacion_basica",
        "solar_facial", "solar_corporal", "capilar_tratante", "acne_piel_grasa"
    ],
    "suplementos": [
        "descanso_estres", "energia_vitalidad", "articular_osea",
        "defensas_inmunidad", "digestivo_probioticos", "control_peso_l2"
    ],
    "higiene_bucal": [
        "higiene_diaria_basica", "sensibilidad_encias", "estetica_blanqueamiento",
        "tratamiento_bucal", "ortodoncia_interdentales"
    ],
    # ADR-004: Added Q1 2026
    "infantil": [
        "nutricion_infantil", "higiene_bebe", "cuidados_especificos",
        "biberoneria_accesorios"
    ],
    "sexual": [
        "preservativos_lubricantes", "test_fertilidad_embarazo",
        "salud_intima_femenina"
    ],
    "control_peso": [
        "sustitutivos_comida", "quemagrasas_drenantes", "control_apetito"
    ],
}


def has_l2_subcategories(l1_category: str) -> bool:
    """Check if an L1 category has L2 drill-down available."""
    return l1_category in L1_WITH_L2


def get_l2_for_l1(l1_category: str) -> list:
    """Get L2 subcategories for an L1 category."""
    return L1_TO_L2.get(l1_category, [])


def get_l2_info(l2_category: str) -> Dict:
    """Get display info for an L2 subcategory."""
    return L2_CATEGORIES.get(l2_category, {"name": l2_category.title(), "color": "#95a5a6", "archetype": ""})
