"""
Componentes para Tab "Categorías y Marcas" (Issue #493)

Análisis de marcas por categoría NECESIDAD con métricas de nivel consultoría:
- HHI (concentración de mercado)
- Cuadrante de Valor (Margen vs Volumen)
- Brand Duel Mode (comparación lado a lado)
- Evolución de cuota de mercado
- Boxplot de precios (canibalización)
"""

import dash_bootstrap_components as dbc
import plotly.express as px
import plotly.graph_objects as go
from dash import dcc, html

from styles.design_tokens import COLORS, SPACING


# Colores para cuadrantes de valor
QUADRANT_COLORS = {
    "star": "#28a745",       # Verde - Estrellas
    "traffic": "#17a2b8",    # Azul - Generadores de tráfico
    "opportunity": "#ffc107", # Amarillo - Oportunidades
    "review": "#dc3545",     # Rojo - Revisar
}


def create_hhi_education_modal():
    """
    Modal educativo sobre el índice HHI con enfoque en capas (UX Cebolla).

    Capa 1: Texto corto con los 5 diagnósticos (directo al grano)
    Capa 2: Acordeón con teoría profunda (trade-offs, checklist, etc.)

    Returns:
        dbc.Modal con contenido educativo en capas
    """
    # Capa 1: Los 5 diagnósticos (texto corto, visible siempre)
    diagnostics_summary = html.Div([
        html.P([
            "Cruzamos tu ",
            html.Strong("concentración de compras (HHI)"),
            " con tu ",
            html.Strong("rentabilidad"),
            " para detectar tu estrategia real:",
        ], className="mb-3"),

        # Los 5 diagnósticos
        html.Div([
            html.Div([
                html.Span("🎯", style={"fontSize": "1.3rem"}),
                html.Strong(" Especialista Exitoso", className="text-success"),
                html.Span(" — Pocas marcas, pero muy rentables. Eres socio clave de tus proveedores.",
                          className="text-muted"),
            ], className="mb-2"),
            html.Div([
                html.Span("⚠️", style={"fontSize": "1.3rem"}),
                html.Strong(" Riesgo de Dependencia", className="text-danger"),
                html.Span(" — Pocas marcas y baja rentabilidad. Estás 'casado' con proveedores "
                          "que no te pagan esa exclusividad.",
                          className="text-muted"),
            ], className="mb-2"),
            html.Div([
                html.Span("⭐", style={"fontSize": "1.3rem"}),
                html.Strong(" Generalista Premium", className="text-primary"),
                html.Span(" — Mucha variedad y buen margen. Control total y el cliente paga tu surtido.",
                          className="text-muted"),
            ], className="mb-2"),
            html.Div([
                html.Span("📊", style={"fontSize": "1.3rem"}),
                html.Strong(" Generalista de Volumen", className="text-info"),
                html.Span(" — Mucha variedad a bajo margen. Tu negocio es la rotación y cobertura total.",
                          className="text-muted"),
            ], className="mb-2"),
            html.Div([
                html.Span("⚖️", style={"fontSize": "1.3rem"}),
                html.Strong(" Surtido Balanceado", className="text-secondary"),
                html.Span(" — En la media del mercado. Monitorizar sin alarmas.",
                          className="text-muted"),
            ]),
        ], className="ps-2"),
    ])

    # Capa 2: Contenido profundo (en acordeón)
    deep_content = dbc.Accordion([
        # Acordeón 1: Escala de referencia
        dbc.AccordionItem([
            dbc.Row([
                dbc.Col([
                    dbc.Card([
                        dbc.CardBody([
                            html.Div("< 1.500", className="h5 text-success mb-1"),
                            html.Div("Diversificado", className="fw-bold small"),
                        ], className="text-center py-2"),
                    ], className="border-success"),
                ], width=4),
                dbc.Col([
                    dbc.Card([
                        dbc.CardBody([
                            html.Div("1.500 - 2.500", className="h5 text-warning mb-1"),
                            html.Div("Moderado", className="fw-bold small"),
                        ], className="text-center py-2"),
                    ], className="border-warning"),
                ], width=4),
                dbc.Col([
                    dbc.Card([
                        dbc.CardBody([
                            html.Div("> 2.500", className="h5 text-danger mb-1"),
                            html.Div("Concentrado", className="fw-bold small"),
                        ], className="text-center py-2"),
                    ], className="border-danger"),
                ], width=4),
            ]),
        ], title="📏 Escala de Referencia HHI"),

        # Acordeón 2: Estrategias comparadas
        dbc.AccordionItem([
            dbc.Row([
                dbc.Col([
                    html.H6([
                        html.I(className="fas fa-expand-arrows-alt me-2 text-info"),
                        "Generalista (HHI Bajo)",
                    ], className="mb-2"),
                    html.Ul([
                        html.Li("Gama amplia de marcas"),
                        html.Li("Gana por cobertura"),
                        html.Li("Costes operativos más altos"),
                        html.Li([html.Strong("Resiliencia: "), "Alta"]),
                    ], className="small"),
                ], width=6),
                dbc.Col([
                    html.H6([
                        html.I(className="fas fa-crosshairs me-2 text-primary"),
                        "Especialista (HHI Alto)",
                    ], className="mb-2"),
                    html.Ul([
                        html.Li("Pocas marcas, mucho volumen"),
                        html.Li("Eficiencia logística"),
                        html.Li("Mejores rappels/descuentos"),
                        html.Li([html.Strong("Resiliencia: "), "Depende del Plan B"]),
                    ], className="small"),
                ], width=6),
            ]),
            html.P([
                html.I(className="fas fa-lightbulb me-2 text-warning"),
                "Un HHI alto ",
                html.Strong("no es un error"),
                " si es una decisión estratégica con contrapartida en margen.",
            ], className="mt-3 mb-0 small"),
        ], title="⚖️ Generalista vs. Especialista"),

        # Acordeón 3: Checklist del especialista
        dbc.AccordionItem([
            html.P("Si tu HHI es alto por elección, verifica estas 3 condiciones:", className="mb-3"),
            html.Div([
                html.Div([
                    html.I(className="fas fa-euro-sign me-2 text-success"),
                    html.Strong("1. Contrapartida en Margen"),
                    html.P([
                        "¿Te pagan la fidelidad? Mejores precios, exclusividad, rappels. ",
                        html.Span("Si tienes HHI alto y mismo margen que la competencia, pierdes.",
                                  className="text-danger"),
                    ], className="small text-muted ps-4 mb-2"),
                ]),
                html.Div([
                    html.I(className="fas fa-star me-2 text-success"),
                    html.Strong("2. Producto Diferenciador"),
                    html.P("Lo que vendes es tan bueno que el cliente no echa de menos otras opciones.",
                           className="small text-muted ps-4 mb-2"),
                ]),
                html.Div([
                    html.I(className="fas fa-shield-alt me-2 text-success"),
                    html.Strong("3. Plan de Contingencia"),
                    html.P("Si tu proveedor principal falla, tienes alternativa rápida. "
                           "Riesgo de ruptura = cero.",
                           className="small text-muted ps-4 mb-0"),
                ]),
            ]),
        ], title="✅ Checklist: ¿Mi HHI Alto es Seguro?"),

        # Acordeón 4: Fuentes de datos de mercado
        dbc.AccordionItem([
            html.P([
                "Para validar si el mercado también es concentrado, busca datos en:",
            ], className="mb-2"),
            html.Ul([
                html.Li([html.Strong("IQVIA / IMS Health: "), "Cuotas de mercado por categoría."]),
                html.Li([html.Strong("Tu distribuidor: "), "Datos de ventas agregadas de tu zona."]),
                html.Li([html.Strong("Cooperativas: "), "FEFE, colegios profesionales."]),
                html.Li([html.Strong("Estudios sectoriales: "), "Consultoras de retail farmacéutico."]),
            ], className="small"),
        ], title="🔍 ¿Dónde Conseguir el HHI de Mercado?"),
    ], start_collapsed=True, className="mt-4")

    return dbc.Modal([
        dbc.ModalHeader(
            dbc.ModalTitle([
                html.I(className="fas fa-graduation-cap me-2"),
                "¿Qué me dice el Diagnóstico de HHI?",
            ]),
            close_button=True,
        ),
        dbc.ModalBody([
            # Capa 1: Diagnósticos (siempre visible)
            diagnostics_summary,

            # Separador
            html.Hr(className="my-3"),

            # Capa 2: Contenido profundo (acordeón)
            html.P([
                html.I(className="fas fa-book-open me-2 text-secondary"),
                html.Strong("Profundizar en el concepto"),
            ], className="text-muted small mb-2"),
            deep_content,
        ]),
        dbc.ModalFooter([
            html.Small(
                "💡 HHI alto = Apuesta estratégica. Asegúrate de que te compensa.",
                className="text-muted me-auto",
            ),
            dbc.Button(
                "Entendido",
                id="ventalibre-hhi-modal-close",
                color="primary",
            ),
        ]),
    ],
        id="ventalibre-hhi-education-modal",
        size="lg",
        scrollable=True,
        is_open=False,
    )


def create_category_selector():
    """
    Selector de categoría NECESIDAD con dropdown.

    Returns:
        dbc.Card con dropdown de categorías
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-tags me-2", style={"color": COLORS["info"]}),
            "Seleccionar Categoría",
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Dropdown(
                id="ventalibre-brands-necesidad-dropdown",
                placeholder="Selecciona una categoría NECESIDAD...",
                clearable=False,
                style={"marginBottom": SPACING.get("m", "1rem")},
            ),
            html.Small(
                "Selecciona una categoría para analizar sus marcas",
                className="text-muted",
            ),
        ]),
    ], className="h-100 shadow-sm")


def create_hhi_indicator():
    """
    Indicador visual de concentración HHI (Issue #539 - Mejora UX).

    Returns:
        dbc.Card con gauge/badge de HHI + interpretación contextualizada + escala visual
    """
    # Tooltip resumido (el detalle está en el modal)
    hhi_help_content = html.Div([
        html.P([
            html.Strong("Tu HHI Interno: "),
            "Mide tu dependencia de proveedores en esta categoría.",
        ], className="mb-2 small"),
        html.Ul([
            html.Li([
                html.Span("< 1.500", style={"color": "#28a745", "fontWeight": "bold"}),
                " Diversificado",
            ], className="small"),
            html.Li([
                html.Span("1.500 - 2.500", style={"color": "#ffc107", "fontWeight": "bold"}),
                " Moderado",
            ], className="small"),
            html.Li([
                html.Span("> 2.500", style={"color": "#dc3545", "fontWeight": "bold"}),
                " Alta dependencia",
            ], className="small"),
        ], className="mb-2 ps-3"),
        html.P([
            "Haz clic en ",
            html.Strong("'Entender HHI'"),
            " para ver cómo interpretar este dato.",
        ], className="mb-0 small text-muted"),
    ], style={"maxWidth": "280px"})

    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-chart-line me-2", style={"color": COLORS["primary"]}),
                "Tu Dependencia (HHI)",
                dbc.Badge(
                    "?",
                    id="ventalibre-hhi-help-badge",
                    color="secondary",
                    className="ms-2",
                    style={"cursor": "pointer"},
                ),
                dbc.Popover(
                    hhi_help_content,
                    target="ventalibre-hhi-help-badge",
                    trigger="hover",
                    placement="bottom",
                ),
            ], className="d-flex align-items-center"),
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-hhi-loading",
                type="circle",
                children=html.Div(id="ventalibre-hhi-indicator"),
            ),
            # Botón para abrir modal educativo
            html.Div([
                dbc.Button(
                    html.Span([
                        html.I(className="fas fa-graduation-cap me-2"),
                        "Entender HHI",
                    ]),
                    id="ventalibre-hhi-modal-open",
                    color="link",
                    size="sm",
                    className="p-0 mt-2",
                ),
            ], className="text-center"),
        ]),
    ], className="h-100 shadow-sm")


def _create_hhi_scale_visual(hhi: float) -> html.Div:
    """
    Crear escala visual con zonas coloreadas para HHI (Issue #539).

    Args:
        hhi: Valor del HHI (0-10000)

    Returns:
        html.Div con escala visual
    """
    # Calcular posición del marcador (0-100%)
    marker_pct = min(100, max(0, (hhi / 10000) * 100))

    # Zonas coloreadas: Atomizado (0-15%), Moderado (15-25%), Oligopolio (25-100%)
    # Nota: 1500/10000 = 15%, 2500/10000 = 25%
    return html.Div([
        # Barra con zonas coloreadas
        html.Div([
            # Zona Atomizado (verde)
            html.Div(style={
                "position": "absolute",
                "left": "0%",
                "width": "15%",
                "height": "100%",
                "backgroundColor": "#28a745",
                "borderRadius": "4px 0 0 4px",
            }),
            # Zona Moderado (amarillo)
            html.Div(style={
                "position": "absolute",
                "left": "15%",
                "width": "10%",
                "height": "100%",
                "backgroundColor": "#ffc107",
            }),
            # Zona Oligopolio (rojo)
            html.Div(style={
                "position": "absolute",
                "left": "25%",
                "width": "75%",
                "height": "100%",
                "backgroundColor": "#dc3545",
                "borderRadius": "0 4px 4px 0",
            }),
            # Marcador de posición actual
            html.Div([
                html.Div(style={
                    "width": "0",
                    "height": "0",
                    "borderLeft": "6px solid transparent",
                    "borderRight": "6px solid transparent",
                    "borderTop": "8px solid #212529",
                }),
            ], style={
                "position": "absolute",
                "left": f"calc({marker_pct}% - 6px)",
                "top": "-10px",
            }),
        ], style={
            "position": "relative",
            "height": "12px",
            "marginBottom": "8px",
            "marginTop": "12px",
        }),
        # Etiquetas de la escala
        html.Div([
            html.Span("0", className="small text-muted"),
            html.Span("1.500", className="small text-muted", style={"position": "absolute", "left": "15%", "transform": "translateX(-50%)"}),
            html.Span("2.500", className="small text-muted", style={"position": "absolute", "left": "25%", "transform": "translateX(-50%)"}),
            html.Span("10.000", className="small text-muted", style={"position": "absolute", "right": "0"}),
        ], style={"position": "relative", "height": "20px"}),
    ])


def calculate_hhi_diagnosis(
    hhi: float,
    avg_margin: float,
    category_avg_margin: float = None,
) -> dict:
    """
    Calcular diagnóstico cruzando HHI × Margen.

    Lógica de 5 diagnósticos:
    - 🎯 Especialista Exitoso: HHI alto + Margen alto
    - ⚠️ Riesgo de Dependencia: HHI alto + Margen bajo
    - ⭐ Generalista Premium: HHI bajo + Margen alto
    - 📊 Generalista de Volumen: HHI bajo + Margen bajo
    - ⚖️ Surtido Balanceado: Zona intermedia (HHI medio o margen neutro)

    Args:
        hhi: Valor HHI de la categoría (0-10000)
        avg_margin: Margen promedio de la farmacia en esta categoría (%)
        category_avg_margin: Margen promedio de referencia. Si None, usa umbrales fijos.

    Returns:
        Dict con emoji, title, description, color, css_class, action
    """
    # Umbrales HHI
    HHI_LOW = 1500
    HHI_HIGH = 2500

    # Umbrales de margen
    if category_avg_margin is not None and category_avg_margin > 0:
        # Modo relativo: buffer ±5% sobre baseline
        MARGIN_BUFFER = 0.05
        margin_high_threshold = category_avg_margin * (1 + MARGIN_BUFFER)
        margin_low_threshold = category_avg_margin * (1 - MARGIN_BUFFER)
    else:
        # Modo absoluto: umbrales fijos para Fase 1
        # Estos valores son típicos para parafarmacia/OTC
        margin_high_threshold = 25.0  # >25% = Alto
        margin_low_threshold = 15.0   # <15% = Bajo

    # Clasificar HHI
    hhi_is_high = hhi > HHI_HIGH
    hhi_is_low = hhi < HHI_LOW

    # Clasificar Margen
    margin_is_high = avg_margin > margin_high_threshold
    margin_is_low = avg_margin < margin_low_threshold

    # Diagnóstico según matriz (4 cuadrantes + zona neutra)
    if hhi_is_high and margin_is_high:
        return {
            "emoji": "🎯",
            "title": "Especialista Exitoso",
            "description": "Pocas marcas, pero muy rentables. Socio clave de proveedores.",
            "color": "success",
            "css_class": "text-success",
            "action": "Mantener estrategia. Negociar exclusividades.",
        }
    elif hhi_is_high and margin_is_low:
        return {
            "emoji": "⚠️",
            "title": "Riesgo de Dependencia",
            "description": "Pocas marcas y baja rentabilidad. Revisar condiciones.",
            "color": "danger",
            "css_class": "text-danger",
            "action": "Negociar rappels o introducir alternativas.",
        }
    elif hhi_is_low and margin_is_high:
        return {
            "emoji": "⭐",
            "title": "Generalista Premium",
            "description": "Variedad y buen margen. Control total del surtido.",
            "color": "primary",
            "css_class": "text-primary",
            "action": "Optimizar rotación de referencias menos rentables.",
        }
    elif hhi_is_low and margin_is_low:
        return {
            "emoji": "📊",
            "title": "Generalista de Volumen",
            "description": "Negocio de rotación y cobertura total.",
            "color": "info",
            "css_class": "text-info",
            "action": "Asegurar eficiencia operativa y volumen.",
        }
    else:
        # Zona intermedia: HHI medio O margen neutro
        return {
            "emoji": "⚖️",
            "title": "Surtido Balanceado",
            "description": "En la media del mercado. Sin alarmas.",
            "color": "secondary",
            "css_class": "text-secondary",
            "action": "Monitorizar. Oportunidad de especialización.",
        }


def _get_hhi_recommendation(level: str) -> dict:
    """
    Obtener recomendación accionable según nivel HHI Interno (Issue #539).

    DEPRECATED: Usar calculate_hhi_diagnosis() para diagnóstico completo.
    Se mantiene por compatibilidad con código existente.

    Args:
        level: "low", "medium", "high"

    Returns:
        Dict con icon, text, action para la recomendación
    """
    recommendations = {
        "low": {
            "icon": "fas fa-expand-arrows-alt",
            "text": "Estrategia Generalista",
            "action": "Buena resiliencia. Tienes alternativas si un proveedor falla.",
            "color": "success",
        },
        "medium": {
            "icon": "fas fa-balance-scale",
            "text": "Equilibrio",
            "action": "Balance entre eficiencia y resiliencia. Revisa márgenes.",
            "color": "warning",
        },
        "high": {
            "icon": "fas fa-crosshairs",
            "text": "Estrategia Especialista",
            "action": "¿Tienes mejores márgenes que la competencia? Si no, revisa.",
            "color": "danger",
        },
    }
    return recommendations.get(level, recommendations["low"])


def create_hhi_display(hhi_data: dict, brands_data: list = None) -> html.Div:
    """
    Crear visualización del HHI con diagnóstico cruzado HHI × Margen.

    Incluye:
    - Diagnóstico principal (emoji + título + descripción)
    - Valor HHI con escala visual
    - Acción recomendada

    Args:
        hhi_data: Dict con hhi, hhi_interpretation
        brands_data: Lista de marcas con avg_margin para calcular margen de categoría

    Returns:
        html.Div con visualización completa del HHI y diagnóstico
    """
    if not hhi_data or "hhi" not in hhi_data:
        return html.Div([
            html.P("Selecciona una categoría para ver el análisis", className="text-muted text-center"),
        ])

    hhi = hhi_data.get("hhi", 0)
    interpretation = hhi_data.get("hhi_interpretation", {})
    level = interpretation.get("level", "low")

    # Calcular diagnóstico cruzado si hay datos de margen
    diagnosis = None
    if brands_data:
        # Calcular margen promedio ponderado por ventas
        total_sales = sum(b.get("sales", 0) for b in brands_data)
        if total_sales > 0:
            weighted_margin = sum(
                b.get("avg_margin", 0) * b.get("sales", 0)
                for b in brands_data
            ) / total_sales

            # Fase 1: Usar umbrales fijos (category_avg_margin=None)
            # Fase 2: Obtener baseline real por tipo de categoría
            diagnosis = calculate_hhi_diagnosis(hhi, weighted_margin, category_avg_margin=None)

    # Si no hay diagnóstico, usar el sistema legacy
    if not diagnosis:
        recommendation = _get_hhi_recommendation(level)
        diagnosis = {
            "emoji": "📊" if level == "low" else ("⚠️" if level == "high" else "⚖️"),
            "title": recommendation["text"],
            "description": recommendation["action"],
            "color": recommendation["color"],
            "css_class": f"text-{recommendation['color']}",
            "action": recommendation["action"],
        }

    return html.Div([
        # Diagnóstico principal (prominente)
        html.Div([
            html.Span(diagnosis["emoji"], style={"fontSize": "2rem"}),
            html.Div([
                html.Strong(diagnosis["title"], className=diagnosis["css_class"],
                           style={"fontSize": "1.1rem"}),
            ]),
        ], className="text-center mb-2"),

        # Descripción corta
        html.P(
            diagnosis["description"],
            className="text-center text-muted small mb-3",
        ),

        # Valor HHI
        html.Div([
            html.Span("HHI: ", className="text-muted small"),
            html.Span(
                f"{hhi:,.0f}",
                style={"fontSize": "1.5rem", "fontWeight": "bold"},
                className=diagnosis["css_class"],
            ),
            html.Span(" / 10.000", className="text-muted small"),
        ], className="text-center mb-2"),

        # Escala visual compacta
        _create_hhi_scale_visual(hhi),

        # Acción recomendada
        dbc.Card([
            dbc.CardBody([
                html.Div([
                    html.I(className="fas fa-arrow-right me-2", style={"color": COLORS.get(diagnosis["color"], "#6c757d")}),
                    html.Small(diagnosis["action"], className="text-muted"),
                ]),
            ], className="py-2"),
        ], className="border-0 mt-2", style={"backgroundColor": "#f8f9fa"}),
    ])


def create_brands_chart_card():
    """
    Card con treemap/barras de marcas y toggle.

    Returns:
        dbc.Card con visualización de marcas
    """
    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-chart-pie me-2"),
                "Distribución por Marca",
            ], className="d-flex align-items-center"),
            # Toggle Treemap/Barras
            dbc.ButtonGroup([
                dbc.Button(
                    html.Span([html.I(className="fas fa-th-large me-1"), "Treemap"]),
                    id="ventalibre-brands-treemap-btn",
                    color="primary",
                    outline=False,
                    size="sm",
                    className="px-2",
                ),
                dbc.Button(
                    html.Span([html.I(className="fas fa-bars me-1"), "Barras"]),
                    id="ventalibre-brands-bars-btn",
                    color="primary",
                    outline=True,
                    size="sm",
                    className="px-2",
                ),
            ], size="sm"),
        ], className="d-flex justify-content-between align-items-center", style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-brands-chart-loading",
                type="circle",
                children=dcc.Graph(
                    id="ventalibre-brands-chart",
                    config={"responsive": True, "displayModeBar": False},
                    style={"height": "350px"},
                ),
            ),
        ]),
    ], className="h-100 shadow-sm")


def create_brands_table_card():
    """
    Card con tabla de marcas (cuota, margen, tendencia).

    Returns:
        dbc.Card con tabla de marcas
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-table me-2"),
            "Detalle por Marca",
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-brands-table-loading",
                type="circle",
                children=html.Div(id="ventalibre-brands-table-container"),
            ),
        ]),
    ], className="h-100 shadow-sm")


def create_brands_table(brands_data: list) -> html.Div:
    """
    Crear tabla de marcas con datos reales.

    Args:
        brands_data: Lista de dicts con brand, sales, share, avg_margin, units

    Returns:
        dbc.Table con datos de marcas
    """
    if not brands_data:
        return html.Div([
            html.P("No hay datos de marcas disponibles", className="text-muted text-center py-3"),
        ])

    # Cabecera
    header = html.Thead(html.Tr([
        html.Th("Marca", style={"width": "30%"}),
        html.Th("Ventas", className="text-end"),
        html.Th("Cuota", className="text-end"),
        html.Th("Margen", className="text-end"),
        html.Th("Uds.", className="text-end"),
    ]))

    # Filas
    rows = []
    for brand in brands_data[:15]:  # Limitar a 15 filas
        rows.append(html.Tr([
            html.Td(brand.get("brand", "").title()),
            html.Td(f"{brand.get('sales', 0):,.0f} €", className="text-end"),
            html.Td(f"{brand.get('share', 0):.1f}%", className="text-end"),
            html.Td(f"{brand.get('avg_margin', 0):.1f}%", className="text-end"),
            html.Td(f"{brand.get('units', 0):,}", className="text-end"),
        ]))

    return dbc.Table(
        [header, html.Tbody(rows)],
        striped=True,
        hover=True,
        responsive=True,
        size="sm",
    )


def create_value_quadrant_card():
    """
    Card con scatter plot Margen vs Volumen (Cuadrante de Valor).

    Returns:
        dbc.Card con scatter plot
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-crosshairs me-2", style={"color": COLORS["success"]}),
            "Cuadrante de Valor",
            dbc.Badge("?", id="ventalibre-quadrant-help", color="secondary", className="ms-2"),
            dbc.Tooltip(
                "Estrellas: alto volumen + alto margen. Tráfico: alto volumen + bajo margen. "
                "Oportunidades: bajo volumen + alto margen. Revisar: bajo volumen + bajo margen.",
                target="ventalibre-quadrant-help",
            ),
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-quadrant-loading",
                type="circle",
                children=dcc.Graph(
                    id="ventalibre-value-quadrant-chart",
                    config={"responsive": True, "displayModeBar": False},
                    style={"height": "350px"},
                ),
            ),
        ]),
    ], className="h-100 shadow-sm")


def create_value_quadrant_figure(quadrant_data: dict) -> go.Figure:
    """
    Crear scatter plot del cuadrante de valor.

    Args:
        quadrant_data: Dict con brands y thresholds

    Returns:
        Plotly Figure
    """
    brands = quadrant_data.get("brands", [])
    thresholds = quadrant_data.get("thresholds", {})

    if not brands:
        fig = go.Figure()
        fig.add_annotation(
            text="Selecciona una categoría para ver el cuadrante",
            xref="paper", yref="paper",
            x=0.5, y=0.5, showarrow=False,
            font=dict(size=14, color="gray"),
        )
        fig.update_layout(
            xaxis=dict(visible=False),
            yaxis=dict(visible=False),
            plot_bgcolor="white",
        )
        return fig

    # Preparar datos
    x_vals = [b["sales"] for b in brands]
    y_vals = [b["margin_pct"] for b in brands]
    names = [b["brand"].title() for b in brands]
    colors = [QUADRANT_COLORS.get(b.get("quadrant", "review"), "#6c757d") for b in brands]

    fig = go.Figure()

    # Scatter de marcas
    fig.add_trace(go.Scatter(
        x=x_vals,
        y=y_vals,
        mode="markers+text",
        text=names,
        textposition="top center",
        marker=dict(size=12, color=colors, line=dict(width=1, color="white")),
        hovertemplate="<b>%{text}</b><br>Ventas: %{x:,.0f}€<br>Margen: %{y:.1f}%<extra></extra>",
    ))

    # Líneas de threshold (medianas)
    median_sales = thresholds.get("median_sales", 0)
    median_margin = thresholds.get("median_margin", 0)

    fig.add_hline(y=median_margin, line_dash="dash", line_color="gray", opacity=0.5)
    fig.add_vline(x=median_sales, line_dash="dash", line_color="gray", opacity=0.5)

    # Layout
    fig.update_layout(
        xaxis_title="Volumen de Ventas (€)",
        yaxis_title="Margen (%)",
        showlegend=False,
        margin=dict(t=30, b=50, l=60, r=20),
        plot_bgcolor="white",
        xaxis=dict(gridcolor="#eee", zeroline=False),
        yaxis=dict(gridcolor="#eee", zeroline=False),
    )

    return fig


def create_market_evolution_card():
    """
    Card con gráfico de áreas apiladas 100% (evolución cuota).

    Returns:
        dbc.Card con área chart
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-chart-area me-2", style={"color": COLORS["info"]}),
            "Evolución de Cuota de Mercado",
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-evolution-loading",
                type="circle",
                children=dcc.Graph(
                    id="ventalibre-market-evolution-chart",
                    config={"responsive": True, "displayModeBar": False},
                    style={"height": "350px"},
                ),
            ),
        ]),
    ], className="h-100 shadow-sm")


def create_market_evolution_figure(evolution_data: dict) -> go.Figure:
    """
    Crear gráfico de áreas apiladas 100%.

    Args:
        evolution_data: Dict con time_series y top_brands

    Returns:
        Plotly Figure
    """
    time_series = evolution_data.get("time_series", [])
    top_brands = evolution_data.get("top_brands", [])

    if not time_series or not top_brands:
        fig = go.Figure()
        fig.add_annotation(
            text="Selecciona una categoría para ver la evolución",
            xref="paper", yref="paper",
            x=0.5, y=0.5, showarrow=False,
            font=dict(size=14, color="gray"),
        )
        fig.update_layout(
            xaxis=dict(visible=False),
            yaxis=dict(visible=False),
            plot_bgcolor="white",
        )
        return fig

    months = [ts["month"] for ts in time_series]

    fig = go.Figure()

    # Colores para marcas
    colors = px.colors.qualitative.Set2
    all_brands = top_brands + ["Otras"]

    for i, brand in enumerate(all_brands):
        values = [ts.get(brand, 0) for ts in time_series]
        fig.add_trace(go.Scatter(
            x=months,
            y=values,
            name=brand.title(),
            mode="lines",
            stackgroup="one",
            groupnorm="percent",
            line=dict(width=0.5),
            fillcolor=colors[i % len(colors)],
            hovertemplate=f"<b>{brand.title()}</b><br>%{{x}}: %{{y:.1f}}%<extra></extra>",
        ))

    fig.update_layout(
        yaxis=dict(
            title="Cuota de Mercado (%)",
            range=[0, 100],
            ticksuffix="%",
            gridcolor="#eee",
        ),
        xaxis=dict(title="Mes", gridcolor="#eee"),
        legend=dict(orientation="h", yanchor="bottom", y=1.02, xanchor="right", x=1),
        margin=dict(t=50, b=50, l=60, r=20),
        plot_bgcolor="white",
        hovermode="x unified",
    )

    return fig


def create_price_boxplot_card():
    """
    Card con boxplot de precios por marca.

    Returns:
        dbc.Card con boxplot
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-box me-2", style={"color": COLORS["warning"]}),
            "Distribución de Precios por Marca",
            dbc.Badge("?", id="ventalibre-boxplot-help", color="secondary", className="ms-2"),
            dbc.Tooltip(
                "Solapamiento de cajas indica competencia directa (canibalización). "
                "Cajas muy estiradas indican catálogo de precios inconsistente.",
                target="ventalibre-boxplot-help",
            ),
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            dcc.Loading(
                id="ventalibre-boxplot-loading",
                type="circle",
                children=dcc.Graph(
                    id="ventalibre-price-boxplot-chart",
                    config={"responsive": True, "displayModeBar": False},
                    # Altura dinámica controlada por create_price_boxplot_figure()
                ),
            ),
        ]),
    ], className="shadow-sm")


def create_price_boxplot_figure(price_data: dict) -> go.Figure:
    """
    Crear boxplot horizontal de precios por marca.

    Args:
        price_data: Dict con brands (cada uno con min, q1, median, q3, max)

    Returns:
        Plotly Figure con altura dinámica según número de marcas
    """
    brands = price_data.get("brands", [])

    if not brands:
        fig = go.Figure()
        fig.add_annotation(
            text="Selecciona una categoría para ver la distribución de precios",
            xref="paper", yref="paper",
            x=0.5, y=0.5, showarrow=False,
            font=dict(size=14, color="gray"),
        )
        fig.update_layout(
            xaxis=dict(visible=False),
            yaxis=dict(visible=False),
            plot_bgcolor="white",
        )
        return fig

    fig = go.Figure()

    # Colores distintos para cada marca
    brand_colors = px.colors.qualitative.Set2

    for i, brand_data in enumerate(brands):
        color = brand_colors[i % len(brand_colors)]
        fig.add_trace(go.Box(
            name=brand_data["brand"].title(),
            lowerfence=[brand_data["min"]],
            q1=[brand_data["q1"]],
            median=[brand_data["median"]],
            q3=[brand_data["q3"]],
            upperfence=[brand_data["max"]],
            orientation="h",
            marker=dict(color=color),
            line=dict(color=color, width=1.5),
            fillcolor=color,
            opacity=0.7,
            hovertemplate="<b>%{y}</b><br>Mediana: %{median:.2f}€<br>Rango: %{lowerfence:.2f}€ - %{upperfence:.2f}€<extra></extra>",
            boxpoints=False,  # No mostrar puntos individuales
        ))

    # Altura dinámica: mínimo 200px, 40px por marca
    num_brands = len(brands)
    dynamic_height = max(200, min(500, 40 + num_brands * 35))

    fig.update_layout(
        xaxis_title="Precio (€)",
        yaxis=dict(
            autorange="reversed",  # Orden descendente
            tickfont=dict(size=11),
        ),
        showlegend=False,
        margin=dict(t=20, b=50, l=120, r=20),  # Más espacio para nombres
        plot_bgcolor="white",
        xaxis=dict(gridcolor="#eee", zeroline=False),
        height=dynamic_height,
        boxgap=0.3,      # Espacio entre grupos de cajas
        boxgroupgap=0.1,  # Espacio dentro de grupos
    )

    return fig


def create_brand_duel_section():
    """
    Sección colapsible de Brand Duel Mode.

    Returns:
        html.Div con collapse de Brand Duel
    """
    return html.Div([
        dbc.Button(
            html.Span([
                html.I(className="fas fa-balance-scale me-2"),
                "Comparar Marcas (Brand Duel)",
            ]),
            id="ventalibre-brand-duel-toggle-btn",
            color="outline-primary",
            className="mb-3 w-100",
        ),
        dbc.Collapse(
            id="ventalibre-brand-duel-collapse",
            is_open=False,
            children=dbc.Card([
                dbc.CardBody([
                    dbc.Row([
                        dbc.Col([
                            dbc.Label("Marca A"),
                            dcc.Dropdown(
                                id="ventalibre-brand-duel-dropdown-a",
                                placeholder="Selecciona primera marca...",
                            ),
                        ], width=12, lg=5),
                        dbc.Col([
                            html.Div([
                                html.I(className="fas fa-arrows-alt-h fa-2x text-muted"),
                            ], className="text-center pt-4 d-none d-lg-block"),
                        ], width=12, lg=2),
                        dbc.Col([
                            dbc.Label("Marca B"),
                            dcc.Dropdown(
                                id="ventalibre-brand-duel-dropdown-b",
                                placeholder="Selecciona segunda marca...",
                            ),
                        ], width=12, lg=5),
                    ], className="mb-4"),
                    dcc.Loading(
                        id="ventalibre-duel-loading",
                        type="circle",
                        children=html.Div(id="ventalibre-brand-duel-content"),
                    ),
                ]),
            ], className="shadow-sm"),
        ),
    ])


def create_brand_duel_display(duel_data: dict) -> html.Div:
    """
    Crear visualización del duelo de marcas.

    Args:
        duel_data: Dict con brand_a, brand_b, winners

    Returns:
        html.Div con comparación lado a lado
    """
    if not duel_data or "brand_a" not in duel_data or "brand_b" not in duel_data:
        return html.Div([
            html.P("Selecciona dos marcas para compararlas", className="text-muted text-center py-3"),
        ])

    brand_a = duel_data.get("brand_a", {})
    brand_b = duel_data.get("brand_b", {})
    winners = duel_data.get("winners", {})

    metrics = [
        ("Cuota de Mercado", "share", "%", True),
        ("Margen Promedio", "margin_pct", "%", True),
        ("Ticket Promedio", "avg_ticket", "€", True),
        ("Unidades Vendidas", "units", "", True),
        ("Margen Total", "total_margin", "€", True),
    ]

    rows = []
    for label, key, suffix, higher_better in metrics:
        val_a = brand_a.get(key, 0)
        val_b = brand_b.get(key, 0)
        winner = winners.get(key, "tie")

        # Formatear valores
        if suffix == "€":
            val_a_str = f"{val_a:,.0f} €"
            val_b_str = f"{val_b:,.0f} €"
        elif suffix == "%":
            val_a_str = f"{val_a:.1f}%"
            val_b_str = f"{val_b:.1f}%"
        else:
            val_a_str = f"{val_a:,}"
            val_b_str = f"{val_b:,}"

        # Estilos de ganador
        style_a = {"color": COLORS.get("success"), "fontWeight": "bold"} if winner == "a" else {}
        style_b = {"color": COLORS.get("success"), "fontWeight": "bold"} if winner == "b" else {}
        trophy_a = " 🏆" if winner == "a" else ""
        trophy_b = " 🏆" if winner == "b" else ""

        rows.append(html.Tr([
            html.Td(html.Span([val_a_str, trophy_a], style=style_a), className="text-end"),
            html.Td(html.Strong(label), className="text-center"),
            html.Td(html.Span([val_b_str, trophy_b], style=style_b)),
        ]))

    return dbc.Row([
        dbc.Col([
            html.H5(brand_a.get("brand", "").title(), className="text-center text-primary mb-3"),
        ], width=12, lg=5),
        dbc.Col([], width=12, lg=2),
        dbc.Col([
            html.H5(brand_b.get("brand", "").title(), className="text-center text-primary mb-3"),
        ], width=12, lg=5),
        dbc.Col([
            dbc.Table([
                html.Tbody(rows),
            ], bordered=True, hover=True, responsive=True, className="text-center"),
        ], width=12),
    ])


# =============================================================================
# Issue #539: HHI Matrix - Scatter Plot HHI × Margen por Categoría
# =============================================================================


# Colores para los 5 diagnósticos HHI × Margen por CATEGORÍA
# Nota: Diferente de QUADRANT_COLORS que aplica a MARCAS (margen × volumen)
# - QUADRANT_COLORS: brand performance dentro de una categoría
# - DIAGNOSIS_COLORS: estrategia de la farmacia por categoría NECESIDAD
DIAGNOSIS_COLORS = {
    "specialist_success": "#28a745",    # Verde - Especialista Exitoso (HHI alto + Margen alto)
    "dependency_risk": "#dc3545",       # Rojo - Riesgo de Dependencia (HHI alto + Margen bajo)
    "generalist_premium": "#007bff",    # Azul - Generalista Premium (HHI bajo + Margen alto)
    "generalist_volume": "#17a2b8",     # Cyan - Generalista de Volumen (HHI bajo + Margen bajo)
    "balanced": "#6c757d",              # Gris - Surtido Balanceado (centro)
}


def create_hhi_matrix_card():
    """
    Card con scatter plot HHI × Margen por categoría (Issue #539).

    Returns:
        dbc.Card con scatter plot y tabla de auditoría
    """
    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-bullseye me-2", style={"color": COLORS["primary"]}),
                "Mapa Estratégico: HHI × Margen",
                dbc.Badge(
                    "?",
                    id="ventalibre-hhi-matrix-help",
                    color="secondary",
                    className="ms-2",
                    style={"cursor": "pointer"},
                ),
                dbc.Popover([
                    html.P([
                        "Cada punto es una ",
                        html.Strong("categoría NECESIDAD"),
                        " de tu farmacia.",
                    ], className="mb-2"),
                    html.Ul([
                        html.Li([
                            html.Span("🎯", className="me-1"),
                            html.Strong("Especialista Exitoso: "),
                            "HHI alto + Margen alto",
                        ], className="small"),
                        html.Li([
                            html.Span("⚠️", className="me-1"),
                            html.Strong("Riesgo Dependencia: "),
                            "HHI alto + Margen bajo",
                        ], className="small"),
                        html.Li([
                            html.Span("⭐", className="me-1"),
                            html.Strong("Generalista Premium: "),
                            "HHI bajo + Margen alto",
                        ], className="small"),
                        html.Li([
                            html.Span("📊", className="me-1"),
                            html.Strong("Generalista Volumen: "),
                            "HHI bajo + Margen bajo",
                        ], className="small"),
                    ], className="mb-0 ps-3"),
                ],
                    target="ventalibre-hhi-matrix-help",
                    trigger="hover",
                    placement="bottom",
                ),
            ], className="d-flex align-items-center"),
        ], style={
            "backgroundColor": COLORS.get("bg_tertiary", "#f8f9fa"),
            "fontWeight": "600",
        }),
        dbc.CardBody([
            # Scatter plot
            dcc.Loading(
                id="ventalibre-hhi-matrix-loading",
                type="circle",
                children=dcc.Graph(
                    id="ventalibre-hhi-matrix-chart",
                    config={"responsive": True, "displayModeBar": False},
                    style={"height": "400px"},
                ),
            ),
            # Separador
            html.Hr(className="my-3"),
            # Tabla de auditoría
            html.H6([
                html.I(className="fas fa-clipboard-list me-2"),
                "Auditoría por Categoría",
            ], className="mb-3"),
            dcc.Loading(
                id="ventalibre-hhi-table-loading",
                type="circle",
                children=html.Div(id="ventalibre-hhi-matrix-table"),
            ),
        ]),
    ], className="shadow-sm mb-4")


def create_hhi_matrix_figure(matrix_data: dict) -> go.Figure:
    """
    Crear scatter plot HHI × Margen por categoría.

    Args:
        matrix_data: Dict con categories, thresholds, summary

    Returns:
        Plotly Figure con scatter plot y zonas coloreadas
    """
    categories = matrix_data.get("categories", [])
    thresholds = matrix_data.get("thresholds", {})

    if not categories:
        fig = go.Figure()
        fig.add_annotation(
            text="No hay datos suficientes para el mapa estratégico",
            xref="paper", yref="paper",
            x=0.5, y=0.5, showarrow=False,
            font=dict(size=14, color="gray"),
        )
        fig.update_layout(
            xaxis=dict(visible=False),
            yaxis=dict(visible=False),
            plot_bgcolor="white",
        )
        return fig

    # Extraer umbrales
    hhi_low = thresholds.get("hhi_low", 1500)
    hhi_high = thresholds.get("hhi_high", 2500)
    margin_low = thresholds.get("margin_low", 15)
    margin_high = thresholds.get("margin_high", 25)

    # Preparar datos
    x_vals = [c["hhi"] for c in categories]
    y_vals = [c["avg_margin"] for c in categories]
    sizes = [max(10, min(50, c["total_sales"] / 1000)) for c in categories]  # Escalar tamaño
    colors = [DIAGNOSIS_COLORS.get(c["diagnosis"]["quadrant"], "#6c757d") for c in categories]
    names = [c["category"].replace("_", " ").title() for c in categories]

    # Hover con Pareto Top 3
    hover_texts = []
    for c in categories:
        top_brands_text = "<br>".join([
            f"  • {b['brand'].title()}: {b['share']:.1f}%"
            for b in c.get("top_brands", [])[:3]
        ])
        hover_texts.append(
            f"<b>{c['category'].replace('_', ' ').title()}</b><br>"
            f"HHI: {c['hhi']:,.0f}<br>"
            f"Margen: {c['avg_margin']:.1f}%<br>"
            f"Ventas: €{c['total_sales']:,.0f}<br>"
            f"<b>Top 3 Marcas:</b><br>{top_brands_text}<br>"
            f"<b>{c['diagnosis']['emoji']} {c['diagnosis']['title']}</b>"
        )

    fig = go.Figure()

    # Añadir zonas de fondo para los cuadrantes
    # Nota: Usamos shapes en lugar de scatter para las zonas
    max_hhi = max(x_vals) * 1.1 if x_vals else 5000
    max_margin = max(y_vals) * 1.1 if y_vals else 40

    # Zona 1: Generalista Premium (izquierda arriba) - Azul claro
    fig.add_shape(
        type="rect",
        x0=0, x1=hhi_low,
        y0=margin_high, y1=max_margin,
        fillcolor="rgba(0, 123, 255, 0.1)",
        line=dict(width=0),
        layer="below",
    )
    # Zona 2: Especialista Exitoso (derecha arriba) - Verde claro
    fig.add_shape(
        type="rect",
        x0=hhi_high, x1=max_hhi,
        y0=margin_high, y1=max_margin,
        fillcolor="rgba(40, 167, 69, 0.1)",
        line=dict(width=0),
        layer="below",
    )
    # Zona 3: Generalista Volumen (izquierda abajo) - Cyan claro
    fig.add_shape(
        type="rect",
        x0=0, x1=hhi_low,
        y0=0, y1=margin_low,
        fillcolor="rgba(23, 162, 184, 0.1)",
        line=dict(width=0),
        layer="below",
    )
    # Zona 4: Riesgo Dependencia (derecha abajo) - Rojo claro
    fig.add_shape(
        type="rect",
        x0=hhi_high, x1=max_hhi,
        y0=0, y1=margin_low,
        fillcolor="rgba(220, 53, 69, 0.1)",
        line=dict(width=0),
        layer="below",
    )

    # Líneas de umbral
    fig.add_hline(y=margin_low, line_dash="dash", line_color="#ffc107", opacity=0.7,
                  annotation_text=f"Margen bajo ({margin_low}%)", annotation_position="right")
    fig.add_hline(y=margin_high, line_dash="dash", line_color="#28a745", opacity=0.7,
                  annotation_text=f"Margen alto ({margin_high}%)", annotation_position="right")
    fig.add_vline(x=hhi_low, line_dash="dash", line_color="#17a2b8", opacity=0.7,
                  annotation_text="Diversificado", annotation_position="top")
    fig.add_vline(x=hhi_high, line_dash="dash", line_color="#dc3545", opacity=0.7,
                  annotation_text="Concentrado", annotation_position="top")

    # Scatter de categorías
    fig.add_trace(go.Scatter(
        x=x_vals,
        y=y_vals,
        mode="markers+text",
        text=names,
        textposition="top center",
        textfont=dict(size=9),
        marker=dict(
            size=sizes,
            color=colors,
            line=dict(width=1, color="white"),
            opacity=0.85,
        ),
        hovertemplate="%{customdata}<extra></extra>",
        customdata=hover_texts,
    ))

    # Layout
    fig.update_layout(
        xaxis_title="HHI (Concentración de Proveedores)",
        yaxis_title="Margen Promedio (%)",
        showlegend=False,
        margin=dict(t=40, b=60, l=60, r=40),
        plot_bgcolor="white",
        xaxis=dict(
            gridcolor="#eee",
            zeroline=False,
            range=[0, max_hhi],
        ),
        yaxis=dict(
            gridcolor="#eee",
            zeroline=False,
            range=[0, max_margin],
        ),
        hovermode="closest",
    )

    return fig


def create_hhi_matrix_table(matrix_data: dict) -> html.Div:
    """
    Crear tabla de auditoría para el HHI Matrix.

    Args:
        matrix_data: Dict con categories, thresholds, summary

    Returns:
        html.Div con tabla de auditoría
    """
    categories = matrix_data.get("categories", [])

    if not categories:
        return html.Div([
            html.P("No hay datos de categorías", className="text-muted text-center py-3"),
        ])

    # Cabecera
    header = html.Thead(html.Tr([
        html.Th("Diagnóstico", style={"width": "10%"}),
        html.Th("Categoría", style={"width": "25%"}),
        html.Th("HHI", className="text-end", style={"width": "12%"}),
        html.Th("Margen", className="text-end", style={"width": "12%"}),
        html.Th("Ventas", className="text-end", style={"width": "15%"}),
        html.Th("Top 3 Marcas", style={"width": "26%"}),
    ]))

    # Filas (limitar a 20 para no saturar)
    rows = []
    for c in categories[:20]:
        diagnosis = c.get("diagnosis", {})
        top_brands = c.get("top_brands", [])

        # Formatear top brands
        brands_text = ", ".join([
            f"{b['brand'].title()} ({b['share']:.0f}%)"
            for b in top_brands[:3]
        ]) if top_brands else "-"

        # Color de fila según diagnóstico
        row_style = {}
        if diagnosis.get("quadrant") == "dependency_risk":
            row_style = {"backgroundColor": "rgba(220, 53, 69, 0.05)"}
        elif diagnosis.get("quadrant") == "specialist_success":
            row_style = {"backgroundColor": "rgba(40, 167, 69, 0.05)"}

        rows.append(html.Tr([
            html.Td([
                html.Span(diagnosis.get("emoji", "⚖️"), style={"fontSize": "1.2rem"}),
            ], className="text-center"),
            html.Td(c["category"].replace("_", " ").title()),
            html.Td(f"{c['hhi']:,.0f}", className="text-end"),
            html.Td(f"{c['avg_margin']:.1f}%", className="text-end"),
            html.Td(f"€{c['total_sales']:,.0f}", className="text-end"),
            html.Td(brands_text, className="small"),
        ], style=row_style))

    # Summary row
    summary = matrix_data.get("summary", {})
    if summary:
        rows.append(html.Tr([
            html.Td(""),
            html.Td(html.Strong(f"TOTAL ({summary.get('total_categories', 0)} categorías)")),
            html.Td(html.Strong(f"{summary.get('avg_hhi', 0):,.0f}"), className="text-end"),
            html.Td(html.Strong(f"{summary.get('avg_margin', 0):.1f}%"), className="text-end"),
            html.Td(html.Strong(f"€{summary.get('total_sales', 0):,.0f}"), className="text-end"),
            html.Td(""),
        ], style={"backgroundColor": "#f8f9fa"}))

    return dbc.Table(
        [header, html.Tbody(rows)],
        striped=True,
        hover=True,
        responsive=True,
        size="sm",
        className="mb-0",
    )
