# frontend/components/ventalibre/alternatives_table.py
"""
Tabla de productos alternativos para Tab 3 (Issue #494).

Muestra productos de la misma categoría NECESIDAD pero diferente marca.
Útil para switching analysis y optimización de surtido.
"""

from typing import List, Optional

from dash import html
import dash_bootstrap_components as dbc


def create_alternatives_table(
    alternatives: Optional[List[dict]] = None,
    category: Optional[str] = None,
    base_margin: Optional[float] = None,
) -> html.Div:
    """
    Crea la tabla de productos alternativos.

    Args:
        alternatives: Lista de alternativas con codigo_nacional, product_name, etc.
        category: Categoría NECESIDAD común
        base_margin: Margen del producto base para comparación

    Returns:
        html.Div con la tabla de alternativas
    """
    if not alternatives:
        return create_alternatives_placeholder()

    return dbc.Card([
        dbc.CardHeader([
            html.Div([
                html.I(className="fas fa-exchange-alt me-2"),
                html.Span("Alternativas (Switching)", className="fw-bold"),
                dbc.Badge(
                    f"{len(alternatives)} productos",
                    color="info",
                    className="ms-2",
                ),
            ], className="d-flex align-items-center"),
            html.Small(
                f"Misma necesidad: {category.replace('_', ' ').title() if category else 'N/A'}",
                className="text-muted d-block mt-1",
            ),
        ]),
        dbc.CardBody([
            dbc.Table([
                html.Thead([
                    html.Tr([
                        html.Th("Producto", style={"width": "40%"}),
                        html.Th("Marca", style={"width": "15%"}),
                        html.Th("Ventas", className="text-end", style={"width": "15%"}),
                        html.Th("Margen", className="text-end", style={"width": "15%"}),
                        html.Th("vs Base", className="text-end", style={"width": "15%"}),
                    ]),
                ]),
                html.Tbody([
                    _create_alternative_row(alt, base_margin)
                    for alt in alternatives[:10]  # Limitar a 10
                ]),
            ], hover=True, striped=True, size="sm", className="mb-0"),
        ]),
    ], className="shadow-sm")


def _create_alternative_row(alt: dict, base_margin: Optional[float]) -> html.Tr:
    """Crea una fila de la tabla de alternativas."""
    cn = alt.get("codigo_nacional", "")
    name = alt.get("product_name", "")
    brand = alt.get("detected_brand") or "—"
    sales = alt.get("total_sales", 0)
    margin = alt.get("margin_percent", 0)
    margin_diff = alt.get("margin_diff_vs_selected", 0)

    # Color según diferencia de margen
    if margin_diff > 3:
        diff_color = "success"
        diff_icon = "↑"
    elif margin_diff < -3:
        diff_color = "danger"
        diff_icon = "↓"
    else:
        diff_color = "secondary"
        diff_icon = "→"

    return html.Tr([
        html.Td([
            html.Div(name[:35] + "..." if len(name) > 35 else name, className="text-truncate"),
            html.Small(cn, className="text-muted"),
        ]),
        html.Td(
            dbc.Badge(brand.upper()[:10], color="light", text_color="dark"),
        ),
        html.Td(f"€{sales:,.0f}", className="text-end"),
        html.Td(f"{margin:.1f}%", className="text-end"),
        html.Td([
            html.Span(
                f"{diff_icon} {margin_diff:+.1f} pp",
                className=f"text-{diff_color} fw-bold",
            ),
        ], className="text-end"),
    ])


def create_alternatives_placeholder() -> html.Div:
    """Placeholder cuando no hay alternativas o no hay producto seleccionado."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-exchange-alt me-2"),
            html.Span("Alternativas (Switching)", className="fw-bold"),
        ]),
        dbc.CardBody([
            html.Div([
                html.I(className="fas fa-info-circle fa-2x text-muted mb-2"),
                html.P(
                    "Selecciona un producto para ver alternativas",
                    className="text-muted mb-0 small",
                ),
            ], className="text-center py-3"),
        ]),
    ], className="shadow-sm bg-light")


def create_alternatives_empty(category: str) -> html.Div:
    """Mensaje cuando no hay alternativas disponibles."""
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-exchange-alt me-2"),
            html.Span("Alternativas (Switching)", className="fw-bold"),
        ]),
        dbc.CardBody([
            dbc.Alert([
                html.I(className="fas fa-check-circle me-2"),
                f"No hay alternativas en la categoría '{category.replace('_', ' ').title()}'.",
            ], color="info", className="mb-0"),
        ]),
    ], className="shadow-sm")
