# frontend/components/upload_progress.py
"""
Componentes UI para visualización de progreso de upload.
Extraídos de callbacks/upload.py para mejorar modularidad y reducir tamaño de archivo.

Componentes incluidos:
- Progress cards (upload, enrichment, error)
- History components
- Summary components
- Helper functions para formateo y detección de fases
"""

import logging
from datetime import datetime

import dash_bootstrap_components as dbc
from dash import html

from components.base import BaseCard, Title
from components.empty_state import empty_upload_history_state
from components.toast_manager import success_toast
from utils.api_client import backend_client
from utils.helpers import format_currency, format_number

logger = logging.getLogger(__name__)


# ========================================
# DELETE PREVIEW
# ========================================


def create_delete_preview(data: dict) -> html.Div:
    """
    Crear preview de lo que se eliminará.

    Args:
        data: Datos del preview del backend

    Returns:
        Componente html.Div con el resumen
    """
    filename = data.get("filename", "Archivo desconocido")
    sales_count = data.get("sales_count", 0)
    date_range = data.get("date_range", {})
    total_amount = data.get("total_amount", 0)

    date_from = date_range.get("from", "-")
    date_to = date_range.get("to", "-")

    return html.Div(
        [
            # Nombre del archivo
            html.Div(
                [
                    html.I(className="fas fa-file me-2 text-primary"),
                    html.Strong(filename),
                ],
                className="mb-3",
            ),
            # Estadísticas
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-shopping-cart me-2 text-info"),
                                    html.Strong(format_number(sales_count)),
                                    html.Span(" ventas", className="text-muted ms-1"),
                                ]
                            ),
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-euro-sign me-2 text-success"),
                                    html.Strong(format_currency(total_amount)),
                                ]
                            ),
                        ],
                        width=6,
                    ),
                ],
                className="mb-2",
            ),
            # Rango de fechas
            html.Div(
                [
                    html.I(className="fas fa-calendar-alt me-2 text-secondary"),
                    html.Span(f"{date_from} - {date_to}", className="text-muted"),
                ],
                className="small",
            ),
        ],
        className="p-3 bg-light rounded",
    )


# ========================================
# PROGRESS CARD PRINCIPAL
# ========================================


def create_progress_card(
    filename,
    status,
    upload_id,
    rows_total=None,
    rows_processed=None,
    rows_with_errors=None,
    error_message=None,
    processing_completed_at=None,
    processing_notes=None,
    percentage=None,
    warnings=None,  # Issue #412: Avisos no-bloqueantes
):
    """
    Crear tarjeta de progreso de upload con estado visual mejorado.
    Muestra información detallada sobre las fases del procesamiento.

    ACTUALIZADO (Issue #274): Usa percentage del backend para progreso preciso.
    """

    # Detectar fase actual basándose en las notas de procesamiento
    phase_info = _detect_processing_phase(status, processing_notes, rows_total, rows_processed)

    # Issue #274: Usar percentage del backend si está disponible
    if percentage is not None:
        phase_info["progress"] = percentage

    # Crear componentes de progreso múltiple
    progress_components = _create_multi_phase_progress(phase_info, status, rows_total, rows_processed)

    # Crear contenido de la tarjeta con información detallada
    card_content = [
        # Header mejorado con fase actual
        dbc.CardHeader(
            [
                html.Div(
                    [
                        html.Div(
                            [html.I(className=f"{phase_info['icon']} me-2"), f"Procesando: {filename}"],
                            className="d-flex align-items-center",
                        ),
                        html.Span(phase_info["phase_name"], className=f"badge bg-{phase_info['color']} ms-auto"),
                    ],
                    className="d-flex justify-content-between align-items-center",
                )
            ]
        ),
        dbc.CardBody(
            [
                # Progreso múltiple con fases
                progress_components,
                # Información detallada por secciones
                _create_detailed_info_section(
                    upload_id,
                    rows_total,
                    rows_processed,
                    rows_with_errors,
                    processing_notes,
                    processing_completed_at,
                    error_message,
                    phase_info,
                    warnings,  # Issue #412
                ),
            ]
        ),
    ]

    return dbc.Card(card_content, color=phase_info["color"], outline=True, className="shadow-sm")


def _detect_processing_phase(status, processing_notes, rows_total, rows_processed):
    """
    Detecta la fase actual del procesamiento basándose en el estado y notas.

    ACTUALIZADO (Issue #329): Soporte para nuevos estados PARSING, VALIDATING, SAVING
    """
    if status == "error":
        return {
            "phase_name": "Error",
            "icon": "fas fa-exclamation-triangle",
            "color": "danger",
            "progress": 0,
            "description": "Se ha producido un error durante el procesamiento",
        }

    if status == "completed":
        return {
            "phase_name": "Completado",
            "icon": "fas fa-check-circle",
            "color": "success",
            "progress": 100,
            "description": "Procesamiento completado exitosamente",
        }

    # ISSUE #329: Nuevos estados granulares
    if status == "parsing":
        return {
            "phase_name": "Parseando Archivo",
            "icon": "fas fa-file-csv",
            "color": "info",
            "progress": 10,
            "description": "Leyendo y parseando archivo ERP (Farmatic, Farmanager, etc.)",
        }

    if status == "validating":
        return {
            "phase_name": "Validando Datos",
            "icon": "fas fa-check-double",
            "color": "info",
            "progress": 30,
            "description": "Validando estructura y detectando duplicados",
        }

    if status == "saving":
        return {
            "phase_name": "Guardando en BD",
            "icon": "fas fa-database",
            "color": "primary",
            "progress": 45,
            "description": "Guardando datos de ventas en la base de datos",
        }

    # Analizar notas para detectar subfases específicas (dentro de PROCESSING)
    if processing_notes:
        notes_lower = processing_notes.lower()

        if "descargando nomenclator" in notes_lower or "nomenclator oficial" in notes_lower:
            return {
                "phase_name": "Descargando Nomenclator",
                "icon": "fas fa-download",
                "color": "info",
                "progress": 15,
                "description": "Descargando base de datos oficial de medicamentos del Ministerio de Sanidad",
            }

        elif "nomenclator cargado" in notes_lower or "procesando archivo" in notes_lower:
            return {
                "phase_name": "Procesando CSV",
                "icon": "fas fa-file-csv",
                "color": "primary",
                "progress": 30,
                "description": "Leyendo y validando datos del archivo CSV",
            }

        elif "guardando" in notes_lower:
            return {
                "phase_name": "Guardando Datos",
                "icon": "fas fa-database",
                "color": "primary",
                "progress": 50,
                "description": "Guardando datos de ventas en la base de datos",
            }

        elif "enriquecimiento" in notes_lower or "enriqueciendo" in notes_lower or "cima" in notes_lower:
            return {
                "phase_name": "Enriquecimiento",
                "icon": "fas fa-magic",
                "color": "warning",
                "progress": 75,
                "description": "Enriqueciendo productos con datos oficiales (CIMA, Nomenclator)",
            }

    # Determinar fase basándose en progreso numérico
    if rows_total and rows_processed:
        progress_percent = (rows_processed / rows_total) * 100
        if progress_percent < 25:
            return {
                "phase_name": "Iniciando Procesamiento",
                "icon": "fas fa-play-circle",
                "color": "primary",
                "progress": progress_percent,
                "description": "Iniciando procesamiento del archivo",
            }
        elif progress_percent < 75:
            return {
                "phase_name": "Procesando Ventas",
                "icon": "fas fa-cogs",
                "color": "primary",
                "progress": progress_percent,
                "description": f"Procesando registros de ventas ({format_number(rows_processed)} de {format_number(rows_total)})",
            }
        else:
            return {
                "phase_name": "Finalizando",
                "icon": "fas fa-check-circle",
                "color": "success",
                "progress": progress_percent,
                "description": "Aplicando enriquecimientos finales",
            }

    # Estado por defecto para 'processing'
    return {
        "phase_name": "Procesando...",
        "icon": "fas fa-spinner fa-spin",
        "color": "primary",
        "progress": 25,
        "description": "Procesamiento en curso",
    }


def _create_multi_phase_progress(phase_info, status, rows_total, rows_processed):
    """
    Crear componente de progreso multi-fase visual.
    """
    # Definir las fases del proceso
    phases = [
        {"name": "Carga", "icon": "fas fa-upload", "weight": 20},
        {"name": "Validación", "icon": "fas fa-check", "weight": 30},
        {"name": "Procesamiento", "icon": "fas fa-cogs", "weight": 30},
        {"name": "Enriquecimiento", "icon": "fas fa-magic", "weight": 20},
    ]

    # Calcular progreso general
    if status == "completed":
        overall_progress = 100
    elif status in ["error", "pending"]:
        overall_progress = phase_info["progress"]
    elif rows_total and rows_processed:
        # Progreso basado en datos reales, pero ajustado a las fases
        data_progress = (rows_processed / rows_total) * 60  # Los datos representan 60% del proceso total
        phase_progress = phase_info["progress"]
        overall_progress = min(95, max(data_progress, phase_progress))  # Max 95% hasta completar
    else:
        overall_progress = phase_info["progress"]

    return html.Div(
        [
            # Barra de progreso principal
            html.Div(
                [
                    html.H6(
                        [html.I(className=f"{phase_info['icon']} me-2"), phase_info["phase_name"]], className="mb-2"
                    ),
                    dbc.Progress(
                        value=overall_progress,
                        striped=status == "processing",
                        animated=status == "processing",
                        color=phase_info["color"],
                        className="mb-2",
                        style={"height": "12px"},
                    ),
                    html.Small(phase_info["description"], className="text-muted d-block mb-3"),
                ]
            ),
            # Indicadores de fase (solo si está procesando)
            *(
                [
                    html.Div(
                        [
                            html.H6("Fases del Proceso:", className="mb-2 small"),
                            html.Div(
                                [
                                    _create_phase_indicator(phase, i, overall_progress, len(phases))
                                    for i, phase in enumerate(phases)
                                ],
                                className="d-flex justify-content-between mb-2",
                            ),
                        ]
                    )
                ]
                if status == "processing"
                else []
            ),
        ]
    )


def _create_phase_indicator(phase, index, overall_progress, total_phases):
    """
    Crear indicador individual de fase.
    """
    # Calcular si esta fase está activa, completada o pendiente
    phase_threshold = ((index + 1) / total_phases) * 100

    if overall_progress >= phase_threshold:
        icon_class = f"{phase['icon']} text-success"
        text_class = "text-success small fw-bold"
    elif overall_progress > (index / total_phases) * 100:
        icon_class = (
            f"{phase['icon']} text-primary fa-spin"
            if "spinner" not in phase["icon"]
            else "fas fa-spinner fa-spin text-primary"
        )
        text_class = "text-primary small fw-bold"
    else:
        icon_class = f"{phase['icon']} text-muted"
        text_class = "text-muted small"

    return html.Div(
        [html.I(className=icon_class, style={"font-size": "1.2em"}), html.Div(phase["name"], className=text_class)],
        className="text-center flex-fill",
    )


def _create_detailed_info_section(
    upload_id,
    rows_total,
    rows_processed,
    rows_with_errors,
    processing_notes,
    processing_completed_at,
    error_message,
    phase_info,
    warnings=None,  # Issue #412: Avisos no-bloqueantes
):
    """
    Crear sección de información detallada organizada.

    Issue #412: Ahora muestra warnings en amarillo (no-bloqueantes) separados de errores (rojos).
    """
    return html.Div(
        [
            # Información del archivo y estado
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H6("Información del Procesamiento", className="border-bottom pb-2 mb-2"),
                            html.P(
                                [
                                    html.Strong("ID de Upload: "),
                                    html.Code(upload_id[:8] + "..." if upload_id else "N/A"),
                                ],
                                className="mb-2 small",
                            ),
                            # Estadísticas de progreso (solo si hay datos)
                            *(
                                [
                                    html.P(
                                        [
                                            html.I(className="fas fa-list-ol me-2 text-muted"),
                                            html.Strong("Registros totales: "),
                                            format_number(rows_total),
                                        ],
                                        className="mb-1 small",
                                    ),
                                    html.P(
                                        [
                                            html.I(className="fas fa-check-square me-2 text-success"),
                                            html.Strong("Registros procesados: "),
                                            format_number(rows_processed) if rows_processed else "0",
                                        ],
                                        className="mb-1 small",
                                    ),
                                    *(
                                        [
                                            html.P(
                                                [
                                                    html.I(className="fas fa-exclamation-triangle me-2 text-warning"),
                                                    html.Strong("Registros con errores: "),
                                                    html.Span(
                                                        format_number(rows_with_errors),
                                                        className="text-warning fw-bold",
                                                    ),
                                                ],
                                                className="mb-1 small",
                                            )
                                        ]
                                        if rows_with_errors and rows_with_errors > 0
                                        else []
                                    ),
                                ]
                                if rows_total is not None
                                else []
                            ),
                        ]
                    )
                ],
                className="mb-3",
            ),
            # Notas del sistema (más prominentes y organizadas)
            *(
                [
                    html.Div(
                        [
                            html.H6(
                                [html.I(className="fas fa-info-circle me-2"), "Estado del Sistema"],
                                className="border-bottom pb-2 mb-2",
                            ),
                            html.Div(
                                [_format_processing_notes(processing_notes)],
                                className="alert alert-info py-2 px-3 mb-3",
                            ),
                        ]
                    )
                ]
                if processing_notes
                else []
            ),
            # Timestamp de finalización
            *(
                [
                    html.P(
                        [
                            html.I(className="fas fa-calendar-check me-2 text-muted"),
                            html.Strong("Completado: "),
                            format_timestamp(processing_completed_at),
                        ],
                        className="mb-1 small text-muted",
                    )
                ]
                if processing_completed_at
                else []
            ),
            # Issue #412: Warnings no-bloqueantes (amarillo) - separados de errores
            *create_warnings_section(warnings),
            # Mensaje de error o info (prominente si existe)
            *(
                [
                    html.Div(
                        [
                            html.H6(
                                [
                                    html.I(
                                        className=f"fas fa-{'exclamation-circle' if phase_info.get('phase_name') == 'Error' else 'info-circle'} me-2"
                                    ),
                                    "Detalles del Error"
                                    if phase_info.get("phase_name") == "Error"
                                    else "Resumen del Procesamiento",
                                ],
                                className=f"{'text-danger' if phase_info.get('phase_name') == 'Error' else 'text-info'} border-bottom pb-2 mb-2",
                            ),
                            html.P(
                                error_message,
                                className=f"{'text-danger' if phase_info.get('phase_name') == 'Error' else 'text-muted'} mb-0 small",
                            ),
                        ],
                        className=f"alert alert-{'danger' if phase_info.get('phase_name') == 'Error' else 'info'} py-2 px-3",
                    )
                ]
                if error_message
                else []
            ),
        ]
    )


def create_warnings_section(warnings):
    """
    Issue #412: Crear sección de warnings no-bloqueantes (amarillo).
    Helper DRY usado por _create_detailed_info_section() y create_final_progress_summary().

    Args:
        warnings: Lista de diccionarios con warnings [{type, message, technical?}]

    Returns:
        Lista con html.Div de warnings o lista vacía si no hay warnings
    """
    if not warnings or len(warnings) == 0:
        return []

    return [
        html.Div(
            [
                html.H6(
                    [
                        html.I(className="fas fa-info-circle me-2"),
                        "Información del Proceso",
                    ],
                    className="text-warning border-bottom pb-2 mb-2",
                ),
                *[
                    dbc.Alert(
                        [
                            html.I(className="fas fa-lightbulb me-2"),
                            w.get("message", str(w)),
                        ],
                        color="warning",
                        className="py-2 px-3 mb-2",
                    )
                    for w in warnings
                ],
            ],
            className="mb-3",
        )
    ]


def _format_processing_notes(processing_notes):
    """
    Formatear las notas de procesamiento para mejor legibilidad.
    """
    if not processing_notes:
        return ""

    # Detectar diferentes tipos de mensajes y formatearlos apropiadamente
    if "Sistema preparándose" in processing_notes:
        return html.Div(
            [html.I(className="fas fa-cog me-2"), html.Strong("Sistema: "), processing_notes.strip()]
        )

    if "Descargando" in processing_notes:
        return html.Div(
            [
                html.I(className="fas fa-download me-2"),
                html.Strong("Descarga: "),
                processing_notes.strip(),
            ]
        )

    if "cargado" in processing_notes:
        return html.Div(
            [
                html.I(className="fas fa-check-circle me-2"),
                html.Strong("Progreso: "),
                processing_notes.strip(),
            ]
        )

    if "|" in processing_notes:
        # Formatear estadísticas estructuradas
        parts = processing_notes.split("|")
        return html.Div(
            [
                html.Div([html.I(className="fas fa-chart-bar me-2"), html.Strong("Estadísticas: ")]),
                html.Ul(
                    [html.Li(part.strip(), className="small") for part in parts if part.strip()], className="mb-0 small"
                ),
            ]
        )

    # Formato por defecto
    return html.Span(processing_notes)


# ========================================
# ERROR PROGRESS CARD
# ========================================


def create_error_progress_card(upload_id, error):
    """Crear tarjeta de error en progreso."""
    return dbc.Card(
        [
            dbc.CardHeader([html.I(className="fas fa-exclamation-triangle text-danger me-2"), "Error de Monitoreo"]),
            dbc.CardBody(
                [
                    html.P([html.Strong("Upload ID: "), html.Code(upload_id[:8] + "..." if upload_id else "N/A")]),
                    html.P(f"Error: {error}", className="text-danger"),
                ]
            ),
        ],
        color="danger",
        outline=True,
    )


# ========================================
# HISTORY COMPONENTS
# ========================================


def load_upload_history():
    """
    Cargar historial de uploads desde el backend.
    """

    try:
        from utils.pharmacy_context import get_current_pharmacy_id

        response = backend_client.get_upload_history(pharmacy_id=get_current_pharmacy_id(), limit=10)

        if not response.success:
            return create_empty_history_message()

        uploads = response.data

        if not uploads:
            return create_empty_history_message()

        # Crear lista de uploads
        history_items = []

        for upload in uploads:
            status = upload.get("status", "unknown")

            # Configurar estilo según estado
            status_badge = {
                "completed": ("success", "fas fa-check-circle"),
                "error": ("danger", "fas fa-times-circle"),
                "processing": ("primary", "fas fa-spinner fa-spin"),
                "pending": ("info", "fas fa-clock"),
            }.get(status, ("secondary", "fas fa-question-circle"))

            history_items.append(
                dbc.ListGroupItem(
                    [
                        html.Div(
                            [
                                html.Div(
                                    [
                                        html.I(className=f"{status_badge[1]} me-2"),
                                        html.Strong(upload.get("filename", "Sin nombre")),
                                        html.Span(status.upper(), className=f"badge bg-{status_badge[0]} ms-auto"),
                                    ],
                                    className="d-flex justify-content-between align-items-center mb-1",
                                ),
                                html.Small(
                                    [
                                        f"Subido: {format_timestamp(upload.get('uploaded_at'))} | ",
                                        format_number(upload.get("rows_processed", 0)).replace(",", "."),
                                        *(
                                            [f" | Tipo: {upload.get('file_type', 'unknown').upper()}"]
                                            if upload.get("file_type")
                                            else []
                                        ),
                                    ],
                                    className="text-muted",
                                ),
                            ]
                        )
                    ]
                )
            )

        return dbc.Card([dbc.CardBody([dbc.ListGroup(history_items, flush=True)])], className="shadow-sm")

    except Exception as e:
        logger.error(f"Error cargando historial: {str(e)}")
        return create_history_error_message(str(e))


def create_empty_history_message():
    """Mensaje cuando no hay historial - usa componente EmptyState centralizado."""
    return empty_upload_history_state()


def create_history_error_message(error):
    """Mensaje de error en historial."""
    return dbc.Card(
        [
            dbc.CardBody(
                [
                    html.Div(
                        [
                            html.I(className="fas fa-exclamation-triangle fa-2x text-warning mb-3"),
                            html.H6("Error cargando historial", className="text-warning"),
                            html.P(f"Error: {error}", className="text-muted small"),
                        ],
                        className="text-center py-3",
                    )
                ]
            )
        ],
        color="warning",
        outline=True,
    )


# ========================================
# TIMESTAMP FORMATTING
# ========================================


def format_timestamp(timestamp):
    """Formatear timestamp para mostrar."""
    if not timestamp:
        return "N/A"

    try:
        if isinstance(timestamp, str):
            # Parsear timestamp ISO
            dt = datetime.fromisoformat(timestamp.replace("Z", "+00:00"))
        else:
            dt = timestamp

        return dt.strftime("%d/%m/%Y %H:%M")
    except (ValueError, TypeError, AttributeError):
        return str(timestamp)


# ========================================
# ENRICHMENT PROGRESS COMPONENT
# ========================================


def create_enrichment_progress_component(filename: str, enrichment_progress: dict) -> dbc.Card:
    """
    Crear componente de progreso de enriquecimiento durante upload.
    Issue #264 Fase 2 - UI/UX de Progreso de Enriquecimiento

    Args:
        filename: Nombre del archivo
        enrichment_progress: Diccionario con datos de progreso

    Returns:
        Componente dbc.Card con progreso de enriquecimiento
    """
    # Extraer datos de progreso
    total = enrichment_progress.get("total", 0)
    processed = enrichment_progress.get("processed", 0)
    enriched = enrichment_progress.get("enriched", 0)
    manual_review = enrichment_progress.get("manual_review", 0)
    failed = enrichment_progress.get("failed", 0)
    percentage = enrichment_progress.get("percentage", 0)
    phase = enrichment_progress.get("phase", "Procesando...")

    # Coincidencias detalladas
    matches_by_code = enrichment_progress.get("matches_by_code", 0)
    matches_by_name = enrichment_progress.get("matches_by_name", 0)
    matches_by_ean = enrichment_progress.get("matches_by_ean", 0)

    return BaseCard(
        children=[
            dbc.CardHeader(
                [html.I(className="fas fa-magic me-2 text-primary"), f"Enriqueciendo: {filename}"],
                className="d-flex align-items-center",
            ),
            dbc.CardBody(
                [
                    # Título de fase
                    Title("Enriqueciendo datos con CIMA y Nomenclator...", level=5),
                    # Barra de progreso
                    html.Div(
                        [
                            dbc.Progress(
                                value=percentage,
                                striped=True,
                                animated=True,
                                color="primary",
                                className="mb-2",
                                style={"height": "20px"},
                            ),
                            html.Div(
                                [
                                    html.Strong(f"{percentage:.1f}%", className="text-primary"),
                                    html.Span(f" - {phase}", className="text-muted ms-2"),
                                ]
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Contador de registros
                    html.Div(
                        [
                            html.I(className="fas fa-list-ol me-2 text-muted"),
                            html.Strong("Procesados: "),
                            f"{format_number(processed)} de {format_number(total)} registros",
                        ],
                        className="mb-3",
                    ),
                    # Estadísticas detalladas
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-check-circle fa-sm text-success me-2"),
                                            html.Small("Enriquecidos", className="text-muted"),
                                        ]
                                    ),
                                    html.Div(format_number(enriched), className="h6 mb-0 text-success"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-search fa-sm text-warning me-2"),
                                            html.Small("Revisión manual", className="text-muted"),
                                        ]
                                    ),
                                    html.Div(format_number(manual_review), className="h6 mb-0 text-warning"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Div(
                                        [
                                            html.I(className="fas fa-times-circle fa-sm text-danger me-2"),
                                            html.Small("Fallidos", className="text-muted"),
                                        ]
                                    ),
                                    html.Div(format_number(failed), className="h6 mb-0 text-danger"),
                                ],
                                width=4,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Métodos de coincidencia (solo si hay datos)
                    (
                        html.Div(
                            [
                                html.Hr(),
                                html.Small("Coincidencias por método:", className="text-muted d-block mb-2"),
                                html.Ul(
                                    [
                                        html.Li(
                                            [
                                                html.I(className="fas fa-hashtag fa-sm me-2"),
                                                f"Por código nacional: {format_number(matches_by_code)}",
                                            ],
                                            className="small",
                                        ),
                                        html.Li(
                                            [
                                                html.I(className="fas fa-barcode fa-sm me-2"),
                                                f"Por EAN: {format_number(matches_by_ean)}",
                                            ],
                                            className="small",
                                        ),
                                        html.Li(
                                            [
                                                html.I(className="fas fa-file-signature fa-sm me-2"),
                                                f"Por nombre: {format_number(matches_by_name)}",
                                            ],
                                            className="small",
                                        ),
                                    ],
                                    className="mb-0",
                                ),
                            ]
                        )
                        if (matches_by_code + matches_by_name + matches_by_ean > 0)
                        else html.Div()
                    ),
                ]
            ),
        ],
        variant="default",
        shadow="sm",
    )


def create_enrichment_success_toast(filename: str, enrichment_progress: dict) -> dict:
    """
    Crear toast de éxito con estadísticas de enriquecimiento.
    Issue #264 Fase 2 - Toast de Finalización

    Args:
        filename: Nombre del archivo procesado
        enrichment_progress: Diccionario con estadísticas finales

    Returns:
        Diccionario para toast_trigger_store
    """
    # Extraer estadísticas
    enriched = enrichment_progress.get("enriched", 0)
    manual_review = enrichment_progress.get("manual_review", 0)
    matches_by_code = enrichment_progress.get("matches_by_code", 0)
    matches_by_ean = enrichment_progress.get("matches_by_ean", 0)
    matches_by_name = enrichment_progress.get("matches_by_name", 0)

    # Crear mensaje con estadísticas
    message_lines = [f"Enriquecidos: {format_number(enriched)} productos"]

    if manual_review > 0:
        message_lines.append(f"Revisión manual: {format_number(manual_review)} productos")

    # Añadir coincidencias solo si hay datos
    if matches_by_code > 0:
        message_lines.append(f"Código nacional: {format_number(matches_by_code)}")
    if matches_by_ean > 0:
        message_lines.append(f"EAN: {format_number(matches_by_ean)}")
    if matches_by_name > 0:
        message_lines.append(f"Nombre: {format_number(matches_by_name)}")

    message = "\n".join(message_lines)

    return success_toast(message, title=f"Enriquecimiento Completado: {filename}")


# ========================================
# FINAL PROGRESS SUMMARY
# ========================================


def create_final_progress_summary(
    filename,
    status,
    rows_total=None,
    rows_processed=None,
    rows_duplicates=None,
    enrichment_progress=None,
    upload_id=None,
    error_message=None,
    warnings=None,  # Issue #412: Avisos no-bloqueantes
):
    """
    Crear componente de resumen final PERSISTENTE después de completar upload.
    Issue #287 - Solución a feedback visual insuficiente
    Issue #412 - Muestra warnings (amarillo) separados de errores (rojo)

    DIFERENCIA CON create_progress_card():
    - Este componente NO se actualiza (estado final frozen)
    - Tiene botón "Cerrar" explícito para que usuario controle cuándo desaparece
    - Usa colores finales (verde success, rojo error)
    - Muestra estadísticas completas de enriquecimiento
    - Muestra warnings en amarillo (no-bloqueantes)
    - Persiste hasta que usuario cierra manualmente

    Args:
        filename: Nombre del archivo procesado
        status: Estado final ('completed', 'error', 'failed')
        rows_total: Total de registros en el archivo
        rows_processed: Registros procesados exitosamente
        enrichment_progress: Dict con estadísticas de enriquecimiento
        upload_id: ID único del upload (para botón dismiss)
        error_message: Mensaje de error (si status='error')
        warnings: Lista de avisos no-bloqueantes (Issue #412)

    Returns:
        dbc.Card con resumen final persistente
    """
    # Inicializar enrichment_progress si es None
    if enrichment_progress is None:
        enrichment_progress = {}

    # Color y estilo según resultado
    if status == "completed":
        card_color = "success"
        icon = "fas fa-check-circle"
        header_text = f"Procesamiento Completado: {filename}"
        header_class = "bg-success text-white"
    else:  # error o failed
        card_color = "danger"
        icon = "fas fa-times-circle"
        header_text = f"Error en Procesamiento: {filename}"
        header_class = "bg-danger text-white"

    # Construir contenido del card body
    card_body_content = []

    # === SECCIÓN 1: RESUMEN DE VENTAS ===
    if rows_total is not None:
        # Calcular registros nuevos vs duplicados (Issue #330)
        rows_new = (rows_processed - rows_duplicates) if (rows_processed and rows_duplicates) else rows_processed

        card_body_content.extend(
            [
                Title("Datos de Ventas Procesados", level=5),
                # Primera fila: Total, Nuevos, Duplicados
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-list-ol fa-sm text-muted me-2"),
                                        html.Small("Total registros:", className="text-muted"),
                                    ]
                                ),
                                html.Div(format_number(rows_total), className="h5 mb-0 text-primary"),
                            ],
                            width=4,
                        ),
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-plus-circle fa-sm text-success me-2"),
                                        html.Small("Nuevos guardados:", className="text-muted"),
                                    ]
                                ),
                                html.Div(
                                    format_number(rows_new) if rows_new else "0",
                                    className="h5 mb-0 text-success",
                                ),
                            ],
                            width=4,
                        ),
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-copy fa-sm text-warning me-2"),
                                        html.Small("Duplicados rechazados:", className="text-muted"),
                                    ]
                                ),
                                html.Div(
                                    format_number(rows_duplicates) if rows_duplicates else "0",
                                    className="h5 mb-0 text-warning",
                                ),
                            ],
                            width=4,
                        ),
                    ],
                    className="mb-3",
                ),
            ]
        )

    # === SECCIÓN 2: RESUMEN DE ENRIQUECIMIENTO ===
    if enrichment_progress and enrichment_progress.get("enriched", 0) > 0:
        card_body_content.extend(
            [
                html.Hr(),
                Title("Enriquecimiento con CIMA/Nomenclator", level=5),
                dbc.Row(
                    [
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-check-circle fa-sm text-success me-2"),
                                        html.Small("Enriquecidos:", className="text-muted"),
                                    ]
                                ),
                                html.Div(
                                    format_number(enrichment_progress.get("enriched", 0)),
                                    className="h5 mb-0 text-success",
                                ),
                            ],
                            width=4,
                        ),
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-search fa-sm text-warning me-2"),
                                        html.Small("Revisión manual:", className="text-muted"),
                                    ]
                                ),
                                html.Div(
                                    format_number(enrichment_progress.get("manual_review", 0)),
                                    className="h5 mb-0 text-warning",
                                ),
                            ],
                            width=4,
                        ),
                        dbc.Col(
                            [
                                html.Div(
                                    [
                                        html.I(className="fas fa-times-circle fa-sm text-danger me-2"),
                                        html.Small("No enriquecidos:", className="text-muted"),
                                    ]
                                ),
                                html.Div(
                                    format_number(enrichment_progress.get("failed", 0)), className="h5 mb-0 text-danger"
                                ),
                            ],
                            width=4,
                        ),
                    ],
                    className="mb-3",
                ),
            ]
        )

    # === SECCIÓN 3: WARNINGS NO-BLOQUEANTES (Issue #412) ===
    # Usar helper DRY create_warnings_section()
    warnings_section = create_warnings_section(warnings)
    if warnings_section:
        card_body_content.extend([html.Hr()] + warnings_section)

    # === SECCIÓN 4: MENSAJE DE ERROR O INFO (si existe) ===
    if error_message:
        # Detectar si es error real o información de pipeline
        is_actual_error = status in ["error", "failed"]

        # Determinar color, ícono y título según el tipo de mensaje
        if is_actual_error:
            alert_color = "danger"
            alert_icon = "fas fa-exclamation-circle"
            alert_title = "Detalles del error: "
        else:
            alert_color = "info"
            alert_icon = "fas fa-info-circle"
            alert_title = "Resumen del procesamiento: "

        card_body_content.extend(
            [
                html.Hr(),
                dbc.Alert(
                    children=html.Span(
                        [
                            html.I(className=f"{alert_icon} me-2"),
                            html.Strong(alert_title),
                            html.Span(error_message),
                        ]
                    ),
                    color=alert_color,
                    className="mb-3",
                ),
            ]
        )

    # === SECCIÓN 5: PRÓXIMOS PASOS (solo si completó exitosamente) ===
    if status == "completed":
        card_body_content.extend(
            [
                html.Hr(),
                dbc.Alert(
                    children=html.Span(
                        [
                            html.I(className="fas fa-info-circle me-2"),
                            html.Strong("Próximos pasos: "),
                            "Los datos están listos. Puedes consultar el dashboard para ver las ventas o subir otro archivo.",
                        ]
                    ),
                    color="info",
                    className="mb-0",
                ),
            ]
        )

    # Crear card directamente con ID con MATCH para poder cerrar desde callback
    # BaseCard no soporta IDs con pattern matching, usamos dbc.Card directamente
    return dbc.Card(
        id={"type": "upload-summary-card", "index": upload_id or "unknown"},
        children=[
            dbc.CardHeader(
                [
                    html.Div(
                        children=[html.I(className=f"{icon} me-2"), header_text],
                        className="d-flex align-items-center flex-grow-1",
                    ),
                    # Botón "Cerrar" explícito (patrón match para callback)
                    (
                        dbc.Button(
                            html.I(className="fas fa-times"),
                            id={"type": "dismiss-upload-summary", "index": upload_id or "unknown"},
                            color="link",
                            size="sm",
                            className="text-white ms-2",
                            title="Cerrar resumen",
                        )
                        if upload_id
                        else None
                    ),
                ],
                className=f"d-flex justify-content-between align-items-center {header_class}",
            ),
            dbc.CardBody(card_body_content),
        ],
        outline=True,
        color=card_color,
        className="shadow-sm mb-3",
    )
