from utils.helpers import format_number

# frontend/components/upload.py
"""
Componentes para carga de archivos de datos de farmacia.
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import dcc, html
from utils.constants import APP_CONFIG, COLORS, ERP_TYPES


def create_file_upload_zone(
    upload_id: str = "upload-data",
    accepted_files: Optional[List[str]] = None,
    max_size_mb: int = APP_CONFIG["file_upload"]["max_size_mb"],
    multiple: bool = False,
) -> html.Div:
    """
    Crear zona de carga de archivos drag & drop.

    Args:
        upload_id: ID del componente de upload
        accepted_files: Lista de extensiones permitidas
        max_size_mb: Tamaño máximo en MB
        multiple: Si permite múltiples archivos

    Returns:
        Componente html.Div con zona de upload
    """

    if accepted_files is None:
        accepted_files = APP_CONFIG["file_upload"]["allowed_extensions"]

    # Formatear lista de extensiones
    extensions_text = ", ".join(accepted_files)

    return html.Div(
        [
            dcc.Upload(
                id=upload_id,
                children=html.Div(
                    [
                        # Icono principal
                        html.I(className="fas fa-cloud-upload-alt fa-3x text-primary mb-3"),
                        # Texto principal
                        html.H5("Arrastra tu archivo aquí", className="text-primary mb-2"),
                        html.P("o haz clic para seleccionar", className="text-muted mb-3"),
                        # Información de archivos aceptados
                        html.Div(
                            [
                                html.Small(
                                    [html.Strong("Formatos permitidos: "), extensions_text],
                                    className="text-muted d-block",
                                ),
                                html.Small(
                                    [html.Strong("Tamaño máximo: "), f"{max_size_mb} MB"],
                                    className="text-muted d-block",
                                ),
                            ],
                            className="mt-2",
                        ),
                    ]
                ),
                style={
                    "width": "100%",
                    "height": "200px",
                    "lineHeight": "200px",
                    "borderWidth": "2px",
                    "borderStyle": "dashed",
                    "borderRadius": "10px",
                    "borderColor": COLORS["primary"],
                    "textAlign": "center",
                    "background-color": "#f8f9fa",
                    "cursor": "pointer",
                    "transition": "all 0.3s ease",
                },
                multiple=multiple,
                accept=",".join(accepted_files),
                className="upload-zone",
            )
        ]
    )


def create_erp_selector(selector_id: str = "erp-type", default_value: str = "farmatic") -> dbc.Card:
    """
    Crear selector de tipo de ERP.

    Args:
        selector_id: ID del componente selector
        default_value: Valor por defecto

    Returns:
        Componente dbc.Card con selector
    """

    # Crear opciones del selector
    options = [{"label": name, "value": key} for key, name in ERP_TYPES.items()]

    return dbc.Card(
        [
            dbc.CardHeader([html.I(className="fas fa-desktop me-2"), "Tipo de Sistema ERP"]),
            dbc.CardBody(
                [
                    html.P("Selecciona el sistema de gestión de tu farmacia:", className="text-muted mb-3"),
                    dbc.RadioItems(
                        id=selector_id, options=options, value=default_value, inline=False, className="erp-selector"
                    ),
                ]
            ),
        ],
        className="mb-4",
    )


def create_upload_progress(progress: int = 0, status: str = "pending", filename: Optional[str] = None) -> html.Div:
    """
    Crear indicador de progreso de carga.

    Args:
        progress: Porcentaje de progreso (0-100)
        status: Estado de la carga
        filename: Nombre del archivo

    Returns:
        Componente html.Div con progreso
    """

    # Determinar color según el estado
    status_colors = {
        "pending": "secondary",
        "uploading": "info",
        "processing": "warning",
        "completed": "success",
        "error": "danger",
    }

    color = status_colors.get(status, "secondary")

    # Mensajes según el estado
    status_messages = {
        "pending": "Esperando...",
        "uploading": f"Subiendo archivo... {progress}%",
        "processing": "Procesando datos...",
        "completed": "¡Completado exitosamente!",
        "error": "Error en la carga",
    }

    message = status_messages.get(status, "Procesando...")

    elements = []

    # Información del archivo
    if filename:
        elements.append(html.Div([html.I(className="fas fa-file me-2"), html.Strong(filename)], className="mb-2"))

    # Barra de progreso
    if status in ["uploading", "processing"]:
        elements.append(
            dbc.Progress(value=progress, color=color, striped=True, animated=(status == "processing"), className="mb-2")
        )

    # Mensaje de estado
    elements.append(
        html.Div(
            [
                html.I(
                    className=f"fas fa-{'check' if status == 'completed' else 'exclamation-triangle' if status == 'error' else 'clock'} me-2"
                ),
                message,
            ],
            className=f"text-{color}",
        )
    )

    return html.Div(elements, className="upload-progress")


def create_upload_history(uploads: List[Dict[str, Any]]) -> dbc.Card:
    """
    Crear tabla de historial de cargas.

    Args:
        uploads: Lista de uploads con información

    Returns:
        Componente dbc.Card con tabla
    """

    if not uploads:
        return dbc.Card(
            [
                dbc.CardBody(
                    [
                        html.Div(
                            [
                                html.I(className="fas fa-inbox fa-2x text-muted mb-2"),
                                html.P("No hay archivos cargados anteriormente", className="text-muted"),
                            ],
                            className="text-center py-4",
                        )
                    ]
                )
            ]
        )

    # Crear filas de la tabla
    table_rows = []
    for upload in uploads:
        # Formatear fecha
        fecha = upload.get("created_at", "")
        if fecha:
            try:
                from datetime import datetime

                fecha_dt = datetime.fromisoformat(fecha.replace("Z", "+00:00"))
                fecha = fecha_dt.strftime("%d/%m/%Y %H:%M")
            except:
                pass

        # Estado con badge
        status = upload.get("status", "unknown")
        status_badge = create_status_badge(status)

        # Información de procesamiento
        stats = upload.get("processing_stats", {})
        processed = stats.get("rows_processed", 0)
        errors = stats.get("rows_with_errors", 0)

        table_rows.append(
            html.Tr(
                [
                    html.Td(upload.get("original_filename", "Sin nombre")),
                    html.Td(upload.get("file_type", "unknown").upper()),
                    html.Td(fecha),
                    html.Td(status_badge),
                    html.Td(format_number(processed).replace(",", ".")),
                    html.Td(format_number(errors).replace(",", ".") if errors > 0 else "-"),
                    html.Td(
                        [
                            dbc.Button(
                                html.I(className="fas fa-eye"),
                                size="sm",
                                color="outline-primary",
                                id={"type": "view-upload", "index": upload.get("id")},
                                title="Ver detalles",
                                className="me-1",
                            ),
                            dbc.Button(
                                html.I(className="fas fa-trash-alt"),
                                size="sm",
                                color="outline-danger",
                                id={"type": "delete-upload", "index": upload.get("id")},
                                title="Eliminar",
                            ),
                        ],
                        className="text-nowrap",
                    ),
                ]
            )
        )

    return dbc.Card(
        [
            dbc.CardHeader([html.I(className="fas fa-history me-2"), "Historial de Cargas"]),
            dbc.CardBody(
                [
                    dbc.Table(
                        [
                            html.Thead(
                                [
                                    html.Tr(
                                        [
                                            html.Th("Archivo"),
                                            html.Th("Tipo"),
                                            html.Th("Fecha"),
                                            html.Th("Estado"),
                                            html.Th("Procesados"),
                                            html.Th("Errores"),
                                            html.Th("Acciones"),
                                        ]
                                    )
                                ]
                            ),
                            html.Tbody(table_rows),
                        ],
                        striped=True,
                        hover=True,
                        responsive=True,
                    )
                ],
                className="p-0",
            ),
        ],
        className="shadow-sm",
    )


def create_upload_summary(stats: Dict[str, Any]) -> dbc.Card:
    """
    Crear tarjeta de resumen de carga.

    Args:
        stats: Estadísticas del procesamiento

    Returns:
        Componente dbc.Card con resumen
    """

    total_rows = stats.get("total_rows", 0)
    processed_rows = stats.get("rows_processed", 0)
    error_rows = stats.get("rows_with_errors", 0)
    warnings = stats.get("warnings", [])

    # Calcular porcentajes
    success_rate = (processed_rows / total_rows * 100) if total_rows > 0 else 0
    error_rate = (error_rows / total_rows * 100) if total_rows > 0 else 0

    return dbc.Card(
        [
            dbc.CardHeader([html.I(className="fas fa-chart-pie me-2 text-success"), "Resumen de Procesamiento"]),
            dbc.CardBody(
                [
                    # Estadísticas principales
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H4(format_number(total_rows).replace(",", "."), className="text-primary mb-0"),
                                    html.Small("Filas Totales", className="text-muted"),
                                ],
                                width=3,
                                className="text-center",
                            ),
                            dbc.Col(
                                [
                                    html.H4(
                                        format_number(processed_rows).replace(",", "."), className="text-success mb-0"
                                    ),
                                    html.Small("Procesadas", className="text-muted"),
                                ],
                                width=3,
                                className="text-center",
                            ),
                            dbc.Col(
                                [
                                    html.H4(format_number(error_rows).replace(",", "."), className="text-danger mb-0"),
                                    html.Small("Con Errores", className="text-muted"),
                                ],
                                width=3,
                                className="text-center",
                            ),
                            dbc.Col(
                                [
                                    html.H4(f"{success_rate:.1f}%".replace(".", ","), className="text-info mb-0"),
                                    html.Small("Tasa de Éxito", className="text-muted"),
                                ],
                                width=3,
                                className="text-center",
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Barra de progreso visual
                    dbc.Progress(
                        [
                            dbc.Progress(value=success_rate, color="success"),
                            dbc.Progress(value=error_rate, color="danger"),
                        ],
                        className="mb-3",
                    ),
                    # Advertencias si las hay
                    *([html.Hr()] if warnings else []),
                    *[
                        dbc.Alert(
                            [html.I(className="fas fa-exclamation-triangle me-2"), warning],
                            color="warning",
                            className="small mb-2",
                        )
                        for warning in warnings[:3]
                    ],
                ]
            ),
        ],
        className="shadow-sm",
    )


def create_upload_success_alert(filename: str, stats: dict) -> dbc.Alert:
    """
    Crear alerta de éxito después de upload completado.

    Args:
        filename: Nombre del archivo subido
        stats: Estadísticas de procesamiento

    Returns:
        Componente dbc.Alert
    """

    total_rows = stats.get("total_rows", 0)
    processed_rows = stats.get("rows_processed", 0)

    return dbc.Alert(
        [
            html.I(className="fas fa-check-circle me-2"),
            html.Strong("¡Archivo procesado exitosamente!"),
            html.Hr(),
            html.P(
                [
                    f"Archivo: {filename}",
                    html.Br(),
                    format_number(total_rows).replace(",", "."),
                    html.Br(),
                    format_number(processed_rows).replace(",", "."),
                ],
                className="mb-0",
            ),
        ],
        color="success",
        dismissable=True,
        duration=10000,
    )


def create_upload_error_alert(filename: str, error_message: str) -> dbc.Alert:
    """
    Crear alerta de error en upload.

    Args:
        filename: Nombre del archivo
        error_message: Mensaje de error

    Returns:
        Componente dbc.Alert
    """

    return dbc.Alert(
        [
            html.I(className="fas fa-exclamation-triangle me-2"),
            html.Strong("Error al procesar archivo"),
            html.Hr(),
            html.P([f"Archivo: {filename}", html.Br(), f"Error: {error_message}"], className="mb-0"),
        ],
        color="danger",
        dismissable=True,
    )


def create_validation_warnings(warnings: list) -> html.Div:
    """
    Crear lista de advertencias de validación.

    Args:
        warnings: Lista de mensajes de advertencia

    Returns:
        Componente html.Div con advertencias
    """

    if not warnings:
        return html.Div()

    return dbc.Alert(
        [
            html.I(className="fas fa-exclamation-circle me-2"),
            html.Strong(f"Se encontraron {len(warnings)} advertencias:"),
            html.Ul([html.Li(warning) for warning in warnings[:5]], className="mb-0 mt-2"),  # Mostrar máximo 5
        ]
        + ([html.P(f"... y {len(warnings) - 5} más", className="mb-0 mt-2")] if len(warnings) > 5 else []),
        color="warning",
        dismissable=True,
    )


def create_status_badge(status: str) -> dbc.Badge:
    """
    Crear badge de estado para uploads.

    Args:
        status: Estado del upload

    Returns:
        Componente dbc.Badge
    """

    status_config = {
        "pending": {"color": "secondary", "text": "Pendiente"},
        "uploading": {"color": "info", "text": "Subiendo"},
        "processing": {"color": "warning", "text": "Procesando"},
        "completed": {"color": "success", "text": "Completado"},
        "failed": {"color": "danger", "text": "Error"},
        "cancelled": {"color": "dark", "text": "Cancelado"},
    }

    config = status_config.get(status, {"color": "secondary", "text": status.title()})

    return dbc.Badge(config["text"], color=config["color"], pill=True)


def create_erp_help_section(user_erp_type: Optional[str] = None) -> html.Div:
    """
    Crear sección de ayuda con instrucciones de exportación del ERP del usuario.
    Issue #429 - Mejorar UX con ayuda contextual.
    Issue #476 - Instrucciones para ventas e inventario con detección automática.

    Muestra instrucciones para exportar ventas e inventario.
    El sistema detecta automáticamente el tipo de archivo al subirlo.

    Args:
        user_erp_type: Tipo de ERP del usuario ('farmanager', 'farmatic', etc.)

    Returns:
        Componente html.Div con instrucciones específicas del ERP
    """
    # === FARMANAGER ===
    # Instrucciones de VENTAS para Farmanager
    farmanager_sales = html.Div([
        html.H6([
            html.I(className="bi bi-cart3 me-2"),
            "Exportar Ventas"
        ], className="text-primary mb-3"),
        html.Ol([
            html.Li([
                "Ir a ",
                html.Strong("Ventas → Histórico → Líneas de mostrador")
            ]),
            html.Li("Elegir el rango de fechas deseado"),
            html.Li([
                "Pulsar ",
                html.Strong("F10"),
                " o hacer clic en ",
                html.Strong("Exportar a XLS")
            ]),
            html.Li([
                "Seleccionar formato ",
                html.Strong("Detallado")
            ]),
            html.Li([
                "Campos a exportar:",
                html.Ul([
                    html.Li("idOperación"),
                    html.Li("Fecha/Hora"),
                    html.Li("Artículo"),
                    html.Li("Descripción"),
                    html.Li("Unidades"),
                    html.Li("Precio coste"),
                    html.Li("PVP"),
                    html.Li("Descuento"),
                    html.Li("Importe"),
                    html.Li("Empleado"),
                    html.Li("Cliente"),
                ], className="mt-1 mb-0", style={"columns": "2"})
            ]),
        ], className="mb-0")
    ])

    # Instrucciones de INVENTARIO para Farmanager
    farmanager_inventory = html.Div([
        html.H6([
            html.I(className="bi bi-box-seam me-2"),
            "Exportar Inventario"
        ], className="text-success mb-3"),
        html.Ol([
            html.Li([
                "Ir a ",
                html.Strong("Artículos → Listados → Listado de Artículos")
            ]),
            html.Li([
                "Pulsar ",
                html.Strong("F10"),
                " para exportar a XLS"
            ]),
            html.Li([
                "Seleccionar los siguientes campos:",
                html.Ul([
                    html.Li("Código"),
                    html.Li("Descripción"),
                    html.Li("Stock"),
                    html.Li("PCM"),
                    html.Li("PVP"),
                    html.Li("IVA"),
                    html.Li("Ultima compra"),
                    html.Li("Ultima venta"),
                ], className="mt-1 mb-0", style={"columns": "2"})
            ]),
        ], className="mb-0")
    ])

    # Contenedor completo para Farmanager
    farmanager_instructions = html.Div([
        # Nota de detección automática
        dbc.Alert(
            html.Span([
                html.I(className="bi bi-magic me-2"),
                html.Strong("Detección automática: "),
                "El sistema detecta automáticamente si es un archivo de ventas o inventario."
            ]),
            color="info", className="mb-3 py-2"
        ),

        dbc.Row([
            dbc.Col([farmanager_sales], width=12, lg=6, className="mb-3 mb-lg-0"),
            dbc.Col([farmanager_inventory], width=12, lg=6),
        ])
    ])

    # === FARMATIC ===
    # Instrucciones de VENTAS para Farmatic
    farmatic_sales = html.Div([
        html.H6([
            html.I(className="bi bi-cart3 me-2"),
            "Exportar Ventas"
        ], className="text-primary mb-3"),
        html.Ol([
            html.Li([
                "Ir a ",
                html.Strong("Ventas → Histórico de ventas")
            ]),
            html.Li("Seleccionar el rango de fechas deseado"),
            html.Li([
                "Hacer clic en ",
                html.Strong("Configurar"),
                " y seleccionar los campos a exportar"
            ]),
            html.Li([
                "Hacer clic en ",
                html.Strong("Exportar a Excel")
            ]),
        ], className="mb-0")
    ])

    # Instrucciones de INVENTARIO para Farmatic
    farmatic_inventory = html.Div([
        html.H6([
            html.I(className="bi bi-box-seam me-2"),
            "Exportar Inventario"
        ], className="text-success mb-3"),
        html.Ol([
            html.Li([
                "Ir a ",
                html.Strong("Almacén → Listados → Existencias")
            ]),
            html.Li([
                "Hacer clic en ",
                html.Strong("Exportar a Excel")
            ]),
            html.Li("Seleccionar campos de stock, precios y fechas"),
        ], className="mb-0")
    ])

    # Contenedor completo para Farmatic
    farmatic_instructions = html.Div([
        # Nota de detección automática
        dbc.Alert(
            html.Span([
                html.I(className="bi bi-magic me-2"),
                html.Strong("Detección automática: "),
                "El sistema detecta automáticamente si es un archivo de ventas o inventario."
            ]),
            color="info", className="mb-3 py-2"
        ),

        dbc.Row([
            dbc.Col([farmatic_sales], width=12, lg=6, className="mb-3 mb-lg-0"),
            dbc.Col([farmatic_inventory], width=12, lg=6),
        ])
    ])

    # Mensaje genérico si no hay ERP configurado
    generic_message = html.Div([
        dbc.Alert(
            html.Span([
                html.I(className="bi bi-magic me-2"),
                html.Strong("Detección automática: "),
                "Sube tu archivo y el sistema detectará automáticamente si es de ventas o inventario."
            ]),
            color="info", className="mb-3 py-2"
        ),
        html.P([
            html.I(className="bi bi-info-circle me-2 text-info"),
            "Contacta con soporte para configurar tu sistema ERP y ver las instrucciones específicas de exportación."
        ], className="text-muted mb-0")
    ])

    # Seleccionar instrucciones según el ERP del usuario
    if user_erp_type == "farmanager":
        return farmanager_instructions
    elif user_erp_type == "farmatic":
        return farmatic_instructions
    else:
        return generic_message


def create_erp_help_accordion(user_erp_type: Optional[str] = None) -> html.Div:
    """
    DEPRECATED: Usar create_erp_help_section en su lugar.
    Mantenido por compatibilidad - redirige a la nueva función.
    """
    return create_erp_help_section(user_erp_type)
