"""
Sistema de gestión de notificaciones toast con cola y auto-dismiss.
Proporciona feedback visual no intrusivo para operaciones del sistema.
"""

import uuid
from datetime import datetime
from typing import Dict, Literal, Optional

import dash_bootstrap_components as dbc
from dash import ALL, Input, Output, State, dcc, html

ToastType = Literal["success", "warning", "error", "info"]


class ToastManager:
    """Gestor centralizado de notificaciones toast."""

    MAX_TOASTS = 3
    AUTO_DISMISS_DELAY = {"success": 4000, "info": 5000, "warning": 6000, "error": 8000}

    TOAST_ICONS = {
        "success": "fas fa-check-circle",
        "warning": "fas fa-exclamation-triangle",
        "error": "fas fa-times-circle",
        "info": "fas fa-info-circle",
    }

    TOAST_COLORS = {"success": "success", "warning": "warning", "error": "danger", "info": "info"}


def create_toast_container() -> html.Div:
    """
    Crea el contenedor principal para las notificaciones toast.
    Debe añadirse una sola vez en el layout principal.
    """
    return html.Div(
        id="toast-container",
        style={
            "position": "fixed",
            "top": "80px",
            "right": "20px",
            "zIndex": 9999,
            "maxWidth": "400px",
            "width": "100%",
            "pointerEvents": "none",
        },
        children=[
            dcc.Store(id="toast-queue-store", data=[]),
            dcc.Store(id="toast-trigger-store", data={}),
            dcc.Store(id="toast-removal-store", data={}),  # Store intermedio para remociones
            html.Div(id="toast-stack"),
        ],
    )


def create_toast(
    message: str,
    toast_type: ToastType = "info",
    title: Optional[str] = None,
    toast_id: Optional[str] = None,
    dismissible: bool = True,
    auto_dismiss: bool = True,
) -> dbc.Alert:
    """
    Crea una notificación toast individual.

    Args:
        message: Mensaje principal del toast
        toast_type: Tipo de notificación (success, warning, error, info)
        title: Título opcional para el toast
        toast_id: ID único del toast (se genera automáticamente si no se proporciona)
        dismissible: Si el toast puede cerrarse manualmente
        auto_dismiss: Si el toast se cierra automáticamente
    """
    if not toast_id:
        toast_id = str(uuid.uuid4())

    icon = ToastManager.TOAST_ICONS.get(toast_type, "fas fa-bell")
    color = ToastManager.TOAST_COLORS.get(toast_type, "secondary")

    content = []

    if title:
        content.append(
            html.Div([html.I(className=f"{icon} me-2"), html.Strong(title)], className="d-flex align-items-center mb-1")
        )
    else:
        content.append(html.I(className=f"{icon} me-2"))

    content.append(html.Span(message))

    toast = dbc.Alert(
        children=html.Div(content),
        color=color,
        dismissable=dismissible,  # Corregido: dismissable en lugar de dismissible
        is_open=True,
        duration=ToastManager.AUTO_DISMISS_DELAY[toast_type] if auto_dismiss else None,
        id={"type": "toast-item", "index": toast_id},
        className="shadow-sm border-0 mb-2 toast-notification",
        style={"pointerEvents": "auto", "animation": "slideInRight 0.3s ease-out", "minHeight": "60px"},
    )

    return toast


def trigger_toast(message: str, toast_type: ToastType = "info", title: Optional[str] = None) -> Dict:
    """
    Crea un diccionario de datos para disparar un nuevo toast.
    Usar con toast-trigger-store para añadir notificaciones.

    Ejemplo:
        return {"toast-trigger-store": trigger_toast("Operación completada", "success")}
    """
    return {
        "id": str(uuid.uuid4()),
        "message": message,
        "type": toast_type,
        "title": title,
        "timestamp": datetime.now().isoformat(),
    }


# Callbacks para gestionar la cola de toasts
def register_toast_callbacks(app):
    """
    Registra los callbacks necesarios para el sistema de toasts.

    Cumple REGLA #11: ONE INPUT ONE CALLBACK
    - Usa Store intermedio (toast-removal-store) para desacoplar flujos de datos
    - toast-queue-store solo tiene UN callback que lo escribe
    """

    @app.callback(
        Output("toast-removal-store", "data"),
        Input({"type": "toast-item", "index": ALL}, "is_open"),
        prevent_initial_call=True,
    )
    def detect_closed_toasts(toast_states):
        """
        Detecta toasts cerrados y almacena en Store intermedio.

        Flujo: toast-item.is_open → toast-removal-store
        """
        if not toast_states:
            return {}

        # Identificar qué toasts están cerrados (is_open=False)
        closed_indices = [i for i, is_open in enumerate(toast_states) if not is_open]

        return {"closed_indices": closed_indices, "timestamp": datetime.now().isoformat()}

    @app.callback(
        Output("toast-queue-store", "data"),
        [Input("toast-trigger-store", "data"), Input("toast-removal-store", "data")],
        State("toast-queue-store", "data"),
        prevent_initial_call=True,
    )
    def update_toast_queue_consolidated(new_toast, removal_data, current_queue):
        """
        Callback consolidado que maneja TODAS las escrituras a toast-queue-store.

        Cumple REGLA #11: ONE INPUT ONE CALLBACK
        - Múltiples Inputs: nuevo toast, remociones
        - Un solo Output: toast-queue-store

        Flujos:
        1. toast-trigger-store → Añadir nuevo toast
        2. toast-removal-store → Remover toasts cerrados
        """
        from dash import ctx

        if not ctx.triggered:
            return current_queue or []

        trigger_id = ctx.triggered[0]["prop_id"]

        if not current_queue:
            current_queue = []

        # CASO 1: Nuevo toast disparado
        if "toast-trigger-store" in trigger_id:
            if not new_toast:
                return current_queue

            # Añadir nuevo toast
            current_queue.append(new_toast)

            # Limitar a MAX_TOASTS
            if len(current_queue) > ToastManager.MAX_TOASTS:
                current_queue = current_queue[-ToastManager.MAX_TOASTS :]

            return current_queue

        # CASO 2: Remover toasts cerrados
        elif "toast-removal-store" in trigger_id:
            if not removal_data or not removal_data.get("closed_indices"):
                return current_queue

            closed_indices = removal_data["closed_indices"]

            # Filtrar toasts que NO están en closed_indices
            filtered_queue = [toast for i, toast in enumerate(current_queue) if i not in closed_indices]

            return filtered_queue

        return current_queue

    @app.callback(Output("toast-stack", "children"), Input("toast-queue-store", "data"), prevent_initial_call=True)
    def render_toast_stack(queue):
        """Renderiza la pila de toasts desde la cola."""
        if not queue:
            return []

        toasts = []
        for toast_data in queue:
            toast = create_toast(
                message=toast_data.get("message", ""),
                toast_type=toast_data.get("type", "info"),
                title=toast_data.get("title"),
                toast_id=toast_data.get("id"),
            )
            toasts.append(toast)

        return toasts


# Estilos CSS para las animaciones
TOAST_STYLES = """
<style>
@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes slideOutRight {
    from {
        transform: translateX(0);
        opacity: 1;
    }
    to {
        transform: translateX(100%);
        opacity: 0;
    }
}

.toast-notification {
    transition: all 0.3s ease-out;
}

.toast-notification:hover {
    transform: translateX(-5px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
}

/* Responsive para móvil */
@media (max-width: 576px) {
    #toast-container {
        right: 10px !important;
        left: 10px !important;
        max-width: none !important;
    }
}
</style>
"""


# Funciones helper para uso común
def success_toast(message: str, title: str = "Éxito") -> Dict:
    """Crea un toast de éxito."""
    return trigger_toast(message, "success", title)


def error_toast(message: str, title: str = "Error") -> Dict:
    """Crea un toast de error."""
    return trigger_toast(message, "error", title)


def warning_toast(message: str, title: str = "Advertencia") -> Dict:
    """Crea un toast de advertencia."""
    return trigger_toast(message, "warning", title)


def info_toast(message: str, title: str = "Información") -> Dict:
    """Crea un toast informativo."""
    return trigger_toast(message, "info", title)
