"""
Componente unificado de estado del sistema con múltiples modos de visualización.
Reemplaza gradualmente el sistema legacy mantiendo compatibilidad total.
"""

import logging
from datetime import datetime
from typing import Dict, List, Literal

import dash_bootstrap_components as dbc
from dash import Input, Output, State, dcc, html
from dash.exceptions import PreventUpdate
from utils.request_coordinator import request_coordinator

logger = logging.getLogger(__name__)

StatusLevel = Literal["basic", "detailed", "diagnostic"]


class SystemStatusUnified:
    """Gestor unificado de estado del sistema con feature flags."""

    # Feature flags para testing A/B
    FEATURE_FLAGS = {
        "unified_status_enabled": True,
        "detailed_mode_enabled": True,
        "diagnostic_mode_enabled": True,
        "legacy_fallback_enabled": True,
        "real_time_updates": True,
    }

    # Configuración por modo
    MODE_CONFIG = {
        "basic": {
            "title": "Estado del Sistema",
            "refresh_interval": 30000,  # 30 segundos
            "show_details": False,
            "show_diagnostics": False,
            "target_users": ["farmacéuticos", "técnicos"],
        },
        "detailed": {
            "title": "Estado Detallado del Sistema",
            "refresh_interval": 15000,  # 15 segundos
            "show_details": True,
            "show_diagnostics": False,
            "target_users": ["administradores", "gerentes"],
        },
        "diagnostic": {
            "title": "Diagnóstico Completo del Sistema",
            "refresh_interval": 10000,  # 10 segundos
            "show_details": True,
            "show_diagnostics": True,
            "target_users": ["desarrolladores", "soporte técnico"],
        },
    }

    # Iconos y colores por estado
    STATUS_ICONS = {
        "healthy": {"icon": "fas fa-check-circle", "color": "success"},
        "warning": {"icon": "fas fa-exclamation-triangle", "color": "warning"},
        "error": {"icon": "fas fa-times-circle", "color": "danger"},
        "unknown": {"icon": "fas fa-question-circle", "color": "secondary"},
        "loading": {"icon": "fas fa-spinner fa-spin", "color": "info"},
    }


def create_unified_status_panel(mode: StatusLevel = "basic", panel_id: str = "unified-status-panel") -> html.Div:
    """
    Crea el panel unificado de estado según el modo especificado.

    Args:
        mode: Nivel de detalle (basic, detailed, diagnostic)
        panel_id: ID único del panel
    """
    config = SystemStatusUnified.MODE_CONFIG.get(mode, SystemStatusUnified.MODE_CONFIG["basic"])

    # Header del panel
    header = html.Div(
        [
            html.H5([html.I(className="fas fa-heartbeat me-2 text-primary"), config["title"]], className="mb-0"),
            html.Small(f"Actualización automática cada {config['refresh_interval']//1000}s", className="text-muted"),
        ],
        className="d-flex justify-content-between align-items-center mb-3",
    )

    # Contenido principal
    content = html.Div(
        [
            # Stores para datos
            dcc.Store(id=f"{panel_id}-data-store", data={}),
            dcc.Store(
                id=f"{panel_id}-config-store",
                data={
                    "mode": mode,
                    "refresh_interval": config["refresh_interval"],
                    "feature_flags": SystemStatusUnified.FEATURE_FLAGS,
                },
            ),
            # Interval para actualizaciones automáticas
            dcc.Interval(
                id=f"{panel_id}-interval",
                interval=config["refresh_interval"],
                n_intervals=0,
                disabled=not SystemStatusUnified.FEATURE_FLAGS["real_time_updates"],
            ),
            # Indicador de carga
            dbc.Spinner(html.Div(id=f"{panel_id}-content"), id=f"{panel_id}-spinner", color="primary", size="sm"),
        ]
    )

    return dbc.Card([dbc.CardBody([header, content])], className="shadow-sm")


def transform_legacy_to_unified(legacy_data: Dict) -> Dict:
    """
    Transforma datos del formato legacy (/api/v1/system/status)
    al formato unificado esperado por los componentes.
    """
    try:
        # Estado general basado en componentes
        overall_status = "healthy"
        overall_status_text = "Operativo"

        # Verificar estado de servicios críticos
        critical_services = {}

        # Backend (siempre healthy si recibimos respuesta)
        critical_services["backend"] = {
            "status": "healthy",
            "display_name": "Servidor Backend",
            "status_text": "Operativo",
        }

        # Base de datos
        db_status = "healthy" if legacy_data.get("database", {}).get("status") == "connected" else "error"
        critical_services["database"] = {
            "status": db_status,
            "display_name": "Base de Datos",
            "status_text": "Conectado" if db_status == "healthy" else "Error",
        }

        # Cache (Redis)
        cache_status = "healthy" if legacy_data.get("cache", {}).get("status") == "connected" else "warning"
        critical_services["cache"] = {
            "status": cache_status,
            "display_name": "Sistema de Cache",
            "status_text": "Activo" if cache_status == "healthy" else "Limitado",
        }

        # Determinar estado general
        if any(service["status"] == "error" for service in critical_services.values()):
            overall_status = "error"
            overall_status_text = "Error Crítico"
        elif any(service["status"] == "warning" for service in critical_services.values()):
            overall_status = "warning"
            overall_status_text = "Funcionamiento Limitado"

        # Métricas de rendimiento (simplificadas para modo básico)
        performance_metrics = {
            "system_resources": {"cpu_usage": 0, "memory_usage": 0},  # No disponible en legacy
            "database": {
                "active_connections": legacy_data.get("database", {}).get("active_connections", 0),
                "queries_per_second": 0,
            },
            "cache": {"hit_rate": 95.0, "memory_usage": 0},  # Estimación
            "external_apis": {"cima_response_time": 500, "apis_available": 2, "total_apis": 2},  # Estimación
        }

        return {
            "overall_status": overall_status,
            "overall_status_text": overall_status_text,
            "critical_services": critical_services,
            "performance_metrics": performance_metrics,
            "diagnostics": {
                "system_info": {
                    "fastapi_version": "N/A",
                    "python_version": "N/A",
                    "postgres_version": "N/A",
                    "uptime": "N/A",
                },
                "recent_errors": [],
            },
        }

    except Exception as e:
        # Fallback seguro
        return {
            "overall_status": "error",
            "overall_status_text": f"Error procesando datos: {str(e)}",
            "critical_services": {},
            "performance_metrics": {},
            "diagnostics": {},
        }


def create_status_summary_basic(data: Dict) -> List[html.Div]:
    """Crea resumen básico para farmacéuticos."""
    if not data:
        return [create_loading_placeholder()]

    # KPIs principales
    main_kpis = []

    # Estado general
    overall_status = data.get("overall_status", "unknown")
    status_config = SystemStatusUnified.STATUS_ICONS.get(overall_status)

    main_kpis.append(
        dbc.Col(
            [
                dbc.Card(
                    [
                        dbc.CardBody(
                            [
                                html.Div(
                                    [
                                        html.I(
                                            className=f"{status_config['icon']} fa-2x text-{status_config['color']}"
                                        ),
                                        html.Div(
                                            [
                                                html.H6("Estado General", className="text-muted mb-1"),
                                                html.H5(
                                                    data.get("overall_status_text", "Verificando..."),
                                                    className=f"text-{status_config['color']} mb-0",
                                                ),
                                            ],
                                            className="ms-3",
                                        ),
                                    ],
                                    className="d-flex align-items-center",
                                )
                            ]
                        )
                    ],
                    className="border-0 bg-light",
                )
            ],
            width=12,
            lg=6,
            className="mb-3",
        )
    )

    # Servicios críticos
    critical_services = data.get("critical_services", {})
    if critical_services:
        service_items = []
        for service_name, service_data in critical_services.items():
            status = service_data.get("status", "unknown")
            status_config = SystemStatusUnified.STATUS_ICONS.get(status)

            service_items.append(
                html.Li(
                    [
                        html.I(className=f"{status_config['icon']} me-2 text-{status_config['color']}"),
                        html.Span(service_data.get("display_name", service_name)),
                        html.Span(service_data.get("status_text", ""), className="text-muted ms-2 small"),
                    ],
                    className="list-group-item border-0 py-2",
                )
            )

        main_kpis.append(
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardHeader([html.I(className="fas fa-server me-2"), "Servicios Críticos"]),
                            dbc.CardBody(
                                [html.Ul(service_items, className="list-group list-group-flush")], className="p-0"
                            ),
                        ]
                    )
                ],
                width=12,
                lg=6,
                className="mb-3",
            )
        )

    return [dbc.Row(main_kpis)]


def create_status_summary_detailed(data: Dict) -> List[html.Div]:
    """Crea resumen detallado para administradores."""
    if not data:
        return [create_loading_placeholder()]

    components = []

    # Incluir resumen básico
    components.extend(create_status_summary_basic(data))

    # Métricas de rendimiento
    performance = data.get("performance_metrics", {})
    if performance:
        perf_cards = []

        # CPU y Memoria
        if "system_resources" in performance:
            resources = performance["system_resources"]
            perf_cards.append(
                dbc.Col(
                    [
                        dbc.Card(
                            [
                                dbc.CardBody(
                                    [
                                        html.H6([html.I(className="fas fa-microchip me-2"), "Recursos del Sistema"]),
                                        html.Div(
                                            [
                                                html.Div(
                                                    [
                                                        html.Small("CPU:", className="text-muted"),
                                                        html.Span(
                                                            f" {resources.get('cpu_usage', 0):.1f}%",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                                html.Div(
                                                    [
                                                        html.Small("Memoria:", className="text-muted"),
                                                        html.Span(
                                                            f" {resources.get('memory_usage', 0):.1f}%",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                            ]
                                        ),
                                    ]
                                )
                            ]
                        )
                    ],
                    width=12,
                    md=6,
                    lg=3,
                    className="mb-3",
                )
            )

        # Base de datos
        if "database" in performance:
            db_data = performance["database"]
            perf_cards.append(
                dbc.Col(
                    [
                        dbc.Card(
                            [
                                dbc.CardBody(
                                    [
                                        html.H6([html.I(className="fas fa-database me-2"), "Base de Datos"]),
                                        html.Div(
                                            [
                                                html.Div(
                                                    [
                                                        html.Small("Conexiones:", className="text-muted"),
                                                        html.Span(
                                                            f" {db_data.get('active_connections', 0)}",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                                html.Div(
                                                    [
                                                        html.Small("Consultas/seg:", className="text-muted"),
                                                        html.Span(
                                                            f" {db_data.get('queries_per_second', 0):.1f}",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                            ]
                                        ),
                                    ]
                                )
                            ]
                        )
                    ],
                    width=12,
                    md=6,
                    lg=3,
                    className="mb-3",
                )
            )

        # Cache (Redis)
        if "cache" in performance:
            cache_data = performance["cache"]
            perf_cards.append(
                dbc.Col(
                    [
                        dbc.Card(
                            [
                                dbc.CardBody(
                                    [
                                        html.H6([html.I(className="fas fa-bolt me-2"), "Cache"]),
                                        html.Div(
                                            [
                                                html.Div(
                                                    [
                                                        html.Small("Hit Rate:", className="text-muted"),
                                                        html.Span(
                                                            f" {cache_data.get('hit_rate', 0):.1f}%",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                                html.Div(
                                                    [
                                                        html.Small("Memoria:", className="text-muted"),
                                                        html.Span(
                                                            f" {cache_data.get('memory_usage', 0):.1f} MB",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                            ]
                                        ),
                                    ]
                                )
                            ]
                        )
                    ],
                    width=12,
                    md=6,
                    lg=3,
                    className="mb-3",
                )
            )

        # APIs externas
        if "external_apis" in performance:
            api_data = performance["external_apis"]
            perf_cards.append(
                dbc.Col(
                    [
                        dbc.Card(
                            [
                                dbc.CardBody(
                                    [
                                        html.H6([html.I(className="fas fa-cloud me-2"), "APIs Externas"]),
                                        html.Div(
                                            [
                                                html.Div(
                                                    [
                                                        html.Small("CIMA:", className="text-muted"),
                                                        html.Span(
                                                            f" {api_data.get('cima_response_time', 0):.0f}ms",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                                html.Div(
                                                    [
                                                        html.Small("Disponibilidad:", className="text-muted"),
                                                        html.Span(
                                                            f" {api_data.get('apis_available', 0)}/{api_data.get('total_apis', 0)}",
                                                            className="fw-bold",
                                                        ),
                                                    ]
                                                ),
                                            ]
                                        ),
                                    ]
                                )
                            ]
                        )
                    ],
                    width=12,
                    md=6,
                    lg=3,
                    className="mb-3",
                )
            )

        if perf_cards:
            components.append(html.Hr())
            components.append(html.H6("Métricas de Rendimiento"))
            components.append(dbc.Row(perf_cards))

    return components


def create_status_summary_diagnostic(data: Dict) -> List[html.Div]:
    """Crea resumen completo para desarrolladores."""
    if not data:
        return [create_loading_placeholder()]

    components = []

    # Incluir resumen detallado
    components.extend(create_status_summary_detailed(data))

    # Información de diagnóstico técnico
    diagnostics = data.get("diagnostics", {})
    if diagnostics:
        components.append(html.Hr())
        components.append(html.H6("Información Técnica"))

        # Versiones y configuración
        if "system_info" in diagnostics:
            sys_info = diagnostics["system_info"]

            tech_info = dbc.Card(
                [
                    dbc.CardHeader("Información del Sistema"),
                    dbc.CardBody(
                        [
                            dbc.Row(
                                [
                                    dbc.Col(
                                        [
                                            html.Strong("Backend:"),
                                            html.Br(),
                                            html.Small(
                                                f"FastAPI {sys_info.get('fastapi_version', 'N/A')}",
                                                className="text-muted",
                                            ),
                                            html.Br(),
                                            html.Small(
                                                f"Python {sys_info.get('python_version', 'N/A')}",
                                                className="text-muted",
                                            ),
                                        ],
                                        width=6,
                                    ),
                                    dbc.Col(
                                        [
                                            html.Strong("Base de Datos:"),
                                            html.Br(),
                                            html.Small(
                                                f"PostgreSQL {sys_info.get('postgres_version', 'N/A')}",
                                                className="text-muted",
                                            ),
                                            html.Br(),
                                            html.Small(
                                                f"Tiempo activo: {sys_info.get('uptime', 'N/A')}",
                                                className="text-muted",
                                            ),
                                        ],
                                        width=6,
                                    ),
                                ]
                            )
                        ]
                    ),
                ]
            )

            components.append(tech_info)

        # Logs recientes críticos
        if "recent_errors" in diagnostics:
            errors = diagnostics["recent_errors"]
            if errors:
                error_items = []
                for error in errors[:5]:  # Últimos 5 errores
                    error_items.append(
                        html.Li(
                            [
                                html.Code(error.get("timestamp", ""), className="text-muted small"),
                                html.Br(),
                                html.Span(error.get("message", ""), className="text-danger small"),
                            ],
                            className="list-group-item",
                        )
                    )

                error_log = dbc.Card(
                    [
                        dbc.CardHeader(
                            [html.I(className="fas fa-exclamation-triangle me-2 text-warning"), "Errores Recientes"]
                        ),
                        dbc.CardBody([html.Ul(error_items, className="list-group list-group-flush")], className="p-0"),
                    ],
                    className="mt-3",
                )

                components.append(error_log)

    return components


def create_loading_placeholder() -> html.Div:
    """Crea placeholder de carga."""
    return html.Div(
        [
            dbc.Placeholder(animation="wave", className="mb-2"),
            dbc.Placeholder(animation="wave", size="lg"),
            dbc.Placeholder(animation="wave", size="sm"),
        ],
        className="text-center p-4",
    )


# Callbacks del componente
def register_unified_status_callbacks(app):
    """Registra los callbacks para el sistema unificado de estado."""

    # Callback específico para el panel de homepage
    @app.callback(
        Output("homepage-unified-status-data-store", "data"),
        Input("homepage-unified-status-interval", "n_intervals"),
        [State("homepage-unified-status-config-store", "data"), State("auth-state", "data")],  # Issue #404
        prevent_initial_call=False,
    )
    def update_homepage_unified_status_data(n_intervals, config, auth_state):  # Issue #404: Agregar parámetro
        """
        Actualiza los datos del estado unificado desde la API para homepage.

        Issue #404 (DASH006): Verificación proactiva de autenticación antes de API calls.
        """
        # Issue #404: Auth guard BEFORE any logic
        from utils.auth_helpers import is_user_authenticated

        if not is_user_authenticated(auth_state):
            logger.debug("[update_homepage_unified_status_data] User not authenticated - skipping API calls")
            raise PreventUpdate
        if not config or not config.get("feature_flags", {}).get("unified_status_panel", {}).get("enabled", False):
            return {}

        try:
            mode = config.get("mode", "basic")

            # Llamar al endpoint de sistema (ya existente)
            # Usar request_coordinator con bypass de autenticación
            data = request_coordinator.make_request(
                "/api/v1/system/status", params={"include_details": "true"}, cache_ttl=30  # Cache por 30 segundos
            )

            if data:
                # Transformar datos legacy a formato unificado
                unified_data = transform_legacy_to_unified(data)
                unified_data["last_updated"] = datetime.now().isoformat()
                return unified_data
            else:
                # Fallback a datos básicos en caso de error
                return {
                    "overall_status": "warning",
                    "overall_status_text": "Conectando...",
                    "last_updated": datetime.now().isoformat(),
                    "error": "API no disponible",
                }

        except Exception as e:
            # Manejar errores de conexión
            return {
                "overall_status": "error",
                "overall_status_text": "Error de conexión",
                "last_updated": datetime.now().isoformat(),
                "error": str(e),
            }

    @app.callback(
        Output("homepage-unified-status-content", "children"),
        Input("homepage-unified-status-data-store", "data"),
        State("homepage-unified-status-config-store", "data"),
        prevent_initial_call=False,
    )
    def render_homepage_unified_status_content(data, config):
        """Renderiza el contenido del estado según el modo configurado para homepage."""
        if not config:
            return []

        mode = config.get("mode", "basic")

        try:
            if mode == "basic":
                content = create_status_summary_basic(data)
            elif mode == "detailed":
                content = create_status_summary_detailed(data)
            elif mode == "diagnostic":
                content = create_status_summary_diagnostic(data)
            else:
                content = create_status_summary_basic(data)

            # Añadir timestamp de última actualización
            if data and data.get("last_updated"):
                last_update = html.Small(
                    [
                        "Última actualización: ",
                        html.Span(
                            datetime.fromisoformat(data["last_updated"]).strftime("%H:%M:%S"), className="text-primary"
                        ),
                    ],
                    className="text-muted d-block text-end mt-2",
                )
                content.append(last_update)

            return content

        except Exception as e:
            # Fallback a mensaje de error
            error_content = dbc.Alert(
                [html.I(className="fas fa-exclamation-triangle me-2"), f"Error al cargar el estado: {str(e)}"],
                color="warning",
            )
            return [error_content]


# Funciones de utilidad para integration con el sistema existente
def get_unified_status_component(mode: StatusLevel = "basic", component_id: str = "main-status") -> html.Div:
    """
    Función principal para obtener el componente de estado unificado.

    Args:
        mode: Modo de visualización
        component_id: ID único del componente
    """
    # Verificar feature flag
    if not SystemStatusUnified.FEATURE_FLAGS.get("unified_status_enabled", False):
        return html.Div("Componente de estado unificado deshabilitado")

    return create_unified_status_panel(mode, component_id)


def create_feature_flag_store() -> dcc.Store:
    """
    Crea store global para feature flags del sistema unificado.
    """
    return dcc.Store(id="unified-status-feature-flags", data=SystemStatusUnified.FEATURE_FLAGS)


# Función de utilidad para simplificar el acceso desde homepage
def create_homepage_unified_status() -> html.Div:
    """
    Versión específica para homepage con IDs únicos.
    """
    return create_unified_status_panel(mode="basic", panel_id="homepage-unified-status")
