"""
Dashboard de Salud del Sistema con notificaciones inteligentes.
FASE 3: Mejoras UX Honesta - Sistema completo de monitoreo para farmacéuticos.
"""

from typing import Dict, List

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from components.admin_feedback_components import (
    create_friendly_error_alert,
    create_metric_card_with_loading,
    create_status_badge_with_tooltip,
    get_metric_display_value,
)
from components.skeleton_loader import create_skeleton_loader
from dash import Input, Output, callback, ctx, dcc, html
from utils.helpers import format_compact_number
from utils.request_coordinator import request_coordinator


def create_health_overview_card() -> dbc.Card:
    """
    Crea la tarjeta principal del dashboard de salud con estados UX mejorados.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H4(
                        [html.I(className="fas fa-heartbeat me-2 text-success"), "Estado de Salud del Sistema"],
                        className="mb-0",
                    ),
                    html.Div(
                        id="system-status-badge-container",
                        children=create_status_badge_with_tooltip(
                            "loading", "Verificando estado del sistema...", "system-status-badge"
                        ),
                    ),
                ],
                className="d-flex justify-content-between align-items-center",
            ),
            dbc.CardBody(
                [
                    # Métricas principales con loading states
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    create_metric_card_with_loading(
                                        title="Uso CPU",
                                        metric_id="cpu-usage-metric",
                                        default_value="⏳",
                                        tooltip_text="Porcentaje de CPU utilizado por el sistema",
                                        color_class="text-primary",
                                    )
                                ],
                                width=3,
                            ),
                            dbc.Col(
                                [
                                    create_metric_card_with_loading(
                                        title="Memoria Usada",
                                        metric_id="memory-usage-metric",
                                        default_value="⏳",
                                        tooltip_text="Memoria RAM utilizada por la aplicación",
                                        color_class="text-info",
                                    )
                                ],
                                width=3,
                            ),
                            dbc.Col(
                                [
                                    create_metric_card_with_loading(
                                        title="Productos Catálogo",
                                        metric_id="catalog-products-metric",
                                        default_value="⏳",
                                        tooltip_text="Número total de productos en el catálogo",
                                        color_class="text-success",
                                    )
                                ],
                                width=3,
                            ),
                            dbc.Col(
                                [
                                    create_metric_card_with_loading(
                                        title="Última Sync",
                                        metric_id="catalog-freshness-metric",
                                        default_value="⏳",
                                        tooltip_text="Tiempo transcurrido desde la última sincronización",
                                        color_class="text-warning",
                                    )
                                ],
                                width=3,
                            ),
                        ],
                        className="mb-4",
                    ),
                    # Contenedor para gráfico con loading state
                    html.Div(
                        [
                            # Loading skeleton que se oculta cuando cargan datos
                            html.Div(
                                create_skeleton_loader(rows=5, height="30px"),
                                id="health-trends-loading",
                                style={"display": "block"},
                            ),
                            # Gráfico que se muestra cuando cargan datos
                            dcc.Graph(
                                id="health-trends-chart",
                                figure=create_empty_trends_chart(),
                                config={"displayModeBar": False},
                                style={"height": "300px", "display": "none"},
                            ),
                        ]
                    ),
                ]
            ),
            dbc.CardFooter(
                [
                    dbc.ButtonGroup(
                        [
                            dbc.Button(
                                html.Div(
                                    [
                                        html.I(className="fas fa-sync me-1", id="refresh-health-icon"),
                                        html.Span("Actualizar", id="refresh-health-text"),
                                    ]
                                ),
                                id="refresh-health-button",
                                color="outline-primary",
                                size="sm",
                                disabled=False,
                            ),
                            dbc.Button(
                                html.Div([html.I(className="fas fa-camera me-1"), "Snapshot"]),
                                id="take-snapshot-button",
                                color="outline-info",
                                size="sm",
                            ),
                        ]
                    )
                ]
            ),
        ],
        className="h-100",
    )


def create_system_alerts_card() -> dbc.Card:
    """
    Crea la tarjeta de alertas del sistema.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H5(
                        html.Div(
                            [html.I(className="fas fa-exclamation-triangle me-2 text-warning"), "Alertas del Sistema"]
                        ),
                        className="mb-0",
                    ),
                    dbc.Badge("0", id="active-alerts-count", color="danger", pill=True),
                ],
                className="d-flex justify-content-between align-items-center",
            ),
            dbc.CardBody(
                [
                    html.Div(
                        id="system-alerts-list",
                        children=[
                            dbc.Alert(
                                html.Div(
                                    [
                                        html.I(className="fas fa-info-circle me-2"),
                                        "No hay alertas activas. Sistema funcionando correctamente.",
                                    ]
                                ),
                                color="success",
                                className="mb-0",
                            )
                        ],
                    )
                ],
                style={"maxHeight": "400px", "overflowY": "auto"},
            ),
            dbc.CardFooter(
                [
                    dbc.Button(
                        "Ver Todas las Alertas",
                        id="view-all-alerts-button",
                        color="outline-warning",
                        size="sm",
                        className="w-100",
                    )
                ]
            ),
        ],
        className="h-100",
    )


def create_performance_metrics_card() -> dbc.Card:
    """
    Métricas detalladas de performance.
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H5(
                        html.Div([html.I(className="fas fa-tachometer-alt me-2 text-info"), "Métricas de Performance"]),
                        className="mb-0",
                    )
                ]
            ),
            dbc.CardBody(
                [
                    # Métricas de base de datos
                    html.H6("Base de Datos", className="mb-2"),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Small("Conexiones Activas", className="text-muted d-block"),
                                    html.Strong("0", id="db-connections-metric"),
                                ],
                                width=6,
                            ),
                            dbc.Col(
                                [
                                    html.Small("Tiempo Respuesta", className="text-muted d-block"),
                                    html.Strong("0ms", id="db-response-time-metric"),
                                ],
                                width=6,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # APIs externas
                    html.H6("APIs Externas", className="mb-2"),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Small("CIMA API", className="text-muted d-block"),
                                    html.Strong("N/A", id="cima-api-metric"),
                                    dbc.Badge("", id="cima-api-status", className="ms-1"),
                                ],
                                width=6,
                            ),
                            dbc.Col(
                                [
                                    html.Small("Nomenclátor", className="text-muted d-block"),
                                    html.Strong("N/A", id="nomenclator-api-metric"),
                                    dbc.Badge("", id="nomenclator-api-status", className="ms-1"),
                                ],
                                width=6,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Cobertura del catálogo
                    html.H6("Cobertura Catálogo", className="mb-2"),
                    dcc.Graph(
                        id="catalog-coverage-chart",
                        figure=create_coverage_chart(),
                        config={"displayModeBar": False},
                        style={"height": "200px"},
                    ),
                ]
            ),
        ],
        className="h-100",
    )


def create_empty_trends_chart() -> go.Figure:
    """
    Crea un gráfico vacío para las tendencias.
    """
    fig = go.Figure()

    fig.add_trace(go.Scatter(x=[], y=[], mode="lines+markers", name="CPU %", line=dict(color="#007bff")))

    fig.add_trace(
        go.Scatter(x=[], y=[], mode="lines+markers", name="Memoria MB", yaxis="y2", line=dict(color="#17a2b8"))
    )

    fig.update_layout(
        title="Tendencias de Performance (24h)",
        xaxis_title="Tiempo",
        yaxis=dict(title="CPU %", side="left"),
        yaxis2=dict(title="Memoria (MB)", side="right", overlaying="y"),
        height=300,
        showlegend=True,
        legend=dict(x=0, y=1),
        margin=dict(l=40, r=40, t=40, b=40),
    )

    return fig


def create_coverage_chart() -> go.Figure:
    """
    Crea un gráfico de cobertura del catálogo.
    """
    fig = go.Figure(
        data=[
            go.Bar(
                x=["Con CIMA", "Con Nomenclátor", "Sin Datos"],
                y=[0, 0, 0],
                marker_color=["#28a745", "#17a2b8", "#dc3545"],
            )
        ]
    )

    fig.update_layout(
        title="Cobertura de Datos",
        yaxis_title="Número de Productos",
        height=200,
        margin=dict(l=40, r=40, t=40, b=40),
        showlegend=False,
    )

    return fig


def create_system_health_dashboard() -> html.Div:
    """
    Dashboard principal de salud del sistema.
    """
    return html.Div(
        [
            # Header con título
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.H2(
                                html.Div(
                                    [
                                        html.I(className="fas fa-monitor-waveform me-3 text-primary"),
                                        "Dashboard de Salud del Sistema",
                                    ]
                                ),
                                className="mb-1",
                            ),
                            html.P(
                                "Monitoreo en tiempo real para administradores de farmacia", className="text-muted mb-4"
                            ),
                        ]
                    )
                ]
            ),
            # Primera fila: Overview y Alertas
            dbc.Row(
                [dbc.Col([create_health_overview_card()], width=8), dbc.Col([create_system_alerts_card()], width=4)],
                className="mb-4",
            ),
            # Segunda fila: Performance (Centro de notificaciones eliminado - Fase 1)
            dbc.Row([dbc.Col([create_performance_metrics_card()], width=12)], className="mt-4"),
            # Stores para gestión de estado
            dcc.Store(id="health-dashboard-store", data={}),
            # Intervals para actualización
            dcc.Interval(id="health-dashboard-interval", interval=30000, n_intervals=0),  # 30 segundos
        ],
        id="system-health-dashboard",
    )


# ============================================
# CALLBACKS
# ============================================


@callback(
    [
        Output("system-status-badge-container", "children"),
        Output("cpu-usage-metric", "children"),
        Output("memory-usage-metric", "children"),
        Output("catalog-products-metric", "children"),
        Output("catalog-freshness-metric", "children"),
        Output("health-trends-chart", "figure"),
        Output("health-trends-chart", "style"),
        Output("health-trends-loading", "style"),
        Output("system-alerts-list", "children"),
        Output("active-alerts-count", "children"),
        Output("refresh-health-button", "children"),
        Output("refresh-health-button", "disabled"),
        Output("refresh-health-button", "color"),
        Output("health-dashboard-store", "data"),
    ],
    [Input("health-dashboard-interval", "n_intervals"), Input("refresh-health-button", "n_clicks")],
    State("auth-state", "data"),  # Issue #404: Agregar auth protection
    prevent_initial_call=False,
)
def update_health_dashboard(n_intervals, refresh_clicks, auth_state):  # Issue #404: Agregar parámetro
    """
    Actualiza los datos del dashboard de salud con UX mejorado.

    Issue #404 (DASH006): Verificación proactiva de autenticación antes de API calls.
    """
    # Issue #404: Auth guard BEFORE any logic
    from utils.auth_helpers import is_user_authenticated

    if not is_user_authenticated(auth_state):
        logger.debug("[update_health_dashboard] User not authenticated - skipping API calls")
        raise PreventUpdate
    # Determinar si esta actualización fue por click manual
    is_manual_refresh = ctx.triggered and ctx.triggered[0]["prop_id"] == "refresh-health-button.n_clicks"

    # Estados de loading durante refresh manual
    if is_manual_refresh:
        loading_button = html.Div([html.I(className="fas fa-spinner fa-spin me-1"), "Actualizando..."])
        loading_disabled = True
        loading_color = "outline-secondary"
    else:
        loading_button = html.Div([html.I(className="fas fa-sync me-1"), "Actualizar"])
        loading_disabled = False
        loading_color = "outline-primary"

    try:
        # Obtener datos del backend usando request_coordinator con bypass de auth
        response_data = request_coordinator.make_request(
            "/api/v1/system-unified/health-summary", params={"level": "detailed"}, cache_ttl=10  # Cache por 10 segundos
        )

        if response_data:
            data = response_data.get("data", {})

            # Estado general con badge mejorado
            overall_status = data.get("overall_status", "unknown")
            status_tooltips = {
                "healthy": "Todos los servicios funcionan correctamente",
                "warning": "Algunos servicios presentan advertencias menores",
                "critical": "Servicios críticos presentan problemas",
                "error": "Error grave en servicios principales",
            }

            status_map = {"healthy": "online", "warning": "warning", "critical": "offline", "error": "offline"}

            badge_status = status_map.get(overall_status, "unknown")
            tooltip_text = status_tooltips.get(overall_status, "Estado del sistema desconocido")

            status_badge = create_status_badge_with_tooltip(badge_status, tooltip_text, "system-status-badge")

            # Métricas actuales con mejor manejo
            metrics = data.get("metrics", {})
            components = data.get("components", {})
            database_component = components.get("database", {})

            cpu_percent = metrics.get("cpu_usage_percent")
            memory_mb = metrics.get("memory_usage_mb")
            catalog_products = database_component.get("catalog_products")
            freshness_hours = None  # No disponible en nueva estructura

            # Usar funciones de formato mejoradas
            cpu_text = get_metric_display_value(
                cpu_percent, format_func=lambda x: f"{x:.1f}%" if x is not None else None
            )
            memory_text = get_metric_display_value(
                memory_mb, format_func=lambda x: f"{format_compact_number(x)} MB" if x else None
            )
            products_text = get_metric_display_value(
                catalog_products, format_func=lambda x: format_compact_number(x) if x else "0"
            )
            freshness_text = get_metric_display_value(
                freshness_hours, format_func=lambda x: f"{x:.1f}h" if x is not None else None
            )

            # Gráfico de tendencias (simulado por ahora - nuevas API no incluye históricos)
            trends = {}  # La nueva API no incluye trends_24h
            trends_figure = create_trends_chart(trends)

            # Estados de visualización del gráfico
            chart_style = {"height": "300px", "display": "block"}
            loading_style = {"display": "none"}

            # Alertas con mejor UX (generadas desde componentes con error)
            alerts = []
            for component_name, component_data in components.items():
                if component_data.get("status") == "error":
                    alerts.append(
                        {
                            "title": f"Error en {component_name}",
                            "message": component_data.get("error", "Error desconocido"),
                            "severity": "high",
                            "count": 1,
                            "last_occurred": component_data.get("last_check", "Desconocido"),
                        }
                    )
            alerts_components = create_alerts_list(alerts)
            alerts_count = len(alerts)

            return (
                status_badge,
                cpu_text,
                memory_text,
                products_text,
                freshness_text,
                trends_figure,
                chart_style,
                loading_style,
                alerts_components,
                str(alerts_count),
                loading_button,
                loading_disabled,
                loading_color,
                response_data,
            )

        elif response.status_code == 404:
            # Servicio no encontrado - mensaje friendly
            error_alert = create_friendly_error_alert(
                "not_found",
                f"Endpoint: {backend_url}/api/v1/system-unified/health-summary?level=detailed",
                [
                    "El servicio de salud del sistema no está configurado",
                    "Verificar que el backend esté actualizado",
                    "Esta funcionalidad puede no estar disponible en esta versión",
                ],
            )

            status_badge = create_status_badge_with_tooltip(
                "offline", "Servicio de monitoreo no disponible", "system-status-badge"
            )

            return (
                status_badge,
                "❌ No disponible",
                "❌ No disponible",
                "❌ No disponible",
                "❌ No disponible",
                create_empty_trends_chart(),
                {"height": "300px", "display": "block"},
                {"display": "none"},
                [error_alert],
                "0",
                loading_button,
                loading_disabled,
                loading_color,
                {},
            )

        else:
            # Error al obtener datos (request_coordinator retornó None)
            error_alert = create_friendly_error_alert(
                "server",
                "Error al conectar con el servidor",
                [
                    "No se pudo obtener respuesta del servidor",
                    "Verificar logs del backend para más detalles",
                    "Puede ser un problema de autenticación (401) o servidor (500)",
                ],
            )

            status_badge = create_status_badge_with_tooltip(
                "offline", "Error del servidor de monitoreo", "system-status-badge"
            )

            return (
                status_badge,
                "🔄 Reintentando...",
                "🔄 Reintentando...",
                "🔄 Reintentando...",
                "🔄 Reintentando...",
                create_empty_trends_chart(),
                {"height": "300px", "display": "block"},
                {"display": "none"},
                [error_alert],
                "1",
                loading_button,
                loading_disabled,
                loading_color,
                {},
            )

    except Exception as e:
        # Error genérico
        error_alert = create_friendly_error_alert(
            "server",
            str(e),
            [
                "Error inesperado al obtener los datos",
                "Reintentar en unos minutos",
                "Si persiste, contactar soporte técnico",
            ],
        )

        status_badge = create_status_badge_with_tooltip("unknown", f"Error: {str(e)[:50]}...", "system-status-badge")

        return (
            status_badge,
            "❓ Error desconocido",
            "❓ Error desconocido",
            "❓ Error desconocido",
            "❓ Error desconocido",
            create_empty_trends_chart(),
            {"height": "300px", "display": "block"},
            {"display": "none"},
            [error_alert],
            "1",
            loading_button,
            loading_disabled,
            loading_color,
            {},
        )


def create_trends_chart(trends_data: Dict) -> go.Figure:
    """
    Crea el gráfico de tendencias con datos reales.
    """
    timestamps = trends_data.get("timestamps", [])
    cpu_usage = trends_data.get("cpu_usage", [])
    memory_usage = trends_data.get("memory_usage", [])

    fig = go.Figure()

    if timestamps:
        fig.add_trace(
            go.Scatter(
                x=timestamps,
                y=cpu_usage,
                mode="lines+markers",
                name="CPU %",
                line=dict(color="#007bff", width=2),
                marker=dict(size=4),
            )
        )

        # Convertir MB a GB para mejor visualización
        memory_gb = [mb / 1024 if mb else 0 for mb in memory_usage]

        fig.add_trace(
            go.Scatter(
                x=timestamps,
                y=memory_gb,
                mode="lines+markers",
                name="Memoria GB",
                yaxis="y2",
                line=dict(color="#17a2b8", width=2),
                marker=dict(size=4),
            )
        )

    fig.update_layout(
        title="Tendencias de Performance (24h)",
        xaxis_title="Tiempo",
        yaxis=dict(title="CPU %", side="left", range=[0, 100]),
        yaxis2=dict(title="Memoria (GB)", side="right", overlaying="y"),
        height=300,
        showlegend=True,
        legend=dict(x=0, y=1),
        margin=dict(l=40, r=40, t=40, b=40),
        hovermode="x unified",
    )

    return fig


def create_alerts_list(alerts: List[Dict]) -> List:
    """
    Crea la lista de alertas del sistema.
    """
    if not alerts:
        return [
            dbc.Alert(
                html.Div(
                    [
                        html.I(className="fas fa-check-circle me-2"),
                        "No hay alertas activas. Sistema funcionando correctamente.",
                    ]
                ),
                color="success",
                className="mb-0",
            )
        ]

    alert_components = []
    for alert in alerts[:5]:  # Mostrar solo las 5 más recientes
        severity_color = {"critical": "danger", "high": "danger", "medium": "warning", "low": "info"}.get(
            alert["severity"], "secondary"
        )

        alert_components.append(
            dbc.Alert(
                [
                    html.Div(
                        [
                            html.Strong(alert["title"], className="d-block"),
                            html.Small(alert["message"], className="text-muted"),
                            html.Br(),
                            html.Small(
                                [f"Ocurrencias: {alert['count']} | ", f"Última: {alert['last_occurred'][:16]}"],
                                className="text-muted",
                            ),
                        ]
                    )
                ],
                color=severity_color,
                className="mb-2",
            )
        )

    return alert_components
