"""
Componentes de skeleton loader para mejorar la experiencia de carga.
Parte del Issue #15 - Mejoras UX del sistema de catálogo.
"""


import dash_bootstrap_components as dbc
from dash import html


def create_skeleton_loader(
    lines: int = 3, show_title: bool = True, show_button: bool = False, className: str = ""
) -> html.Div:
    """
    Crea un skeleton loader básico para estados de carga.

    Args:
        lines: Número de líneas de texto skeleton
        show_title: Si mostrar un título skeleton
        show_button: Si mostrar un botón skeleton
        className: Clases CSS adicionales

    Returns:
        Componente skeleton loader
    """

    children = []

    if show_title:
        children.append(html.Div(className="skeleton skeleton-title"))

    for i in range(lines):
        # Variar el ancho para parecer más natural
        width_class = "w-100" if i % 2 == 0 else "w-75"
        children.append(html.Div(className=f"skeleton skeleton-text {width_class}"))

    if show_button:
        children.append(html.Div(className="skeleton skeleton-button mt-3"))

    return html.Div(children=children, className=f"skeleton-loader {className}")


def create_card_skeleton(title: str = "", show_icon: bool = True) -> dbc.Card:
    """
    Crea un skeleton loader en formato de tarjeta.

    Args:
        title: Título de la tarjeta (opcional)
        show_icon: Si mostrar icono en el header

    Returns:
        Tarjeta con skeleton loader
    """

    header_content = []
    if show_icon:
        header_content.append(html.I(className="fas fa-spinner fa-spin me-2 text-muted"))
    header_content.append(html.Span(title or "Cargando...", className="text-muted"))

    return dbc.Card(
        [dbc.CardHeader(header_content), dbc.CardBody([create_skeleton_loader(lines=4, show_title=False)])],
        className="shadow-sm",
    )


def create_table_skeleton(rows: int = 5, cols: int = 4) -> html.Table:
    """
    Crea un skeleton loader para tablas.

    Args:
        rows: Número de filas
        cols: Número de columnas

    Returns:
        Tabla con skeleton loader
    """

    # Header
    header_cells = [html.Th(html.Div(className="skeleton skeleton-text w-75"), className="p-2") for _ in range(cols)]

    # Body rows
    body_rows = []
    for _ in range(rows):
        cells = [html.Td(html.Div(className="skeleton skeleton-text w-100"), className="p-2") for _ in range(cols)]
        body_rows.append(html.Tr(cells))

    return html.Table([html.Thead(html.Tr(header_cells)), html.Tbody(body_rows)], className="table")


def create_stats_skeleton() -> dbc.Row:
    """
    Crea skeleton loaders para tarjetas de estadísticas.

    Returns:
        Fila con 4 tarjetas skeleton
    """

    cards = []
    for _ in range(4):
        card = dbc.Col(
            [
                dbc.Card(
                    [
                        dbc.CardBody(
                            [
                                html.Div(className="skeleton skeleton-text w-50 mb-2"),
                                html.Div(className="skeleton skeleton-title"),
                            ]
                        )
                    ],
                    className="shadow-sm",
                )
            ],
            width=12,
            lg=3,
            className="mb-3",
        )
        cards.append(card)

    return dbc.Row(cards)


def create_chart_skeleton(height: str = "300px") -> html.Div:
    """
    Crea un skeleton loader para gráficos.

    Args:
        height: Altura del skeleton

    Returns:
        Skeleton loader para gráfico
    """

    return html.Div(
        [
            # Título del gráfico
            html.Div(className="skeleton skeleton-title mb-3"),
            # Área del gráfico
            html.Div(className="skeleton", style={"height": height, "width": "100%"}),
            # Leyenda
            html.Div(
                [
                    html.Div(className="skeleton skeleton-text w-25 d-inline-block me-3"),
                    html.Div(className="skeleton skeleton-text w-25 d-inline-block me-3"),
                    html.Div(className="skeleton skeleton-text w-25 d-inline-block"),
                ],
                className="mt-3 text-center",
            ),
        ]
    )


def create_catalog_panel_skeleton() -> dbc.Card:
    """
    Skeleton loader específico para el panel de control del catálogo.

    Returns:
        Panel de catálogo con skeleton loader
    """

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-database fa-spin me-2 text-primary"),
                    html.Span("Cargando información del catálogo...", className="text-primary"),
                ]
            ),
            dbc.CardBody(
                [
                    # Estado
                    html.Div(
                        [
                            html.Div(className="skeleton skeleton-text w-50 mb-2"),
                            html.Div(className="skeleton skeleton-text w-75"),
                        ],
                        className="mb-3",
                    ),
                    # Estadísticas
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Div(className="skeleton skeleton-text w-100 mb-1"),
                                    html.Div(className="skeleton skeleton-title"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Div(className="skeleton skeleton-text w-100 mb-1"),
                                    html.Div(className="skeleton skeleton-title"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Div(className="skeleton skeleton-text w-100 mb-1"),
                                    html.Div(className="skeleton skeleton-title"),
                                ],
                                width=4,
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Botón
                    html.Div(className="skeleton skeleton-button"),
                ]
            ),
        ],
        className="shadow-sm catalog-panel-update",
    )
