# frontend/components/proactive_alerts/empty_state.py
"""
EmptyState Component - Estado vacio gratificante (Pivot 2026)

Muestra un mensaje positivo cuando no hay alertas pendientes:
- Check verde grande (100px)
- "Todo al dia, buen trabajo"
- Timestamp de ultima actualizacion

Diseno: Celebra la ausencia de problemas en lugar de mostrar vacio.

Usage:
    from components.proactive_alerts import EmptyState

    # En callback cuando no hay alertas
    if not has_alerts:
        return EmptyState(last_update="2026-01-10T15:30:00Z")

See: #602 ProactiveAlerts Component
"""

from datetime import datetime, timezone
from typing import Optional

from dash import html
from dash_iconify import DashIconify

from styles.design_tokens import (
    BORDER_RADIUS,
    COLORS,
    SHADOWS,
    SPACING,
    TYPOGRAPHY,
)


# =============================================================================
# Helper Functions
# =============================================================================


def _format_last_update(iso_timestamp: Optional[str]) -> str:
    """
    Formatea el timestamp de ultima actualizacion.

    Args:
        iso_timestamp: Timestamp ISO (ej: "2026-01-10T15:30:00Z")

    Returns:
        String formateado (ej: "Actualizado hace 5 min")
    """
    if not iso_timestamp:
        return ""

    try:
        # Parsear timestamp
        dt = datetime.fromisoformat(iso_timestamp.replace("Z", "+00:00"))
        now = datetime.now(timezone.utc)
        diff = now - dt

        minutes = int(diff.total_seconds() / 60)

        if minutes < 1:
            return "Actualizado ahora"
        elif minutes < 60:
            return f"Actualizado hace {minutes} min"
        elif minutes < 1440:
            hours = minutes // 60
            return f"Actualizado hace {hours}h"
        else:
            days = minutes // 1440
            return f"Actualizado hace {days}d"
    except (ValueError, TypeError):
        return ""


# =============================================================================
# EmptyState Component
# =============================================================================


def EmptyState(
    title: str = "Todo al dia",
    subtitle: str = "No hay alertas pendientes. Buen trabajo!",
    last_update: Optional[str] = None,
) -> html.Div:
    """
    Crea el estado vacio gratificante.

    Args:
        title: Titulo principal (default: "Todo al dia")
        subtitle: Subtitulo descriptivo
        last_update: Timestamp ISO de ultima actualizacion

    Returns:
        html.Div con el estado vacio
    """
    # Icono de check grande
    check_icon = html.Div(
        DashIconify(
            icon="mdi:check-circle",
            width=100,
            height=100,
            style={"color": COLORS["success"]},
        ),
        style={
            "marginBottom": SPACING["l"],
            "animation": "empty-state-bounce 0.5s ease-out",
        },
    )

    # Titulo
    title_element = html.H3(
        title,
        style={
            "fontSize": TYPOGRAPHY["h2"]["size"],
            "fontWeight": TYPOGRAPHY["h2"]["weight"],
            "color": COLORS["text_primary"],
            "marginBottom": SPACING["s"],
        },
    )

    # Subtitulo
    subtitle_element = html.P(
        subtitle,
        style={
            "fontSize": TYPOGRAPHY["body"]["size"],
            "color": COLORS["text_secondary"],
            "marginBottom": SPACING["m"],
        },
    )

    # Timestamp
    timestamp_text = _format_last_update(last_update)
    timestamp_element = html.Span(
        timestamp_text,
        style={
            "fontSize": TYPOGRAPHY["caption"]["size"],
            "color": COLORS["text_muted"],
        },
    ) if timestamp_text else None

    return html.Div(
        [
            check_icon,
            title_element,
            subtitle_element,
            timestamp_element,
        ],
        style={
            "display": "flex",
            "flexDirection": "column",
            "alignItems": "center",
            "justifyContent": "center",
            "padding": SPACING["2xl"],
            "backgroundColor": COLORS["bg_primary"],
            "borderRadius": BORDER_RADIUS["lg"],
            "boxShadow": SHADOWS["sm"],
            "border": f"1px solid {COLORS['border_light']}",
            "textAlign": "center",
            "minHeight": "300px",
        },
        className="proactive-alerts-empty-state",
    )


# =============================================================================
# CSS for animations
# =============================================================================

EMPTY_STATE_CSS = """
/* EmptyState Animations */
@keyframes empty-state-bounce {
    0% {
        transform: scale(0);
        opacity: 0;
    }
    50% {
        transform: scale(1.1);
    }
    100% {
        transform: scale(1);
        opacity: 1;
    }
}

@keyframes empty-state-check {
    0% {
        stroke-dashoffset: 100;
    }
    100% {
        stroke-dashoffset: 0;
    }
}

.proactive-alerts-empty-state {
    transition: all 0.3s ease-in-out;
}

.proactive-alerts-empty-state:hover {
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.1);
}
"""


# =============================================================================
# Exports
# =============================================================================

__all__ = [
    "EmptyState",
    "EMPTY_STATE_CSS",
]
