# frontend/components/proactive_alerts/alerts_grid.py
"""
AlertsGrid Component - Grid responsive de alertas proactivas (Pivot 2026)

Grid que muestra las alertas ordenadas por severidad:
1. critical (rojo) - Primero
2. warning (ambar)
3. info (azul)
4. ok (verde) - Ultimo

Layout responsive:
- Desktop (>768px): Grid 2x2
- Tablet/Mobile (<=768px): Stack vertical

Usage:
    from components.proactive_alerts import AlertsGrid

    # Desde global-alerts-store
    @app.callback(
        Output('alerts-grid-container', 'children'),
        Input('global-alerts-store', 'data')
    )
    def render_alerts(alerts_data):
        return AlertsGrid(alerts_data)

See: #602 ProactiveAlerts Component
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import html

from styles.design_tokens import SPACING

from components.proactive_alerts.alert_card import AlertCard
from components.proactive_alerts.empty_state import EmptyState


# =============================================================================
# Severity Order (para ordenacion)
# =============================================================================

SEVERITY_ORDER = {
    "critical": 0,
    "warning": 1,
    "info": 2,
    "ok": 3,
}


# =============================================================================
# Helper Functions
# =============================================================================


def _sort_alerts_by_severity(
    modules: Dict[str, Dict[str, Any]]
) -> List[tuple]:
    """
    Ordena las alertas por severidad (critical primero).

    Args:
        modules: Dict de alertas por modulo desde global-alerts-store

    Returns:
        Lista de tuplas (module_name, module_data) ordenadas
    """
    if not modules:
        return []

    # Filtrar solo alertas que no son "ok" (o que tienen count > 0)
    active_alerts = [
        (name, data)
        for name, data in modules.items()
        if data.get("status") != "ok" or data.get("count", 0) > 0
    ]

    # Ordenar por severidad
    return sorted(
        active_alerts,
        key=lambda x: SEVERITY_ORDER.get(x[1].get("status", "info"), 99)
    )


def _has_active_alerts(alerts_data: Optional[Dict[str, Any]]) -> bool:
    """
    Determina si hay alertas activas para mostrar.

    Args:
        alerts_data: Estado completo del global-alerts-store

    Returns:
        True si hay alertas que mostrar
    """
    if not alerts_data:
        return False

    summary = alerts_data.get("summary", {})
    critical = summary.get("critical_count", 0)
    warning = summary.get("warning_count", 0)
    info = summary.get("info_count", 0)

    return (critical + warning + info) > 0


# =============================================================================
# AlertsGrid Component
# =============================================================================


def AlertsGrid(
    alerts_data: Optional[Dict[str, Any]] = None,
    max_visible: int = 4,
    show_empty_state: bool = True,
) -> html.Div:
    """
    Crea el grid de alertas proactivas.

    Args:
        alerts_data: Estado completo del global-alerts-store
            {
                "summary": {...},
                "modules": {
                    "encargos": {
                        "status": "critical",
                        "count": 1129,
                        "value": 19453.22,
                        "msg": "1.129 encargos > 15 dias",
                        "action_label": "Gestionar",
                        "action_link": "/encargos",
                        "icon": "mdi:clipboard-text-clock"
                    },
                    ...
                },
                "system": {...}
            }
        max_visible: Numero maximo de cards a mostrar (default 4)
        show_empty_state: Mostrar estado vacio gratificante si no hay alertas

    Returns:
        html.Div con el grid de alertas o empty state
    """
    # Si no hay alertas activas, mostrar empty state
    if not _has_active_alerts(alerts_data):
        if show_empty_state:
            last_update = None
            if alerts_data:
                last_update = alerts_data.get("summary", {}).get("last_update")
            return EmptyState(last_update=last_update)
        return html.Div()

    # Obtener y ordenar alertas
    modules = alerts_data.get("modules", {})
    sorted_alerts = _sort_alerts_by_severity(modules)

    # Limitar a max_visible
    visible_alerts = sorted_alerts[:max_visible]

    # Crear cards
    cards = []
    for module_name, module_data in visible_alerts:
        card = AlertCard(
            status=module_data.get("status", "info"),
            icon=module_data.get("icon", "mdi:alert-circle"),
            count=module_data.get("count", 0),
            value=module_data.get("value"),
            msg=module_data.get("msg", ""),
            action_label=module_data.get("action_label", "Ver"),
            action_link=module_data.get("action_link", "#"),
            card_id=f"alert-card-{module_name}",
        )
        cards.append(
            dbc.Col(
                card,
                xs=12,  # Full width en mobile
                md=6,   # 2 columnas en tablet+
                className="mb-3",
            )
        )

    # Indicador de alertas adicionales
    remaining = len(sorted_alerts) - max_visible
    if remaining > 0:
        more_indicator = html.Div(
            f"+{remaining} alertas mas",
            style={
                "textAlign": "center",
                "color": "#6c757d",
                "fontSize": "0.875rem",
                "padding": SPACING["s"],
            },
        )
    else:
        more_indicator = None

    return html.Div(
        [
            dbc.Row(
                cards,
                className="g-3",  # Gap entre cards
            ),
            more_indicator,
        ],
        className="proactive-alerts-grid",
        style={"marginBottom": SPACING["l"]},
    )


# =============================================================================
# CSS for animations (opcional, para inyectar en assets)
# =============================================================================

ALERTS_GRID_CSS = """
/* ProactiveAlerts Grid Animations */
.proactive-alert-card {
    transition: transform 0.2s ease-in-out, box-shadow 0.2s ease-in-out;
}

.proactive-alert-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15) !important;
}

/* Animacion de entrada */
@keyframes alert-card-enter {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.proactive-alerts-grid .proactive-alert-card {
    animation: alert-card-enter 0.3s ease-out forwards;
}

.proactive-alerts-grid .col:nth-child(1) .proactive-alert-card { animation-delay: 0s; }
.proactive-alerts-grid .col:nth-child(2) .proactive-alert-card { animation-delay: 0.05s; }
.proactive-alerts-grid .col:nth-child(3) .proactive-alert-card { animation-delay: 0.1s; }
.proactive-alerts-grid .col:nth-child(4) .proactive-alert-card { animation-delay: 0.15s; }
"""


# =============================================================================
# Exports
# =============================================================================

__all__ = [
    "AlertsGrid",
    "ALERTS_GRID_CSS",
    "SEVERITY_ORDER",
]
