# frontend/components/proactive_alerts/alert_card.py
"""
AlertCard Component - Card individual de alerta proactiva (Pivot 2026)

Card que muestra una alerta individual con:
- Icono representativo
- Contador de items
- Valor economico (opcional)
- Mensaje descriptivo
- Boton de accion

Colores por severidad:
- critical: Rojo (danger)
- warning: Ambar (warning)
- info: Azul claro (info)
- ok: Verde (success)

Usage:
    from components.proactive_alerts import AlertCard

    AlertCard(
        status="critical",
        icon="mdi:clipboard-text-clock",
        count=1129,
        value=19453.22,
        msg="1.129 encargos > 15 dias",
        action_label="Gestionar",
        action_link="/encargos"
    )

See: #602 ProactiveAlerts Component
"""

from typing import Any, Dict, Optional

import dash_bootstrap_components as dbc
from dash import html
from dash_iconify import DashIconify

from styles.design_tokens import (
    BORDER_RADIUS,
    COLORS,
    SHADOWS,
    SPACING,
    TYPOGRAPHY,
)
from utils.helpers import format_currency, format_number


# =============================================================================
# Severity Configuration
# =============================================================================

SEVERITY_CONFIG: Dict[str, Dict[str, Any]] = {
    "critical": {
        "bg_color": "#f8d7da",  # Bootstrap danger-light
        "border_color": COLORS["danger"],
        "icon_color": COLORS["danger"],
        "text_color": COLORS["danger_dark"],
        "button_color": "danger",
    },
    "warning": {
        "bg_color": "#fff3cd",  # Bootstrap warning-light
        "border_color": COLORS["warning_dark"],
        "icon_color": COLORS["warning_dark"],
        "text_color": COLORS["gray_800"],
        "button_color": "warning",
    },
    "info": {
        "bg_color": "#cff4fc",  # Bootstrap info-light
        "border_color": COLORS["info"],
        "icon_color": COLORS["info"],
        "text_color": COLORS["info_dark"],
        "button_color": "info",
    },
    "ok": {
        "bg_color": "#d4edda",  # Bootstrap success-light
        "border_color": COLORS["success"],
        "icon_color": COLORS["success"],
        "text_color": COLORS["success_dark"],
        "button_color": "success",
    },
}


# =============================================================================
# Style Functions
# =============================================================================


def _get_card_style(status: str) -> Dict[str, Any]:
    """Genera estilos de la card segun severidad."""
    config = SEVERITY_CONFIG.get(status, SEVERITY_CONFIG["info"])

    return {
        "backgroundColor": config["bg_color"],
        "borderLeft": f"4px solid {config['border_color']}",
        "borderRadius": BORDER_RADIUS["md"],
        "padding": SPACING["m"],
        "boxShadow": SHADOWS["sm"],
        "height": "100%",
        "display": "flex",
        "flexDirection": "column",
        "transition": "all 0.2s ease-in-out",
    }


def _get_icon_container_style(status: str) -> Dict[str, Any]:
    """Genera estilos del contenedor de icono."""
    config = SEVERITY_CONFIG.get(status, SEVERITY_CONFIG["info"])

    return {
        "width": "48px",
        "height": "48px",
        "borderRadius": BORDER_RADIUS["lg"],
        "backgroundColor": config["border_color"],
        "display": "flex",
        "alignItems": "center",
        "justifyContent": "center",
        "flexShrink": 0,
    }


# =============================================================================
# AlertCard Component
# =============================================================================


def AlertCard(
    status: str = "info",
    icon: str = "mdi:alert-circle",
    count: int = 0,
    value: Optional[float] = None,
    msg: str = "",
    action_label: str = "Ver",
    action_link: str = "#",
    card_id: Optional[str] = None,
) -> html.Div:
    """
    Crea una card de alerta individual.

    Args:
        status: Severidad de la alerta (critical, warning, info, ok)
        icon: Icono dash-iconify (ej: "mdi:clipboard-text-clock")
        count: Numero de items que requieren atencion
        value: Valor economico en EUR (opcional)
        msg: Mensaje descriptivo para el usuario
        action_label: Texto del boton de accion
        action_link: URL de navegacion al hacer click
        card_id: ID opcional para el componente

    Returns:
        html.Div con la card de alerta
    """
    config = SEVERITY_CONFIG.get(status, SEVERITY_CONFIG["info"])

    # Header: Icono + Contador
    header = html.Div(
        [
            # Icono en circulo coloreado
            html.Div(
                DashIconify(
                    icon=icon,
                    width=24,
                    height=24,
                    style={"color": COLORS["white"]},
                ),
                style=_get_icon_container_style(status),
            ),
            # Contador grande
            html.Div(
                [
                    html.Span(
                        format_number(count if count is not None else 0),
                        style={
                            "fontSize": TYPOGRAPHY["h2"]["size"],
                            "fontWeight": TYPOGRAPHY["h2"]["weight"],
                            "color": config["text_color"],
                            "lineHeight": "1",
                        },
                    ),
                    # Valor economico (si existe)
                    html.Span(
                        format_currency(value, decimals=0) if value else "",
                        style={
                            "fontSize": TYPOGRAPHY["body_small"]["size"],
                            "color": config["text_color"],
                            "marginLeft": SPACING["s"],
                            "opacity": "0.8",
                        },
                    ) if value else None,
                ],
                style={
                    "display": "flex",
                    "alignItems": "baseline",
                    "marginLeft": SPACING["m"],
                },
            ),
        ],
        style={
            "display": "flex",
            "alignItems": "center",
            "marginBottom": SPACING["s"],
        },
    )

    # Mensaje descriptivo
    message = html.P(
        msg,
        style={
            "fontSize": TYPOGRAPHY["body"]["size"],
            "color": config["text_color"],
            "margin": 0,
            "marginBottom": SPACING["m"],
            "flex": 1,
        },
    )

    # Boton de accion
    action_button = dbc.Button(
        [
            action_label,
            DashIconify(
                icon="mdi:arrow-right",
                width=16,
                className="ms-1",
            ),
        ],
        href=action_link,
        color=config["button_color"],
        size="sm",
        className="mt-auto",
        style={"alignSelf": "flex-start"},
    )

    # Construir card
    card_props = {
        "style": _get_card_style(status),
        "className": "proactive-alert-card",
    }
    if card_id:
        card_props["id"] = card_id

    return html.Div(
        [header, message, action_button],
        **card_props,
    )


# =============================================================================
# Exports
# =============================================================================

__all__ = [
    "AlertCard",
    "SEVERITY_CONFIG",
]
