"""
Componente de visualización: Waterfall chart para análisis de crecimiento.

Comparación YoY/MoM/QoQ con barras conectadas.
Verde (crecimiento), Rojo (decrecimiento), Azul (totales).

Issue #400 - Sprint 3: Visualizaciones
Issue #454 - Chips clicables para mejorar clickabilidad de barras pequeñas
"""

import plotly.graph_objects as go
from dash import dcc, html
from typing import Dict, List, Union
import dash_bootstrap_components as dbc
from dash_iconify import DashIconify

# Import centralizado de format_category_name (Issue #436 - DRY)
from utils.constants import format_category_name
from styles.design_tokens import COLORS


def _format_period_label(period: Union[str, Dict, None], default: str) -> str:
    """
    Formatea etiqueta de período para el eje X del waterfall.

    El backend puede devolver:
    - str: "T1 2025" (ya formateado)
    - dict: {"from": "2025-01-01", "to": "2025-03-31"}
    - None: usar valor por defecto

    Args:
        period: Período en cualquier formato
        default: Valor por defecto si period es None

    Returns:
        Etiqueta legible (ej: "Ene-Mar 2025")
    """
    if period is None:
        return default

    if isinstance(period, str):
        return period

    if isinstance(period, dict):
        # Extraer fechas del dict {"from": "...", "to": "..."}
        from_date = period.get('from', '')
        to_date = period.get('to', '')

        if from_date and to_date:
            try:
                # Parsear fechas ISO
                from datetime import datetime
                from_dt = datetime.fromisoformat(from_date.replace('Z', '+00:00'))
                to_dt = datetime.fromisoformat(to_date.replace('Z', '+00:00'))

                # Formato: "Ene-Mar 2025" o "T1 2025"
                month_names = {
                    1: 'Ene', 2: 'Feb', 3: 'Mar', 4: 'Abr', 5: 'May', 6: 'Jun',
                    7: 'Jul', 8: 'Ago', 9: 'Sep', 10: 'Oct', 11: 'Nov', 12: 'Dic'
                }
                from_month = month_names.get(from_dt.month, str(from_dt.month))
                to_month = month_names.get(to_dt.month, str(to_dt.month))

                if from_dt.year == to_dt.year:
                    return f"{from_month}-{to_month} {from_dt.year}"
                else:
                    return f"{from_month} {from_dt.year}-{to_month} {to_dt.year}"
            except (ValueError, AttributeError):
                pass

        return default

    return default


def create_category_chips(waterfall_data: Dict) -> html.Div:
    """
    Crea chips clicables para cada categoría del waterfall.

    Issue #454: Mejora la clickabilidad cuando las barras son muy pequeñas.
    Los chips proporcionan una alternativa de interacción con tamaño consistente.

    Args:
        waterfall_data: Respuesta de /waterfall-analysis endpoint

    Returns:
        html.Div con chips de categorías
    """
    if not waterfall_data or not waterfall_data.get('waterfall_data'):
        return html.Div()  # Empty div si no hay datos

    category_changes = waterfall_data.get('waterfall_data', [])

    # Ordenar por impacto absoluto (mismo orden que el gráfico)
    sorted_changes = sorted(
        category_changes,
        key=lambda x: abs(x.get('absolute_change', 0)),
        reverse=True
    )[:10]

    chips = []
    for change in sorted_changes:
        category = change.get('category', '')
        absolute_change = float(change.get('absolute_change', 0))
        is_positive = absolute_change >= 0
        readable_name = format_category_name(category)

        # Chip con botón outline coloreado según dirección
        chip = dbc.Button(
            [
                html.Span(readable_name, style={"fontSize": "0.75rem"}),
                html.Span(" "),
                DashIconify(
                    icon="mdi:arrow-up" if is_positive else "mdi:arrow-down",
                    width=12,
                ),
            ],
            id={"type": "waterfall-category-chip", "index": category},
            color="success" if is_positive else "danger",
            outline=True,
            size="sm",
            className="me-1 mb-1",
            style={"padding": "0.2rem 0.5rem"},
        )
        chips.append(chip)

    return html.Div(
        [
            html.Small(
                [
                    DashIconify(icon="mdi:cursor-default-click", width=14, className="me-1"),
                    "Click en categoría para filtrar contribuyentes:",
                ],
                className="text-muted d-block mb-2",
            ),
            html.Div(chips, className="d-flex flex-wrap"),
        ],
        className="mt-3 pt-2 border-top",
    )


def create_product_waterfall_chart(
    contributors_data: Dict,
    category_name: str,
) -> dcc.Graph:
    """
    Crea waterfall chart para productos de una categoría específica.

    Issue #458: Muestra top 15 productos con mayor impacto en la variación.
    Usa datos del endpoint /top-contributors con filtro de categoría.

    Args:
        contributors_data: Respuesta de /top-contributors con filtro de categoría
        category_name: Nombre legible de la categoría filtrada

    Returns:
        dcc.Graph con waterfall de productos configurado
    """
    if not contributors_data or not contributors_data.get('contributors'):
        return dcc.Graph(
            id="product-waterfall-chart",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": "No hay datos de productos disponibles",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": COLORS["text_secondary"]}
                    }],
                    "xaxis": {"visible": False},
                    "yaxis": {"visible": False},
                    "height": 500
                }
            },
            config={"displayModeBar": False},
            style={"height": "500px"},
        )

    contributors = contributors_data.get('contributors', [])
    total_change = float(contributors_data.get('total_change', 0))

    # Calcular el total de ventas previas para la categoría
    # Sumamos previous_sales de todos los productos
    category_previous_total = sum(
        float(c.get('previous_sales', 0) or 0) for c in contributors
    )
    category_current_total = sum(
        float(c.get('current_sales', 0) or 0) for c in contributors
    )

    # Filtrar top 15 productos por impacto absoluto
    sorted_products = sorted(
        contributors,
        key=lambda x: abs(float(x.get('variation_euros', 0) or 0)),
        reverse=True
    )[:15]

    # Preparar datos para waterfall
    x_labels = [f"Base {category_name[:15]}..."] if len(category_name) > 15 else [f"Base {category_name}"]
    y_values = [category_previous_total]
    measure_types = ['absolute']
    text_labels = [f"€{category_previous_total:,.0f}"]
    hover_texts = [f"Total anterior: €{category_previous_total:,.2f}"]
    custom_data = [None]

    # Añadir cambios por producto
    for product in sorted_products:
        product_name = product.get('active_ingredient', 'N/A')
        variation = float(product.get('variation_euros', 0) or 0)
        variation_pct = float(product.get('variation_percent', 0) or 0)
        units_var = product.get('variation_units')

        # Truncar nombre si es muy largo
        display_name = product_name[:20] + "..." if len(product_name) > 20 else product_name

        x_labels.append(display_name)
        y_values.append(variation)
        measure_types.append('relative')
        custom_data.append(product_name)

        sign = "+" if variation >= 0 else ""
        text_labels.append(f"{sign}€{variation:,.0f}")

        units_text = ""
        if units_var is not None:
            units_sign = "+" if units_var > 0 else ""
            units_text = f"<br>Unidades: {units_sign}{units_var:,}"

        hover_texts.append(
            f"<b>{product_name}</b><br>"
            f"Cambio: {sign}€{variation:,.2f} ({sign}{variation_pct:.1f}%)"
            f"{units_text}"
        )

    # Total final de la categoría
    x_labels.append(f"Total {category_name[:10]}...")
    y_values.append(category_current_total)
    measure_types.append('total')
    text_labels.append(f"€{category_current_total:,.0f}")

    growth_pct = ((category_current_total - category_previous_total) / category_previous_total * 100) if category_previous_total > 0 else 0
    hover_texts.append(
        f"Total actual: €{category_current_total:,.2f}<br>"
        f"Variación: {growth_pct:+.1f}%"
    )
    custom_data.append(None)

    # Crear figura waterfall
    fig = go.Figure(go.Waterfall(
        x=x_labels,
        y=y_values,
        measure=measure_types,
        text=text_labels,
        textposition='outside',
        hovertext=hover_texts,
        hoverinfo='text',
        customdata=custom_data,
        connector={
            'mode': 'between',
            'line': {
                'width': 2,
                'color': COLORS["gray_500"],
                'dash': 'solid'
            }
        },
        increasing={
            'marker': {
                'color': '#27ae60',
                'line': {'color': '#229954', 'width': 1}
            }
        },
        decreasing={
            'marker': {
                'color': '#e74c3c',
                'line': {'color': '#c0392b', 'width': 1}
            }
        },
        totals={
            'marker': {
                'color': '#3498db',
                'line': {'color': '#2980b9', 'width': 1}
            }
        }
    ))

    fig.update_layout(
        title={
            'text': f'Top 15 Productos<br><sub>{category_name}</sub>',
            'x': 0.5,
            'xanchor': 'center',
            'font': {'size': 16, 'color': '#2c3e50'}
        },
        xaxis={
            'title': '',
            'showgrid': False,
            'linecolor': '#bdc3c7',
            'tickangle': -45,
            'tickfont': {'size': 9}
        },
        yaxis={
            'title': 'Ventas (€)',
            'showgrid': True,
            'gridcolor': '#ecf0f1',
            'linecolor': '#bdc3c7',
            'tickformat': ',.0f',
            'separatethousands': True
        },
        height=500,
        margin={'l': 70, 'r': 30, 't': 80, 'b': 140},
        plot_bgcolor='white',
        paper_bgcolor='white',
        showlegend=False
    )

    return dcc.Graph(
        id="product-waterfall-chart",
        figure=fig,
        config={
            'displayModeBar': True,
            'displaylogo': False,
            'modeBarButtonsToRemove': ['lasso2d', 'select2d'],
            'locale': 'es',
            'responsive': True
        },
        style={"height": "500px"},
    )


def create_waterfall_chart(waterfall_data: Dict) -> dcc.Graph:
    """
    Crea waterfall chart de análisis de crecimiento comparativo.

    Muestra top 10 categorías con mayor impacto absoluto.
    Conectores entre barras para visualizar flujo.

    Args:
        waterfall_data: Respuesta de /waterfall-analysis endpoint

    Returns:
        dcc.Graph con waterfall configurado
    """
    # Backend returns 'waterfall_data' (list of changes), not 'category_changes'
    if not waterfall_data or not waterfall_data.get('waterfall_data'):
        return dcc.Graph(
            id="waterfall-analysis-chart",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": "No hay datos de comparación disponibles",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": COLORS["text_secondary"]}
                    }],
                    "xaxis": {"visible": False},
                    "yaxis": {"visible": False},
                    "height": 500
                }
            },
            config={"displayModeBar": False}
        )

    # Extract from backend response structure
    summary = waterfall_data.get('summary', {})
    base_total = float(summary.get('total_previous', 0) or waterfall_data.get('base_total', 0))
    current_total = float(summary.get('total_current', 0) or waterfall_data.get('current_total', 0))
    # Calculate growth_percentage if not present
    growth_percentage = summary.get('growth_percentage') or waterfall_data.get('growth_percentage')
    if growth_percentage is None and base_total > 0:
        growth_percentage = ((current_total - base_total) / base_total) * 100
    growth_percentage = float(growth_percentage or 0)
    # Backend returns 'waterfall_data' list, not 'category_changes'
    category_changes = waterfall_data.get('waterfall_data', [])
    # Extract comparison metadata
    comparison = waterfall_data.get('comparison', {})
    comparison_period = comparison.get('type', 'yoy') or waterfall_data.get('comparison_period', 'yoy')
    # Fix: previous_period/current_period pueden ser dicts, no strings
    base_label = _format_period_label(
        comparison.get('previous_period') or waterfall_data.get('base_period_label'),
        'Período Base'
    )
    current_label = _format_period_label(
        comparison.get('current_period') or waterfall_data.get('current_period_label'),
        'Período Actual'
    )

    # Filtrar top 10 categorías por impacto absoluto
    sorted_changes = sorted(
        category_changes,
        key=lambda x: abs(x.get('absolute_change', 0)),
        reverse=True
    )[:10]

    # Preparar datos para waterfall
    x_labels = [base_label]
    y_values = [base_total]
    measure_types = ['absolute']
    text_labels = [f"€{base_total:,.0f}"]
    hover_texts = [f"{base_label}: €{base_total:,.2f}"]
    # customdata almacena category_key para click-to-filter (Issue: Click waterfall → filter contributors)
    custom_data = [None]  # Base period no tiene categoría

    # Añadir cambios por categoría
    for change in sorted_changes:
        category = change.get('category', '')
        absolute_change = float(change.get('absolute_change', 0))
        percentage_change = float(change.get('percentage_change', 0))

        # Nombre legible de categoría (usando import centralizado)
        readable_name = format_category_name(category)

        x_labels.append(readable_name)
        y_values.append(absolute_change)
        measure_types.append('relative')
        custom_data.append(category)  # category_key para filtrar

        # Texto y hover diferenciados por signo
        sign = "+" if absolute_change >= 0 else ""
        text_labels.append(f"{sign}€{absolute_change:,.0f}")
        hover_texts.append(
            f"{readable_name}<br>"
            f"Cambio: {sign}€{absolute_change:,.2f} ({sign}{percentage_change:.1f}%)<br>"
            f"<i>Haz clic para filtrar contribuyentes</i>"
        )

    # Total final
    x_labels.append(current_label)
    y_values.append(current_total)
    measure_types.append('total')
    text_labels.append(f"€{current_total:,.0f}")
    hover_texts.append(
        f"{current_label}: €{current_total:,.2f}<br>"
        f"Crecimiento total: {growth_percentage:+.1f}%"
    )
    custom_data.append(None)  # Total final no tiene categoría

    # Crear figura waterfall
    fig = go.Figure(go.Waterfall(
        x=x_labels,
        y=y_values,
        measure=measure_types,
        text=text_labels,
        textposition='outside',
        hovertext=hover_texts,
        hoverinfo='text',
        customdata=custom_data,  # category_key para click-to-filter
        connector={
            'mode': 'between',
            'line': {
                'width': 2,
                'color': COLORS["gray_500"],  # Design System token
                'dash': 'solid'
            }
        },
        increasing={
            'marker': {
                'color': '#27ae60',
                'line': {'color': '#229954', 'width': 1}
            }
        },
        decreasing={
            'marker': {
                'color': '#e74c3c',
                'line': {'color': '#c0392b', 'width': 1}
            }
        },
        totals={
            'marker': {
                'color': '#3498db',
                'line': {'color': '#2980b9', 'width': 1}
            }
        }
    ))

    # Título descriptivo según período de comparación
    comparison_titles = {
        'yoy': 'Análisis Año sobre Año (YoY)',
        'mom': 'Análisis Mes sobre Mes (MoM)',
        'qoq': 'Análisis Trimestre sobre Trimestre (QoQ)'
    }
    title_text = comparison_titles.get(comparison_period, 'Análisis Comparativo de Crecimiento')

    fig.update_layout(
        title={
            'text': f'{title_text}<br><sub>Top 10 Categorías con Mayor Impacto</sub>',
            'x': 0.5,
            'xanchor': 'center',
            'font': {'size': 18, 'color': '#2c3e50'}
        },
        xaxis={
            'title': 'Categorías',
            'showgrid': False,
            'linecolor': '#bdc3c7',
            'tickangle': -45
        },
        yaxis={
            'title': 'Ventas (€)',
            'showgrid': True,
            'gridcolor': '#ecf0f1',
            'linecolor': '#bdc3c7',
            'tickformat': ',.0f',
            'separatethousands': True
        },
        height=500,
        margin={'l': 80, 'r': 80, 't': 100, 'b': 120},  # r: 40→80 para mostrar barra final
        plot_bgcolor='white',
        paper_bgcolor='white',
        showlegend=False
    )

    return dcc.Graph(
        id="waterfall-analysis-chart",
        figure=fig,
        config={
            'displayModeBar': True,
            'displaylogo': False,
            'modeBarButtonsToRemove': ['lasso2d', 'select2d'],
            'locale': 'es',
            'responsive': True  # Design review fix
        }
    )
