"""
Componente de Descomposición STL y Forecast para Tab 3 Seasonality.

Issue #498: Visualización de descomposición STL y forecast Holt-Winters.

Características:
- Gráfico multi-traza con componentes STL (observed, trend, seasonal, residual)
- Gráfico de forecast con intervalo de confianza
- Subplots organizados verticalmente para fácil comparación
- Tooltips con valores exactos
"""

import logging
from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html
from plotly.subplots import make_subplots

from utils.helpers import format_currency

logger = logging.getLogger(__name__)

# Configuración de los gráficos
CHART_CONFIG = {
    "height_decomposition": 600,
    "height_forecast": 350,
    "color_observed": "#2c3e50",
    "color_trend": "#3498db",
    "color_seasonal": "#27ae60",
    "color_residual": "#e74c3c",
    "color_forecast": "#9b59b6",
    "color_confidence": "rgba(155, 89, 182, 0.2)",
    "color_historical": "#95a5a6",
}

# Nombres de meses en español
MONTH_NAMES = {
    1: "Enero", 2: "Febrero", 3: "Marzo", 4: "Abril",
    5: "Mayo", 6: "Junio", 7: "Julio", 8: "Agosto",
    9: "Septiembre", 10: "Octubre", 11: "Noviembre", 12: "Diciembre",
}


def create_decomposition_chart(
    decomposition: List[Dict[str, Any]],
    summary: Optional[Dict[str, Any]] = None,
) -> go.Figure:
    """
    Crea gráfico de descomposición STL con 4 subplots.

    Args:
        decomposition: Lista de puntos con date, observed, trend, seasonal, residual
        summary: Resumen con trend_direction, seasonal_strength

    Returns:
        Figura Plotly con subplots STL
    """
    if not decomposition:
        return _create_empty_decomposition_chart()

    # Extraer datos
    dates = [d.get("date", "") for d in decomposition]
    observed = [d.get("observed", 0) for d in decomposition]
    trend = [d.get("trend", 0) for d in decomposition]
    seasonal = [d.get("seasonal", 0) for d in decomposition]
    residual = [d.get("residual", 0) for d in decomposition]

    # Crear subplots
    fig = make_subplots(
        rows=4, cols=1,
        shared_xaxes=True,
        vertical_spacing=0.08,
        subplot_titles=(
            "Serie Original (Observado)",
            "Tendencia",
            "Componente Estacional",
            "Residuo (Ruido)",
        ),
    )

    # Hover template común
    hover_template = "<b>%{x}</b><br>%{customdata}<extra></extra>"

    # 1. Serie observada
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=observed,
            mode="lines+markers",
            name="Observado",
            line=dict(color=CHART_CONFIG["color_observed"], width=2),
            marker=dict(size=4),
            customdata=[f"Ventas: {format_currency(v)}" for v in observed],
            hovertemplate=hover_template,
        ),
        row=1, col=1,
    )

    # 2. Tendencia
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=trend,
            mode="lines",
            name="Tendencia",
            line=dict(color=CHART_CONFIG["color_trend"], width=3),
            customdata=[f"Tendencia: {format_currency(v)}" for v in trend],
            hovertemplate=hover_template,
        ),
        row=2, col=1,
    )

    # 3. Componente estacional
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=seasonal,
            mode="lines",
            name="Estacional",
            line=dict(color=CHART_CONFIG["color_seasonal"], width=2),
            fill="tozeroy",
            fillcolor="rgba(39, 174, 96, 0.2)",
            customdata=[f"Efecto estacional: {format_currency(v)}" for v in seasonal],
            hovertemplate=hover_template,
        ),
        row=3, col=1,
    )

    # 4. Residuo
    fig.add_trace(
        go.Scatter(
            x=dates,
            y=residual,
            mode="lines",
            name="Residuo",
            line=dict(color=CHART_CONFIG["color_residual"], width=1),
            customdata=[f"Residuo: {format_currency(v)}" for v in residual],
            hovertemplate=hover_template,
        ),
        row=4, col=1,
    )

    # Añadir línea cero en residuo
    fig.add_hline(
        y=0,
        line=dict(color="black", width=1, dash="dash"),
        row=4, col=1,
    )

    # Layout general
    fig.update_layout(
        height=CHART_CONFIG["height_decomposition"],
        showlegend=False,
        margin=dict(l=60, r=40, t=80, b=40),
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
        title={
            "text": "Descomposición STL de Ventas",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
    )

    # Configurar ejes
    for i in range(1, 5):
        fig.update_yaxes(
            title_text="€" if i < 4 else "Variación €",
            showgrid=True,
            gridcolor="rgba(0,0,0,0.1)",
            row=i, col=1,
        )

    fig.update_xaxes(
        showgrid=True,
        gridcolor="rgba(0,0,0,0.1)",
    )

    return fig


def _create_empty_decomposition_chart() -> go.Figure:
    """Crea un gráfico vacío con mensaje."""
    fig = go.Figure()

    fig.add_annotation(
        text="No hay datos suficientes para descomposición STL<br>"
             "(Se requieren al menos 12 meses de datos)",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Descomposición STL de Ventas",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=CHART_CONFIG["height_decomposition"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def create_forecast_chart(
    forecast: List[Dict[str, Any]],
    historical: Optional[List[Dict[str, Any]]] = None,
    summary: Optional[Dict[str, Any]] = None,
    confidence_level: float = 0.95,
) -> go.Figure:
    """
    Crea gráfico de forecast con intervalo de confianza.

    Args:
        forecast: Lista de predicciones con date, forecast, lower_bound, upper_bound
        historical: Datos históricos recientes para contexto
        summary: Resumen con total_forecast, forecast_trend
        confidence_level: Nivel de confianza (0.95 = 95%)

    Returns:
        Figura Plotly con forecast e intervalo
    """
    if not forecast:
        return _create_empty_forecast_chart()

    fig = go.Figure()

    # Datos históricos (si hay)
    if historical:
        hist_dates = [h.get("date", "") for h in historical]
        hist_values = [h.get("value", 0) for h in historical]

        fig.add_trace(
            go.Scatter(
                x=hist_dates,
                y=hist_values,
                mode="lines+markers",
                name="Histórico",
                line=dict(color=CHART_CONFIG["color_historical"], width=2),
                marker=dict(size=6),
                hovertemplate="<b>%{x}</b><br>Ventas: " + "%{y:,.0f}€<extra></extra>",
            )
        )

    # Extraer datos forecast
    forecast_dates = [f.get("date", "") for f in forecast]
    forecast_values = [f.get("forecast", 0) for f in forecast]
    lower_bounds = [f.get("lower_bound", 0) for f in forecast]
    upper_bounds = [f.get("upper_bound", 0) for f in forecast]
    month_names = [f.get("month_name", "") for f in forecast]

    # Banda de confianza (fill between)
    # Primero añadir upper bound
    fig.add_trace(
        go.Scatter(
            x=forecast_dates,
            y=upper_bounds,
            mode="lines",
            name=f"Límite superior ({int(confidence_level*100)}%)",
            line=dict(width=0),
            showlegend=False,
            hoverinfo="skip",
        )
    )

    # Lower bound con fill
    fig.add_trace(
        go.Scatter(
            x=forecast_dates,
            y=lower_bounds,
            mode="lines",
            name=f"Intervalo confianza {int(confidence_level*100)}%",
            line=dict(width=0),
            fill="tonexty",
            fillcolor=CHART_CONFIG["color_confidence"],
            hoverinfo="skip",
        )
    )

    # Línea de forecast principal
    hover_texts = []
    for i, f in enumerate(forecast):
        hover_texts.append(
            f"<b>{month_names[i]}</b><br>"
            f"Predicción: {format_currency(forecast_values[i])}<br>"
            f"Rango: {format_currency(lower_bounds[i])} - {format_currency(upper_bounds[i])}"
        )

    fig.add_trace(
        go.Scatter(
            x=forecast_dates,
            y=forecast_values,
            mode="lines+markers",
            name="Predicción",
            line=dict(color=CHART_CONFIG["color_forecast"], width=3),
            marker=dict(size=8, symbol="diamond"),
            customdata=hover_texts,
            hovertemplate="%{customdata}<extra></extra>",
        )
    )

    # Línea vertical separando histórico de forecast
    if historical and forecast:
        last_hist_date = historical[-1].get("date", "")
        fig.add_vline(
            x=last_hist_date,
            line=dict(color="gray", width=1, dash="dot"),
            annotation=dict(
                text="Hoy",
                font=dict(size=10),
                textangle=-90,
            ),
        )

    # Layout
    fig.update_layout(
        title={
            "text": f"Forecast de Ventas (próximos {len(forecast)} meses)",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(
            title="Fecha",
            showgrid=True,
            gridcolor="rgba(0,0,0,0.1)",
        ),
        yaxis=dict(
            title="Ventas (€)",
            showgrid=True,
            gridcolor="rgba(0,0,0,0.1)",
        ),
        height=CHART_CONFIG["height_forecast"],
        margin=dict(l=60, r=40, t=60, b=60),
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
        legend=dict(
            orientation="h",
            yanchor="bottom",
            y=1.02,
            xanchor="center",
            x=0.5,
        ),
        hovermode="x unified",
    )

    return fig


def _create_empty_forecast_chart() -> go.Figure:
    """Crea un gráfico vacío con mensaje."""
    fig = go.Figure()

    fig.add_annotation(
        text="No hay datos suficientes para generar forecast<br>"
             "(Se requieren al menos 12 meses de datos históricos)",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Forecast de Ventas",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=CHART_CONFIG["height_forecast"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def _create_initial_decomposition_chart() -> go.Figure:
    """Crea un gráfico inicial mostrando estado de carga."""
    fig = go.Figure()

    fig.add_annotation(
        text="Selecciona el tab 'Tendencias' y aplica filtros<br>"
             "para ver la descomposición STL",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Descomposición STL de Ventas",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=CHART_CONFIG["height_decomposition"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def create_decomposition_container(
    graph_id: str = "seasonality-decomposition-chart"
) -> dbc.Card:
    """
    Crea el contenedor del gráfico de descomposición STL.

    Args:
        graph_id: ID del componente Graph

    Returns:
        Card con el gráfico STL
    """
    # Issue #531: Añadir figura inicial para evitar gráfico vacío
    initial_figure = _create_initial_decomposition_chart()

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="mdi mdi-chart-timeline-variant me-2"),
                    "Descomposición STL (Tendencia + Estacionalidad)",
                ],
                className="bg-light",
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="loading-decomposition",
                        type="circle",
                        children=[
                            dcc.Graph(
                                id=graph_id,
                                figure=initial_figure,  # Issue #531: Figura inicial
                                config={
                                    "displayModeBar": True,
                                    "modeBarButtonsToRemove": ["lasso2d", "select2d"],
                                    "responsive": True,
                                },
                                style={"height": f"{CHART_CONFIG['height_decomposition']}px"},
                            ),
                        ],
                    ),
                    html.Div(
                        id="decomposition-legend",
                        className="d-flex justify-content-center gap-4 mt-2 flex-wrap",
                        children=[
                            _create_legend_item("Observado", CHART_CONFIG["color_observed"]),
                            _create_legend_item("Tendencia", CHART_CONFIG["color_trend"]),
                            _create_legend_item("Estacional", CHART_CONFIG["color_seasonal"]),
                            _create_legend_item("Residuo", CHART_CONFIG["color_residual"]),
                        ],
                    ),
                ],
            ),
        ],
        className="shadow-sm",
    )


def _create_initial_forecast_chart() -> go.Figure:
    """Crea un gráfico inicial mostrando estado de carga."""
    fig = go.Figure()

    fig.add_annotation(
        text="Selecciona el tab 'Tendencias' y aplica filtros<br>"
             "para ver la predicción de ventas",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Forecast de Ventas",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=CHART_CONFIG["height_forecast"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def create_forecast_container(
    graph_id: str = "seasonality-forecast-chart"
) -> dbc.Card:
    """
    Crea el contenedor del gráfico de forecast.

    Args:
        graph_id: ID del componente Graph

    Returns:
        Card con el gráfico de forecast
    """
    # Issue #531: Añadir figura inicial para evitar gráfico vacío
    initial_figure = _create_initial_forecast_chart()

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="mdi mdi-crystal-ball me-2"),
                    "Predicción de Ventas (Holt-Winters)",
                ],
                className="bg-light",
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="loading-forecast",
                        type="circle",
                        children=[
                            dcc.Graph(
                                id=graph_id,
                                figure=initial_figure,  # Issue #531: Figura inicial
                                config={
                                    "displayModeBar": False,
                                    "responsive": True,
                                },
                                style={"height": f"{CHART_CONFIG['height_forecast']}px"},
                            ),
                        ],
                    ),
                    html.Div(
                        id="forecast-summary",
                        className="text-center text-muted mt-2",
                        style={"fontSize": "0.85rem"},
                    ),
                ],
            ),
        ],
        className="shadow-sm",
    )


def _create_legend_item(label: str, color: str) -> html.Span:
    """Crea un elemento de leyenda."""
    return html.Span(
        [
            html.Span(
                "━━",
                style={"color": color, "fontSize": "1.2rem", "fontWeight": "bold"},
            ),
            f" {label}",
        ],
        className="text-muted",
        style={"fontSize": "0.85rem"},
    )


def create_decomposition_summary_card(summary: Optional[Dict[str, Any]]) -> dbc.Card:
    """
    Crea card de resumen de la descomposición STL.

    Args:
        summary: Resumen con trend_direction, seasonal_strength

    Returns:
        Card con métricas de resumen
    """
    if not summary:
        return html.Div()

    trend_direction = summary.get("trend_direction", "unknown")
    seasonal_strength = summary.get("seasonal_strength", 0)
    data_points = summary.get("data_points", 0)

    # Mapeo de dirección a español e icono
    direction_map = {
        "up": ("Alcista", "mdi-trending-up", "text-success"),
        "down": ("Bajista", "mdi-trending-down", "text-danger"),
        "stable": ("Estable", "mdi-trending-neutral", "text-secondary"),
        "unknown": ("Sin datos", "mdi-help", "text-muted"),
    }
    trend_label, trend_icon, trend_class = direction_map.get(
        trend_direction,
        direction_map["unknown"],
    )

    # Interpretar fuerza estacional
    if seasonal_strength >= 0.7:
        seasonal_label = "Fuerte"
        seasonal_class = "text-success"
    elif seasonal_strength >= 0.4:
        seasonal_label = "Moderada"
        seasonal_class = "text-warning"
    else:
        seasonal_label = "Débil"
        seasonal_class = "text-danger"

    return dbc.Card(
        [
            dbc.CardBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Small("Tendencia", className="text-muted d-block"),
                                    html.Span(
                                        [
                                            html.I(className=f"mdi {trend_icon} me-1"),
                                            trend_label,
                                        ],
                                        className=f"fw-bold {trend_class}",
                                    ),
                                ],
                                width=4,
                                className="text-center border-end",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Estacionalidad", className="text-muted d-block"),
                                    html.Span(
                                        [
                                            f"{seasonal_strength:.0%} ",
                                            html.Small(f"({seasonal_label})", className=seasonal_class),
                                        ],
                                        className="fw-bold",
                                    ),
                                ],
                                width=4,
                                className="text-center border-end",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Datos analizados", className="text-muted d-block"),
                                    html.Span(
                                        f"{data_points} meses",
                                        className="fw-bold",
                                    ),
                                ],
                                width=4,
                                className="text-center",
                            ),
                        ],
                    ),
                ],
                className="py-2",
            ),
        ],
        className="shadow-sm mb-3",
    )


def create_forecast_summary_card(
    summary: Optional[Dict[str, Any]],
    model_info: Optional[Dict[str, Any]] = None,
) -> dbc.Card:
    """
    Crea card de resumen del forecast.

    Args:
        summary: Resumen con total_forecast, avg_monthly, forecast_trend
        model_info: Información del modelo usado (type, months_available)

    Returns:
        Card con métricas de forecast
    """
    if not summary:
        return html.Div()

    total_forecast = summary.get("total_forecast", 0)
    avg_monthly = summary.get("avg_monthly", 0)
    forecast_trend = summary.get("forecast_trend", "unknown")
    confidence_range = summary.get("confidence_range", "N/A")

    # Información del modelo (si está disponible)
    model_type = model_info.get("type", "Holt-Winters") if model_info else "Holt-Winters"
    months_available = model_info.get("months_available") if model_info else None

    # Mapeo de tendencia
    trend_map = {
        "up": ("Creciente", "mdi-arrow-up-bold", "text-success"),
        "down": ("Decreciente", "mdi-arrow-down-bold", "text-danger"),
        "stable": ("Estable", "mdi-arrow-right-bold", "text-secondary"),
        "unknown": ("Sin datos", "mdi-help", "text-muted"),
    }
    trend_label, trend_icon, trend_class = trend_map.get(
        forecast_trend,
        trend_map["unknown"],
    )

    return dbc.Card(
        [
            dbc.CardBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Small("Total predicho", className="text-muted d-block"),
                                    html.Span(
                                        format_currency(total_forecast),
                                        className="fw-bold text-primary",
                                    ),
                                ],
                                width=3,
                                className="text-center border-end",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Promedio mensual", className="text-muted d-block"),
                                    html.Span(
                                        format_currency(avg_monthly),
                                        className="fw-bold",
                                    ),
                                ],
                                width=3,
                                className="text-center border-end",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Tendencia", className="text-muted d-block"),
                                    html.Span(
                                        [
                                            html.I(className=f"mdi {trend_icon} me-1"),
                                            trend_label,
                                        ],
                                        className=f"fw-bold {trend_class}",
                                    ),
                                ],
                                width=3,
                                className="text-center border-end",
                            ),
                            dbc.Col(
                                [
                                    html.Small("Margen error", className="text-muted d-block"),
                                    html.Span(
                                        confidence_range,
                                        className="fw-bold",
                                    ),
                                ],
                                width=3,
                                className="text-center",
                            ),
                        ],
                    ),
                    # Mostrar info del modelo si no es el completo
                    html.Div(
                        [
                            html.Small(
                                [
                                    html.I(className="mdi mdi-information-outline me-1"),
                                    f"Modelo: {model_type}",
                                    f" ({months_available} meses de datos)" if months_available else "",
                                ],
                                className="text-muted",
                            ),
                        ],
                        className="text-center mt-2 border-top pt-2",
                    ) if model_type != "Holt-Winters (estacional)" else html.Div(),
                ],
                className="py-2",
            ),
        ],
        className="shadow-sm mb-3",
    )


def create_data_quality_alert_stl(data_quality: Optional[Dict[str, Any]]) -> Optional[dbc.Alert]:
    """
    Crea una alerta si la calidad de datos es insuficiente para STL.

    Args:
        data_quality: Indicadores de calidad de datos

    Returns:
        Alert de Bootstrap si hay problemas, None si todo OK
    """
    if not data_quality:
        return None

    sufficient_data = data_quality.get("sufficient_data", False)
    months_available = data_quality.get("months_available", 0)
    min_required = data_quality.get("min_periods_required", 24)
    message = data_quality.get("message", "")

    if sufficient_data:
        return None

    return dbc.Alert(
        html.Span(
            [
                html.I(className="mdi mdi-alert-circle me-2"),
                f"Datos insuficientes: {months_available}/{min_required} meses disponibles. ",
                message or f"Se requieren al menos {min_required} meses de datos para análisis STL.",
            ]
        ),
        color="warning",
        className="mb-3",
    )
