# frontend/components/prescription/stockout_risk_table.py
"""
Componente para visualización de matriz de riesgo de rotura de stock.

Issue #500: Fase 3.5 - Stock-out Risk Matrix.

Muestra:
- Tabla con semáforo de riesgo por producto
- Resumen de productos por nivel de riesgo
- Pedido recomendado para productos en riesgo
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import dcc, html
from dash_iconify import DashIconify

from styles.design_tokens import COLORS, SPACING


def create_stockout_risk_container(
    container_id: str = "stockout-risk-container",
) -> dbc.Card:
    """
    Crea el contenedor principal para la matriz de riesgo de stock.

    Args:
        container_id: ID del contenedor

    Returns:
        dbc.Card: Contenedor con controles y tabla
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:package-variant-closed-alert",
                                width=24,
                                color=COLORS["warning"],
                                className="me-2",
                            ),
                            html.Span(
                                "Previsión de Faltas y Reposición Crítica",
                                style={"fontWeight": "600"},
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}",
                },
            ),
            dbc.CardBody(
                [
                    # Row 1: Controles y resumen
                    dbc.Row(
                        [
                            # Control de días hasta reposición
                            dbc.Col(
                                [
                                    html.Label(
                                        "Días hasta reposición:",
                                        className="mb-2 small text-muted",
                                    ),
                                    dcc.Dropdown(
                                        id="stockout-days-restock",
                                        options=[
                                            {"label": "3 días", "value": 3},
                                            {"label": "5 días", "value": 5},
                                            {"label": "7 días (semanal)", "value": 7},
                                            {"label": "14 días (quincenal)", "value": 14},
                                        ],
                                        value=7,
                                        clearable=False,
                                        className="mb-2",
                                    ),
                                ],
                                width=12,
                                lg=3,
                            ),
                            # Resumen de riesgos (badges)
                            dbc.Col(
                                [
                                    html.Div(
                                        id="stockout-risk-summary-badges",
                                        className="d-flex flex-wrap gap-2 align-items-center",
                                    ),
                                ],
                                width=12,
                                lg=9,
                                className="d-flex align-items-end",
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Row 2: Tabla de riesgo
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dcc.Loading(
                                        id="stockout-risk-table-loading",
                                        type="default",
                                        children=html.Div(
                                            id="stockout-risk-table-container",
                                            style={
                                                "maxHeight": "500px",
                                                "overflowY": "auto",
                                            },
                                        ),
                                    ),
                                ],
                                width=12,
                            ),
                        ],
                    ),
                ],
                id=container_id,
            ),
        ],
        className="shadow-sm mb-4",
    )


def create_stockout_risk_summary_badges(summary: Dict[str, Any]) -> html.Div:
    """
    Crea badges con resumen de productos por nivel de riesgo.

    Args:
        summary: Diccionario con conteos por nivel de riesgo

    Returns:
        html.Div: Contenedor con badges
    """
    if not summary:
        return html.Div("Sin datos de inventario")

    badges = []

    # Badge de mes actual e índice
    current_month = summary.get("current_month", "")
    month_index = summary.get("current_month_index", 1.0)
    if current_month:
        index_color = "success" if month_index >= 1.0 else "warning"
        badges.append(
            dbc.Badge(
                [
                    DashIconify(icon="mdi:calendar", width=14, className="me-1"),
                    f"{current_month}: índice {month_index:.2f}",
                ],
                color=index_color,
                className="me-2",
            )
        )

    # Badges por nivel de riesgo
    risk_configs = [
        ("critical_count", "CRÍTICO", "danger", "mdi:alert-circle"),
        ("high_count", "ALTO", "warning", "mdi:alert"),
        ("ok_count", "OK", "success", "mdi:check-circle"),
        ("medium_count", "MEDIO", "info", "mdi:information"),
        ("excess_count", "EXCESO", "secondary", "mdi:arrow-up-bold"),
    ]

    for key, label, color, icon in risk_configs:
        count = summary.get(key, 0)
        if count > 0:
            badges.append(
                dbc.Badge(
                    [
                        DashIconify(icon=icon, width=14, className="me-1"),
                        f"{count} {label}",
                    ],
                    color=color,
                    className="me-1",
                )
            )

    # Total de productos analizados
    total = summary.get("total_products", 0)
    badges.append(
        html.Small(
            f"({total} productos analizados)",
            className="text-muted ms-2",
        )
    )

    return html.Div(badges, className="d-flex flex-wrap gap-1 align-items-center")


def create_stockout_risk_table(risk_items: List[Dict[str, Any]]) -> dbc.Table:
    """
    Crea tabla con matriz de riesgo de stock-out.

    Args:
        risk_items: Lista de productos con análisis de riesgo

    Returns:
        dbc.Table: Tabla con datos de riesgo
    """
    if not risk_items:
        return html.Div(
            [
                DashIconify(
                    icon="mdi:package-variant-closed-check",
                    width=48,
                    color=COLORS["text_secondary"],
                    className="mb-2",
                ),
                html.P(
                    "No hay datos de inventario de prescripción disponibles.",
                    className="text-muted mb-1",
                ),
                html.Small(
                    "Sube un archivo de inventario para ver el análisis de riesgo.",
                    className="text-muted",
                ),
            ],
            className="text-center py-5",
        )

    # Header de la tabla
    header = html.Thead(
        html.Tr(
            [
                html.Th("Riesgo", style={"width": "80px"}),
                html.Th("Producto"),
                html.Th("Categoría", style={"width": "120px"}),
                html.Th("Stock", style={"width": "70px", "textAlign": "right"}),
                html.Th("Vta/día", style={"width": "70px", "textAlign": "right"}),
                html.Th("Cobertura", style={"width": "80px", "textAlign": "right"}),
                html.Th("Pedir", style={"width": "70px", "textAlign": "right"}),
            ]
        ),
        className="sticky-top bg-white",
    )

    # Filas de la tabla
    rows = []
    for item in risk_items:
        risk_level = item.get("risk_level", "OK")
        risk_color = item.get("risk_color", "secondary")

        # Badge de riesgo
        risk_badge = dbc.Badge(
            risk_level,
            color=risk_color,
            className="w-100",
        )

        # Nombre de producto (truncado)
        product_name = item.get("product_name", "")[:50]
        if len(item.get("product_name", "")) > 50:
            product_name += "..."

        # Categoría formateada
        category = item.get("category", "")
        if category:
            category = category.replace("_", " ").title()[:20]

        # Días de cobertura con color
        days_coverage = item.get("days_of_coverage", 0)
        if days_coverage < 5:
            coverage_style = {"color": COLORS["danger"], "fontWeight": "bold"}
        elif days_coverage < 10:
            coverage_style = {"color": COLORS["warning"]}
        else:
            coverage_style = {"color": COLORS["success"]}

        # Pedido recomendado
        recommended = item.get("recommended_order", 0)
        recommended_cell = (
            html.Strong(f"+{recommended}", style={"color": COLORS["primary"]})
            if recommended > 0
            else "-"
        )

        rows.append(
            html.Tr(
                [
                    html.Td(risk_badge),
                    html.Td(
                        [
                            html.Div(product_name, style={"fontSize": "0.9rem"}),
                            html.Small(
                                item.get("product_code", ""),
                                className="text-muted",
                            ),
                        ]
                    ),
                    html.Td(category, className="small"),
                    html.Td(
                        f"{item.get('current_stock', 0):,}",
                        style={"textAlign": "right"},
                    ),
                    html.Td(
                        f"{item.get('adjusted_daily_sales', 0):.1f}",
                        style={"textAlign": "right"},
                    ),
                    html.Td(
                        f"{days_coverage:.0f}d",
                        style={**coverage_style, "textAlign": "right"},
                    ),
                    html.Td(recommended_cell, style={"textAlign": "right"}),
                ],
                className=f"table-{risk_color}" if risk_color in ["danger", "warning"] else "",
            )
        )

    body = html.Tbody(rows)

    return dbc.Table(
        [header, body],
        bordered=True,
        hover=True,
        striped=False,
        responsive=True,
        size="sm",
        className="mb-0",
    )


def create_empty_inventory_message() -> html.Div:
    """Mensaje cuando no hay inventario disponible."""
    return html.Div(
        [
            DashIconify(
                icon="mdi:package-variant-closed",
                width=64,
                color=COLORS["text_secondary"],
                className="mb-3",
            ),
            html.H5(
                "Sin datos de inventario",
                className="text-muted mb-2",
            ),
            html.P(
                "Para ver el análisis de riesgo de stock, sube un archivo "
                "de inventario desde la sección de Carga de Archivos.",
                className="text-muted mb-3",
                style={"maxWidth": "400px"},
            ),
            dbc.Button(
                html.Span([
                    DashIconify(icon="mdi:upload", width=16, className="me-2"),
                    "Subir Inventario",
                ]),
                href="/upload",
                color="primary",
                outline=True,
            ),
        ],
        className="text-center py-5",
    )
