"""
Componente de visualización: Evolución temporal de ventas de prescripción.

Gráfico de líneas con 14 series (categorías de prescripción).
Top 5 categorías visibles por defecto, resto en legendonly.

Issue #400 - Sprint 3: Visualizaciones
"""

import plotly.graph_objects as go
from dash import dcc
from typing import List, Dict
import pandas as pd

from utils.helpers import generate_plotly_spanish_month_ticks

# Colores diferenciados por categoría (14 colores distintos)
CATEGORY_COLORS = {
    "MEDICAMENTOS": "#1f77b4",
    "MARGEN_ESPECIAL_3": "#ff7f0e",
    "MARGEN_ESPECIAL_2": "#2ca02c",
    "MARGEN_ESPECIAL_1": "#d62728",
    "FORMULAS_MAGISTRALES": "#9467bd",
    "VACUNAS_INDIVIDUALIZADAS": "#8c564b",
    "TIRAS_REACTIVAS_GLUCOSA": "#e377c2",
    "INCONTINENCIA_FINANCIADA": "#7f7f7f",
    "EFECTOS_FINANCIADOS": "#bcbd22",
    "ORTOPEDIA_FINANCIADA": "#17becf",
    "DIETOTERAPICOS": "#aec7e8",
    "CONJUNTO_HOMOGENEO": "#ffbb78",
    "VETERINARIA": "#98df8a",
    "USO_HUMANO_NO_FINANCIADO": "#ff9896"
}


def format_category_name(category: str) -> str:
    """
    Convierte categoría interna a nombre legible.

    Args:
        category: Nombre interno de categoría

    Returns:
        Nombre legible en español
    """
    category_map = {
        "MEDICAMENTOS": "Medicamentos",
        "MARGEN_ESPECIAL_3": "PVP > 578 €",
        "MARGEN_ESPECIAL_2": "PVP > 256 €",
        "MARGEN_ESPECIAL_1": "PVP > 144 €",
        "FORMULAS_MAGISTRALES": "Fórmulas Magistrales",
        "VACUNAS_INDIVIDUALIZADAS": "Vacunas Individualizadas",
        "TIRAS_REACTIVAS_GLUCOSA": "Tiras Reactivas Glucosa",
        "INCONTINENCIA_FINANCIADA": "Incontinencia Financiada",
        "EFECTOS_FINANCIADOS": "Efectos Financiados",
        "ORTOPEDIA_FINANCIADA": "Ortopedia Financiada",
        "DIETOTERAPICOS": "Dietoterápicos",
        "CONJUNTO_HOMOGENEO": "Conjunto Homogéneo",
        "VETERINARIA": "Veterinaria",
        "USO_HUMANO_NO_FINANCIADO": "Uso Humano No Financiado"
    }
    return category_map.get(category, category.replace("_", " ").title())


def create_prescription_evolution_chart(
    time_series_data: List[Dict],
    category_summary: List[Dict],
    chart_type: str = "line"
) -> dcc.Graph:
    """
    Crea gráfico de evolución temporal de ventas de prescripción.

    Top 5 categorías visibles por defecto, resto en legendonly.
    Formato español para números y fechas.

    Args:
        time_series_data: Lista en formato LONG con {date, category, sales, units}
        category_summary: Resumen por categoría para ordenar top 5
        chart_type: Tipo de visualización ("line" o "ribbon")

    Returns:
        dcc.Graph con configuración completa
    """
    # Validación 1: Verificar que hay datos
    if not time_series_data or not category_summary:
        return dcc.Graph(
            id="prescription-evolution-chart",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": "No hay datos disponibles para el período seleccionado",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": "#6c757d"}
                    }],
                    "xaxis": {"visible": False},
                    "yaxis": {"visible": False},
                    "height": 500
                }
            },
            config={"displayModeBar": False}
        )

    # Convertir a DataFrame para facilitar procesamiento
    df = pd.DataFrame(time_series_data)

    # Validación 2: Verificar que DataFrame tiene columnas esperadas
    required_columns = ['date', 'category', 'sales', 'units']
    missing_columns = [col for col in required_columns if col not in df.columns]

    if missing_columns or df.empty:
        error_msg = f"Estructura de datos incorrecta. Columnas faltantes: {missing_columns}" if missing_columns else "DataFrame vacío"
        return dcc.Graph(
            id="prescription-evolution-chart",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": f"Error: {error_msg}",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": "#dc3545"}
                    }],
                    "xaxis": {"visible": False},
                    "yaxis": {"visible": False},
                    "height": 500
                }
            },
            config={"displayModeBar": False}
        )

    # Convertir fecha a datetime
    df['date'] = pd.to_datetime(df['date'])

    # Ordenar categorías por ventas totales (top 5)
    category_order = sorted(
        category_summary,
        key=lambda x: x.get('total_sales', x.get('sales', 0)),  # Backend usa 'total_sales'
        reverse=True
    )
    top_5_categories = [cat['category'] for cat in category_order[:5]]

    # Crear figura
    fig = go.Figure()

    # Añadir serie por categoría (Line Chart o Ribbon Chart)
    for category in sorted(df['category'].unique()):
        category_df = df[df['category'] == category].sort_values('date')

        # Determinar si categoría debe estar visible por defecto
        visible = True if category in top_5_categories else 'legendonly'

        if chart_type == "ribbon":
            # Ribbon Chart: Área apilada (stacked area)
            fig.add_trace(go.Scatter(
                x=category_df['date'],
                y=category_df['sales'],
                mode='lines',
                name=format_category_name(category),
                line={
                    'width': 0.5,
                    'color': CATEGORY_COLORS.get(category, '#636EFA')
                },
                fillcolor=CATEGORY_COLORS.get(category, '#636EFA'),
                stackgroup='one',  # CRÍTICO: Apilar todas las áreas
                groupnorm='',  # No normalizar (mantener valores absolutos)
                visible=visible,
                customdata=category_df['units'],
                hovertemplate=(
                    '<b>%{fullData.name}</b><br>' +
                    'Fecha: %{x|%d/%m/%Y}<br>' +
                    'Ventas: €%{y:,.2f}<br>' +
                    'Unidades: %{customdata:,.0f}<br>' +
                    '<extra></extra>'
                )
            ))
        else:
            # Line Chart: Líneas + marcadores (default)
            fig.add_trace(go.Scatter(
                x=category_df['date'],
                y=category_df['sales'],
                mode='lines+markers',
                name=format_category_name(category),
                line={
                    'color': CATEGORY_COLORS.get(category, '#636EFA'),
                    'width': 2
                },
                marker={'size': 6},
                visible=visible,
                customdata=category_df['units'],
                hovertemplate=(
                    '<b>%{fullData.name}</b><br>' +
                    'Fecha: %{x|%d/%m/%Y}<br>' +
                    'Ventas: €%{y:,.2f}<br>' +
                    'Unidades: %{customdata:,.0f}<br>' +
                    '<extra></extra>'
                )
            ))

    # Generar ticks de meses en español (Plotly no soporta locale de D3)
    all_dates = df['date'].tolist()
    tickvals, ticktext = generate_plotly_spanish_month_ticks(all_dates)

    # Configurar layout con meses en español
    xaxis_config = {
        'title': 'Fecha',
        'showgrid': True,
        'gridcolor': '#ecf0f1',
        'linecolor': '#bdc3c7',
    }

    # Solo usar tickvals/ticktext si tenemos ticks generados
    if tickvals and ticktext:
        xaxis_config['tickvals'] = tickvals
        xaxis_config['ticktext'] = ticktext
    else:
        # Fallback a formato numérico si no hay ticks
        xaxis_config['tickformat'] = '%d/%m/%y'
        xaxis_config['dtick'] = 'M1'

    fig.update_layout(
        title={
            'text': 'Evolución Temporal de Ventas de Prescripción',
            'x': 0.5,
            'xanchor': 'center',
            'font': {'size': 18, 'color': '#2c3e50'}
        },
        xaxis=xaxis_config,
        yaxis={
            'title': 'Ventas (€)',
            'showgrid': True,
            'gridcolor': '#ecf0f1',
            'linecolor': '#bdc3c7',
            'tickformat': ',.0f',
            'separatethousands': True
        },
        hovermode='x unified',
        legend={
            'title': {'text': 'Categorías (click para mostrar/ocultar)'},
            'orientation': 'h',  # Horizontal debajo del gráfico
            'yanchor': 'top',
            'y': -0.15,  # Debajo del gráfico
            'xanchor': 'center',
            'x': 0.5,  # Centrada
            'font': {'size': 10}
        },
        height=500,
        margin={'l': 80, 'r': 40, 't': 60, 'b': 120},  # Más espacio abajo para leyenda
        plot_bgcolor='white',
        paper_bgcolor='white'
    )

    return dcc.Graph(
        id="prescription-evolution-chart",
        figure=fig,
        config={
            'displayModeBar': True,
            'displaylogo': False,
            'modeBarButtonsToRemove': ['lasso2d', 'select2d'],
            'locale': 'es'
        }
    )
