"""
Componente Índice de Estacionalidad Mensual para Tab 3 Seasonality.

Issue #489: Visualización del índice estacional por mes.
Issue #529: Manejo de meses incompletos con indicador visual "(parcial)".

Características:
- Bar chart horizontal con 12 meses
- Colores: Verde (>1.0 = temporada alta), Rojo (<1.0 = temporada baja)
- Línea de referencia en 1.0 (promedio anual)
- Tooltips con ventas promedio y unidades
- Meses incompletos marcados con "(parcial)" y color diferente
"""

import logging
from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html

from utils.helpers import format_currency

logger = logging.getLogger(__name__)

# Configuración del gráfico
CHART_CONFIG = {
    "height": 400,
    "color_high": "#28a745",   # Verde para índice > 1.0
    "color_low": "#dc3545",    # Rojo para índice < 1.0
    "color_neutral": "#6c757d",  # Gris para índice ≈ 1.0
    "color_partial": "#ffc107",  # Issue #529: Amarillo para meses parciales
    "reference_line": 1.0,
}


def create_monthly_index_chart(
    monthly_index: List[Dict[str, Any]],
    summary: Optional[Dict[str, Any]] = None,
) -> go.Figure:
    """
    Crea un gráfico de barras horizontales del índice estacional.

    Issue #529: Meses parciales se muestran con color amarillo y etiqueta "(parcial)".

    Args:
        monthly_index: Lista de 12 puntos con month_name, index, avg_sales, is_partial
        summary: Resumen con peak_month, low_month

    Returns:
        Figura Plotly con el gráfico de barras
    """
    if not monthly_index:
        return _create_empty_chart()

    # Ordenar por mes (1-12)
    sorted_data = sorted(monthly_index, key=lambda x: x.get("month", 0))

    # Issue #529: Extraer datos incluyendo info de meses parciales
    months = []
    indices = []
    avg_sales = []
    avg_units = []
    is_partial_list = []

    for d in sorted_data:
        is_partial = d.get("is_partial", False)
        month_name = d.get("month_name", f"Mes {d.get('month', 0)}")

        # Issue #529: Añadir "(parcial)" al nombre del mes si está incompleto
        if is_partial:
            month_name = f"{month_name} (parcial)"

        months.append(month_name)
        # Issue #529: Usar 0 para índice None (meses parciales)
        indices.append(d.get("index") if d.get("index") is not None else 0)
        avg_sales.append(d.get("avg_sales", 0))
        avg_units.append(d.get("avg_units", 0))
        is_partial_list.append(is_partial)

    # Issue #529: Determinar colores según índice Y si es parcial
    colors = []
    for i, idx in enumerate(indices):
        if is_partial_list[i]:
            # Mes parcial: color amarillo distintivo
            colors.append(CHART_CONFIG["color_partial"])
        elif idx > 1.05:
            colors.append(CHART_CONFIG["color_high"])
        elif idx < 0.95:
            colors.append(CHART_CONFIG["color_low"])
        else:
            colors.append(CHART_CONFIG["color_neutral"])

    # Issue #529: Crear texto hover con info de días si es parcial
    hover_texts = []
    for i, d in enumerate(sorted_data):
        is_partial = d.get("is_partial", False)

        if is_partial:
            days_with_data = d.get("days_with_data", 0)
            days_in_month = d.get("days_in_month", 31)
            hover_texts.append(
                f"<b>{d.get('month_name', months[i])} (PARCIAL)</b><br>"
                f"Datos: {days_with_data} de {days_in_month} días<br>"
                f"Ventas acumuladas: {format_currency(avg_sales[i])}<br>"
                f"Unidades: {avg_units[i]:,}<br>"
                f"<i>Índice no calculado (datos incompletos)</i>"
            )
        else:
            hover_texts.append(
                f"<b>{d.get('month_name', months[i])}</b><br>"
                f"Índice: {indices[i]:.2f}<br>"
                f"Ventas promedio: {format_currency(avg_sales[i])}<br>"
                f"Unidades promedio: {avg_units[i]:,}"
            )

    # Crear figura
    fig = go.Figure()

    # Añadir barras
    fig.add_trace(
        go.Bar(
            y=months,
            x=indices,
            orientation="h",
            marker=dict(
                color=colors,
                line=dict(width=0),
            ),
            # Issue #529: Mostrar "-" para meses parciales
            text=["-" if is_partial_list[i] else f"{idx:.2f}" for i, idx in enumerate(indices)],
            textposition="outside",
            textfont=dict(size=11),
            hovertemplate="%{customdata}<extra></extra>",
            customdata=hover_texts,
        )
    )

    # Añadir línea de referencia (índice = 1.0)
    fig.add_vline(
        x=CHART_CONFIG["reference_line"],
        line=dict(color="black", width=2, dash="dash"),
        annotation=dict(
            text="Promedio anual",
            font=dict(size=10),
            xanchor="left",
        ),
    )

    # Configurar layout
    fig.update_layout(
        title={
            "text": "Índice de Estacionalidad Mensual",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(
            title="Índice (1.0 = promedio)",
            range=[0, max(indices) * 1.2] if indices else [0, 2],
            showgrid=True,
            gridcolor="rgba(0,0,0,0.1)",
        ),
        yaxis=dict(
            title="",
            autorange="reversed",  # Enero arriba
            categoryorder="array",
            categoryarray=months,
        ),
        height=CHART_CONFIG["height"],
        margin=dict(l=100, r=80, t=60, b=60),
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
        showlegend=False,
    )

    # Añadir anotaciones para pico y valle si están disponibles
    if summary:
        peak_month = summary.get("peak_month_name")
        low_month = summary.get("low_month_name")

        if peak_month and peak_month in months:
            peak_idx = months.index(peak_month)
            fig.add_annotation(
                x=indices[peak_idx],
                y=peak_month,
                text="↑ Pico",
                showarrow=False,
                xanchor="left",
                xshift=40,
                font=dict(color=CHART_CONFIG["color_high"], size=10),
            )

        if low_month and low_month in months:
            low_idx = months.index(low_month)
            fig.add_annotation(
                x=indices[low_idx],
                y=low_month,
                text="↓ Valle",
                showarrow=False,
                xanchor="left",
                xshift=40,
                font=dict(color=CHART_CONFIG["color_low"], size=10),
            )

    return fig


def _create_empty_chart() -> go.Figure:
    """Crea un gráfico vacío con mensaje."""
    fig = go.Figure()

    fig.add_annotation(
        text="No hay datos suficientes para calcular el índice estacional<br>"
             "(Se requieren al menos 12 meses de datos)",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Índice de Estacionalidad Mensual",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=CHART_CONFIG["height"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def create_monthly_index_container(graph_id: str = "seasonality-monthly-index-chart") -> dbc.Card:
    """
    Crea el contenedor del gráfico de índice mensual con card y loading.

    Args:
        graph_id: ID del componente Graph

    Returns:
        Card con el gráfico
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="mdi mdi-calendar-month me-2"),
                    "Estacionalidad por Mes",
                ],
                className="bg-light",
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="loading-monthly-index",
                        type="circle",
                        children=[
                            dcc.Graph(
                                id=graph_id,
                                config={
                                    "displayModeBar": False,
                                    "responsive": True,
                                },
                                style={"height": f"{CHART_CONFIG['height']}px"},
                            ),
                        ],
                    ),
                    # Issue #529: Leyenda con indicador de meses parciales
                    html.Div(
                        id="monthly-index-legend",
                        className="d-flex justify-content-center flex-wrap gap-3 mt-2",
                        children=[
                            html.Span(
                                [
                                    html.Span(
                                        "●",
                                        style={"color": CHART_CONFIG["color_high"], "fontSize": "1.2rem"},
                                    ),
                                    " Temporada alta (>1.0)",
                                ],
                                className="text-muted",
                                style={"fontSize": "0.85rem"},
                            ),
                            html.Span(
                                [
                                    html.Span(
                                        "●",
                                        style={"color": CHART_CONFIG["color_low"], "fontSize": "1.2rem"},
                                    ),
                                    " Temporada baja (<1.0)",
                                ],
                                className="text-muted",
                                style={"fontSize": "0.85rem"},
                            ),
                            html.Span(
                                [
                                    html.Span(
                                        "●",
                                        style={"color": CHART_CONFIG["color_partial"], "fontSize": "1.2rem"},
                                    ),
                                    " Mes incompleto",
                                ],
                                className="text-muted",
                                style={"fontSize": "0.85rem"},
                            ),
                        ],
                    ),
                    # Issue #529: Nota dinámica sobre meses parciales (poblada por callback)
                    html.Div(id="monthly-index-partial-note", className="text-center mt-1"),
                ],
            ),
        ],
        className="shadow-sm h-100",
    )


def create_data_quality_alert(data_quality: Optional[Dict[str, Any]]) -> Optional[dbc.Alert]:
    """
    Crea una alerta si la calidad de datos es insuficiente.

    Issue #529: Ahora también muestra info sobre meses parciales.

    Args:
        data_quality: Indicadores de calidad de datos

    Returns:
        Alert de Bootstrap si hay problemas, None si todo OK
    """
    if not data_quality:
        return None

    months_with_data = data_quality.get("months_with_data", 0)
    sufficient_data = data_quality.get("sufficient_data", False)
    # Issue #529: Info sobre meses parciales
    months_partial = data_quality.get("months_partial", 0)
    partial_months_info = data_quality.get("partial_months", [])

    # Construir mensaje de meses parciales
    partial_msg = ""
    if months_partial > 0 and partial_months_info:
        partial_names = [m.get("month_name", "") for m in partial_months_info]
        if len(partial_names) == 1:
            partial_msg = f" {partial_names[0]} tiene datos incompletos (excluido del índice)."
        else:
            partial_msg = f" {', '.join(partial_names)} tienen datos incompletos (excluidos del índice)."

    if sufficient_data and months_partial == 0:
        return None

    # Determinar mensaje principal
    if not sufficient_data:
        main_msg = f"Datos parciales: {months_with_data}/12 meses disponibles."
    else:
        main_msg = ""

    return dbc.Alert(
        html.Span(
            [
                html.I(className="mdi mdi-alert-circle me-2"),
                main_msg,
                partial_msg if partial_msg else (
                    " Para un análisis preciso se recomiendan al menos 12 meses de datos históricos."
                    if not sufficient_data else ""
                ),
            ]
        ),
        color="warning",
        className="mb-3",
    )


def create_partial_month_note(monthly_index: List[Dict[str, Any]]) -> Optional[html.Div]:
    """
    Issue #529: Crea nota informativa sobre meses parciales.

    Args:
        monthly_index: Lista de datos mensuales

    Returns:
        Div con nota sobre meses parciales, o None si no hay
    """
    if not monthly_index:
        return None

    partial_months = [
        m for m in monthly_index
        if m.get("is_partial", False)
    ]

    if not partial_months:
        return None

    # Construir texto informativo
    notes = []
    for pm in partial_months:
        month_name = pm.get("month_name", "Mes")
        days_with = pm.get("days_with_data", 0)
        days_total = pm.get("days_in_month", 31)
        notes.append(f"{month_name}: {days_with}/{days_total} días")

    return html.Div(
        [
            html.Small(
                [
                    html.I(className="mdi mdi-information-outline me-1"),
                    "Meses incompletos: ",
                    ", ".join(notes),
                    " (excluidos del cálculo del índice)",
                ],
                className="text-muted fst-italic",
            )
        ],
        className="text-center",
    )
