"""
Componente Heatmap Día × Hora para Tab 3 Seasonality.

Issue #489: Visualización de patrones de venta por día de semana y hora.
Issue #528: Invertir orientación - días en columnas, horas en filas.

Características:
- Eje X: Días de la semana (L, M, X, J, V, S, D)
- Eje Y: Horas del día (8-21 horas típicas de farmacia)
- Valor: Ventas totales en ese slot
- Colorscale: Blues (más oscuro = más ventas)
"""

import logging
from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html

from utils.helpers import format_currency

logger = logging.getLogger(__name__)

# Configuración del heatmap
HEATMAP_CONFIG = {
    "colorscale": "Blues",
    "min_hour": 8,   # 8:00 AM
    "max_hour": 21,  # 9:00 PM
    "height": 400,   # Aumentado para mejor visualización vertical
}

# Nombres de días en español (completos para hover)
WEEKDAY_LABELS = ["Lunes", "Martes", "Miércoles", "Jueves", "Viernes", "Sábado", "Domingo"]

# Abreviaturas de días en español (Issue #528)
WEEKDAY_ABBREV = ["L", "M", "X", "J", "V", "S", "D"]

# Horas típicas de farmacia
HOUR_LABELS = [f"{h:02d}:00" for h in range(HEATMAP_CONFIG["min_hour"], HEATMAP_CONFIG["max_hour"] + 1)]


def create_hourly_heatmap_chart(
    heatmap_data: List[Dict[str, Any]],
    summary: Optional[Dict[str, Any]] = None,
) -> go.Figure:
    """
    Crea un gráfico heatmap de ventas por día y hora.

    Issue #528: Orientación invertida - días en columnas (X), horas en filas (Y).

    Args:
        heatmap_data: Lista de celdas con weekday, hour, sales
        summary: Resumen con peak_hour, peak_day, total_sales

    Returns:
        Figura Plotly con el heatmap
    """
    if not heatmap_data:
        return _create_empty_heatmap()

    # Preparar matriz 14x7 (horas x días) - Issue #528: orientación invertida
    # Cada fila es una hora, cada columna es un día
    num_hours = len(HOUR_LABELS)
    matrix = [[0.0 for _ in range(7)] for _ in range(num_hours)]
    units_matrix = [[0 for _ in range(7)] for _ in range(num_hours)]

    # Rellenar matriz con datos
    for cell in heatmap_data:
        weekday_idx = cell.get("weekday", 1) - 1  # 0-indexed (0=Lunes)
        hour = cell.get("hour", 0)

        # Filtrar horas fuera de rango
        if hour < HEATMAP_CONFIG["min_hour"] or hour > HEATMAP_CONFIG["max_hour"]:
            continue

        hour_idx = hour - HEATMAP_CONFIG["min_hour"]

        if 0 <= weekday_idx < 7 and 0 <= hour_idx < num_hours:
            # Matriz[hora][día] para tener horas en filas y días en columnas
            matrix[hour_idx][weekday_idx] = float(cell.get("sales", 0))
            units_matrix[hour_idx][weekday_idx] = int(cell.get("units", 0))

    # Crear texto para hover (horas en filas, días en columnas)
    hover_text = []
    for hour_idx, hour_label in enumerate(HOUR_LABELS):
        row_text = []
        for day_idx, day_name in enumerate(WEEKDAY_LABELS):
            sales = matrix[hour_idx][day_idx]
            units = units_matrix[hour_idx][day_idx]
            row_text.append(
                f"<b>{day_name}</b> {hour_label}<br>"
                f"Ventas: {format_currency(sales)}<br>"
                f"Unidades: {units:,}"
            )
        hover_text.append(row_text)

    # Crear figura con ejes invertidos (Issue #528)
    fig = go.Figure(
        data=go.Heatmap(
            z=matrix,
            x=WEEKDAY_ABBREV,   # Días en X (abreviaturas: L, M, X, J, V, S, D)
            y=HOUR_LABELS,      # Horas en Y
            colorscale=HEATMAP_CONFIG["colorscale"],
            hovertemplate="%{text}<extra></extra>",
            text=hover_text,
            showscale=True,
            colorbar=dict(
                title=dict(text="Ventas (€)", side="right"),
                thickness=15,
                len=0.9,
            ),
        )
    )

    # Marcar pico si hay datos de summary
    if summary and summary.get("peak_hour") is not None and summary.get("peak_day") is not None:
        peak_hour = summary["peak_hour"]
        peak_day_idx = summary["peak_day"] - 1  # 0-indexed

        # Verificar que está en rango
        if (HEATMAP_CONFIG["min_hour"] <= peak_hour <= HEATMAP_CONFIG["max_hour"]
                and 0 <= peak_day_idx < 7):
            peak_hour_label = f"{peak_hour:02d}:00"
            peak_day_abbrev = WEEKDAY_ABBREV[peak_day_idx]

            # Añadir anotación para el pico (ejes invertidos)
            fig.add_annotation(
                x=peak_day_abbrev,   # Día en X
                y=peak_hour_label,   # Hora en Y
                text="★",
                showarrow=False,
                font=dict(size=20, color="gold"),
            )

    # Configurar layout (Issue #528: ejes invertidos)
    fig.update_layout(
        title={
            "text": "Patrón de Ventas por Día y Hora",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(
            title="Día de la semana",
            tickangle=0,
            side="bottom",
        ),
        yaxis=dict(
            title="Hora del día",
            autorange="reversed",  # 08:00 arriba, 21:00 abajo
        ),
        height=HEATMAP_CONFIG["height"],
        margin=dict(l=80, r=80, t=60, b=60),
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def _create_empty_heatmap() -> go.Figure:
    """Crea un heatmap vacío con mensaje."""
    fig = go.Figure()

    fig.add_annotation(
        text="No hay datos disponibles para el período seleccionado",
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color="gray"),
    )

    fig.update_layout(
        title={
            "text": "Patrón de Ventas por Día y Hora",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 16},
        },
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
        height=HEATMAP_CONFIG["height"],
        paper_bgcolor="rgba(0,0,0,0)",
        plot_bgcolor="rgba(0,0,0,0)",
    )

    return fig


def create_heatmap_container(graph_id: str = "seasonality-hourly-heatmap") -> dbc.Card:
    """
    Crea el contenedor del heatmap con card y loading.

    Args:
        graph_id: ID del componente Graph

    Returns:
        Card con el gráfico
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="mdi mdi-clock-outline me-2"),
                    "Distribución Horaria de Ventas",
                ],
                className="bg-light",
            ),
            dbc.CardBody(
                [
                    dcc.Loading(
                        id="loading-heatmap",
                        type="circle",
                        children=[
                            dcc.Graph(
                                id=graph_id,
                                config={
                                    "displayModeBar": False,
                                    "responsive": True,
                                },
                                style={"height": f"{HEATMAP_CONFIG['height']}px"},
                            ),
                        ],
                    ),
                    html.Div(
                        id="heatmap-peak-info",
                        className="text-center text-muted mt-2",
                        style={"fontSize": "0.85rem"},
                    ),
                ],
            ),
        ],
        className="shadow-sm h-100",
    )


def create_peak_info_text(summary: Optional[Dict[str, Any]]) -> str:
    """
    Genera texto informativo sobre el pico de ventas.

    Args:
        summary: Resumen con peak_hour, peak_day, total_sales

    Returns:
        Texto descriptivo del pico
    """
    if not summary or summary.get("peak_hour") is None:
        return "Aplica filtros para ver el análisis de patrones horarios"

    peak_day_name = summary.get("peak_day_name", "")
    peak_hour = summary.get("peak_hour", 0)
    peak_sales = summary.get("peak_sales", 0)

    return (
        f"★ Pico de ventas: {peak_day_name} a las {peak_hour:02d}:00 "
        f"({format_currency(peak_sales)} en el período)"
    )
