# frontend/components/prescription/category_patterns_chart.py
"""
Componente para visualización de patrones de estacionalidad por categoría.

Issue #499: Fase 3 - Estacionalidad por Categoría ATC.

Muestra:
- Selector de categoría
- Gráfico de barras con índices mensuales por categoría
- Tabla resumen con picos y valles
"""

from typing import Any, Dict, List, Optional

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import dcc, html
from dash_iconify import DashIconify

from styles.design_tokens import COLORS, SPACING


def create_category_patterns_container(
    container_id: str = "category-patterns-container",
) -> dbc.Card:
    """
    Crea el contenedor principal para patrones por categoría.

    Args:
        container_id: ID del contenedor

    Returns:
        dbc.Card: Contenedor con selector y gráfico
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:chart-timeline-variant-shimmer",
                                width=24,
                                color=COLORS["info"],
                                className="me-2",
                            ),
                            html.Span(
                                "Estacionalidad por Categoría",
                                style={"fontWeight": "600"},
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}",
                },
            ),
            dbc.CardBody(
                [
                    # Row 1: Selector de categoría
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Label(
                                        "Seleccionar Categoría:",
                                        className="mb-2 fw-bold",
                                    ),
                                    dcc.Dropdown(
                                        id="category-patterns-selector",
                                        placeholder="Selecciona una categoría...",
                                        clearable=False,
                                        className="mb-3",
                                    ),
                                ],
                                width=12,
                                lg=6,
                            ),
                            dbc.Col(
                                [
                                    # Resumen de la categoría seleccionada
                                    html.Div(
                                        id="category-patterns-summary",
                                        className="mb-3",
                                    ),
                                ],
                                width=12,
                                lg=6,
                            ),
                        ],
                    ),
                    # Row 2: Gráfico de barras mensual
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    dcc.Graph(
                                        id="category-patterns-chart",
                                        config={
                                            "displayModeBar": False,
                                            "responsive": True,
                                        },
                                        style={"height": "350px"},
                                    ),
                                ],
                                width=12,
                            ),
                        ],
                    ),
                ],
                id=container_id,
            ),
        ],
        className="shadow-sm mb-4",
    )


def create_category_patterns_chart(
    monthly_indices: List[Dict[str, Any]],
    category_display: str,
    peak_period: str,
) -> go.Figure:
    """
    Crea gráfico de barras con índices mensuales para una categoría.

    Args:
        monthly_indices: Lista de 12 índices mensuales
        category_display: Nombre de la categoría para el título
        peak_period: Período de pico (ej: "Nov-Feb")

    Returns:
        go.Figure: Gráfico Plotly
    """
    if not monthly_indices:
        return _create_empty_chart("No hay datos disponibles")

    # Extraer datos
    months = [m["month_name"][:3] for m in monthly_indices]
    indices = [m["index"] for m in monthly_indices]

    # Colores según valor (verde > 1.0, rojo < 1.0)
    colors = [
        COLORS["success"] if idx > 1.0 else COLORS["danger"] if idx < 1.0 else COLORS["secondary"]
        for idx in indices
    ]

    fig = go.Figure()

    # Barras de índices
    fig.add_trace(
        go.Bar(
            x=months,
            y=indices,
            marker_color=colors,
            text=[f"{idx:.2f}" for idx in indices],
            textposition="outside",
            textfont=dict(size=10),
            hovertemplate=(
                "<b>%{x}</b><br>"
                "Índice: %{y:.2f}<br>"
                "<extra></extra>"
            ),
        )
    )

    # Línea de referencia (promedio = 1.0)
    fig.add_hline(
        y=1.0,
        line_dash="dash",
        line_color=COLORS["secondary"],
        annotation_text="Promedio (1.0)",
        annotation_position="bottom right",
    )

    # Layout
    fig.update_layout(
        title={
            "text": f"Índice de Estacionalidad: {category_display}",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 14},
        },
        xaxis_title="Mes",
        yaxis_title="Índice (1.0 = promedio)",
        xaxis=dict(
            tickangle=0,
            tickfont=dict(size=10),
        ),
        yaxis=dict(
            range=[0, max(indices) * 1.2 if indices else 2],
            tickfont=dict(size=10),
        ),
        margin=dict(l=50, r=30, t=50, b=50),
        paper_bgcolor="white",
        plot_bgcolor="white",
        showlegend=False,
        height=350,
    )

    # Grid horizontal sutil
    fig.update_yaxes(
        showgrid=True,
        gridwidth=1,
        gridcolor=COLORS["bg_tertiary"],
    )

    return fig


def create_category_comparison_chart(
    category_patterns: List[Dict[str, Any]],
    selected_month: Optional[int] = None,
) -> go.Figure:
    """
    Crea gráfico de barras horizontales comparando estacionalidad entre categorías.

    Args:
        category_patterns: Lista de patrones por categoría
        selected_month: Mes seleccionado para comparar (1-12), None = fuerza estacional

    Returns:
        go.Figure: Gráfico Plotly
    """
    if not category_patterns:
        return _create_empty_chart("No hay datos de categorías")

    # Limitar a top 8 categorías para legibilidad
    patterns = category_patterns[:8]

    categories = [p["category_display"] for p in patterns]

    if selected_month:
        # Mostrar índice del mes seleccionado
        values = []
        for p in patterns:
            month_data = next(
                (m for m in p["monthly_indices"] if m["month"] == selected_month),
                None,
            )
            values.append(month_data["index"] if month_data else 0)
        title_suffix = f" - {patterns[0]['monthly_indices'][selected_month - 1]['month_name']}"
    else:
        # Mostrar fuerza estacional (diferencia pico - valle)
        values = [p["seasonality_strength"] for p in patterns]
        title_suffix = " - Fuerza Estacional"

    # Colores por intensidad
    max_val = max(values) if values else 1
    colors = [
        f"rgba(13, 110, 253, {min(v / max_val + 0.3, 1)})"
        for v in values
    ]

    fig = go.Figure()

    fig.add_trace(
        go.Bar(
            y=categories,
            x=values,
            orientation="h",
            marker_color=colors,
            text=[f"{v:.2f}" for v in values],
            textposition="outside",
            textfont=dict(size=10),
            hovertemplate=(
                "<b>%{y}</b><br>"
                f"{'Índice' if selected_month else 'Fuerza'}: %{{x:.2f}}<br>"
                "<extra></extra>"
            ),
        )
    )

    if not selected_month:
        # Línea de referencia para fuerza = 0.5 (moderada)
        fig.add_vline(
            x=0.5,
            line_dash="dash",
            line_color=COLORS["secondary"],
            annotation_text="Moderada",
            annotation_position="top",
        )

    fig.update_layout(
        title={
            "text": f"Comparativa de Categorías{title_suffix}",
            "x": 0.5,
            "xanchor": "center",
            "font": {"size": 14},
        },
        xaxis_title="Fuerza Estacional" if not selected_month else "Índice",
        yaxis=dict(
            autorange="reversed",
            tickfont=dict(size=10),
        ),
        margin=dict(l=150, r=50, t=50, b=50),
        paper_bgcolor="white",
        plot_bgcolor="white",
        showlegend=False,
        height=350,
    )

    fig.update_xaxes(
        showgrid=True,
        gridwidth=1,
        gridcolor=COLORS["bg_tertiary"],
    )

    return fig


def create_category_summary_card(
    category_data: Dict[str, Any],
) -> dbc.Card:
    """
    Crea tarjeta resumen para la categoría seleccionada.

    Args:
        category_data: Datos del patrón de la categoría

    Returns:
        dbc.Card: Tarjeta con resumen
    """
    if not category_data:
        return html.Div()

    peak_color = COLORS["success"] if category_data["peak_index"] > 1.2 else COLORS["info"]
    low_color = COLORS["danger"] if category_data["low_index"] < 0.8 else COLORS["warning"]

    return dbc.Row(
        [
            dbc.Col(
                dbc.Card(
                    dbc.CardBody(
                        [
                            html.Small("Período Pico", className="text-muted"),
                            html.H5(
                                category_data["peak_period"],
                                style={"color": peak_color},
                                className="mb-0",
                            ),
                        ],
                        className="text-center py-2",
                    ),
                    className="border-0 bg-light",
                ),
                width=4,
            ),
            dbc.Col(
                dbc.Card(
                    dbc.CardBody(
                        [
                            html.Small("Mes Pico", className="text-muted"),
                            html.H5(
                                f"{category_data['peak_month_name']} ({category_data['peak_index']:.2f})",
                                style={"color": peak_color},
                                className="mb-0",
                            ),
                        ],
                        className="text-center py-2",
                    ),
                    className="border-0 bg-light",
                ),
                width=4,
            ),
            dbc.Col(
                dbc.Card(
                    dbc.CardBody(
                        [
                            html.Small("Mes Valle", className="text-muted"),
                            html.H5(
                                f"{category_data['low_month_name']} ({category_data['low_index']:.2f})",
                                style={"color": low_color},
                                className="mb-0",
                            ),
                        ],
                        className="text-center py-2",
                    ),
                    className="border-0 bg-light",
                ),
                width=4,
            ),
        ],
        className="g-2",
    )


def _create_empty_chart(message: str) -> go.Figure:
    """Crea gráfico vacío con mensaje."""
    fig = go.Figure()
    fig.add_annotation(
        text=message,
        xref="paper",
        yref="paper",
        x=0.5,
        y=0.5,
        showarrow=False,
        font=dict(size=14, color=COLORS["secondary"]),
    )
    fig.update_layout(
        xaxis=dict(showgrid=False, zeroline=False, showticklabels=False),
        yaxis=dict(showgrid=False, zeroline=False, showticklabels=False),
        paper_bgcolor="white",
        plot_bgcolor="white",
        height=350,
    )
    return fig
