"""
Componente de visualización: Treemap jerárquico de distribución ATC.

Drill-down interactivo de 5 niveles (A → A02 → A02B → A02BC → A02BC01).
Tamaño proporcional a ventas, color según porcentaje.

Issue #400 - Sprint 3: Visualizaciones
"""

import plotly.graph_objects as go
from dash import dcc
from typing import Dict, List, Tuple


def flatten_atc_nodes(nodes: List[Dict], parent: str = "") -> Tuple[List, List, List, List, List]:
    """
    Aplana árbol ATC jerárquico para Plotly Treemap.

    Args:
        nodes: Lista de nodos con estructura {code, name, sales, children}
        parent: Código del nodo padre (para recursión)

    Returns:
        Tupla (labels, parents, values, text, codes)
    """
    labels = []
    parents = []
    values = []
    text = []
    codes = []

    for node in nodes:
        code = node.get('code', '')
        name = node.get('name', '')
        sales = node.get('sales', 0)
        units = node.get('units', 0)
        percentage = node.get('percentage', 0)
        children = node.get('children', [])

        # Label corto para el treemap (solo código)
        labels.append(code)
        parents.append(parent)
        values.append(sales)
        codes.append(code)

        # Text detallado para hover
        text.append(
            f"{code} - {name}<br>"
            f"€{sales:,.2f} ({percentage:.1f}%)<br>"
            f"{units:,.0f} unidades"
        )

        # Recursión para hijos
        if children:
            child_labels, child_parents, child_values, child_text, child_codes = flatten_atc_nodes(
                children, parent=code
            )
            labels.extend(child_labels)
            parents.extend(child_parents)
            values.extend(child_values)
            text.extend(child_text)
            codes.extend(child_codes)

    return labels, parents, values, text, codes


def create_atc_treemap(atc_distribution_data: Dict, level: int = 1) -> dcc.Graph:
    """
    Crea treemap jerárquico de distribución ATC.

    Click en nodo para drill-down/drill-up.
    Formato español para números y moneda.

    Args:
        atc_distribution_data: Respuesta de /distribution-by-atc endpoint
        level: Nivel ATC actual (1-5)

    Returns:
        dcc.Graph con treemap interactivo
    """
    if not atc_distribution_data or not atc_distribution_data.get('nodes'):
        return dcc.Graph(
            id="atc-distribution-treemap",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": "No hay datos de distribución ATC disponibles",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": "#6c757d"}
                    }],
                    "xaxis": {"visible": False},
                    "yaxis": {"visible": False},
                    "height": 600
                }
            },
            config={"displayModeBar": False}
        )

    nodes = atc_distribution_data.get('nodes', [])
    total_sales = atc_distribution_data.get('total_sales', 0)
    current_level = atc_distribution_data.get('level', level)

    # Aplanar árbol jerárquico
    labels, parents, values, text, codes = flatten_atc_nodes(nodes)

    if not labels:
        return dcc.Graph(
            id="atc-distribution-treemap",
            figure={
                "data": [],
                "layout": {
                    "annotations": [{
                        "text": "No hay clasificación ATC disponible para el período seleccionado",
                        "xref": "paper",
                        "yref": "paper",
                        "showarrow": False,
                        "font": {"size": 14, "color": "#6c757d"}
                    }],
                    "height": 600
                }
            }
        )

    # Crear treemap
    fig = go.Figure(go.Treemap(
        labels=labels,
        parents=parents,
        values=values,
        text=text,
        textposition='middle center',
        hovertemplate='%{text}<extra></extra>',
        marker={
            'colorscale': 'Blues',
            'cmid': total_sales / 2 if total_sales > 0 else 0,
            'line': {'width': 2, 'color': 'white'}
        },
        branchvalues='total',
        pathbar={
            'visible': True,
            'thickness': 30,
            'textfont': {'size': 12}
        }
    ))

    # Nivel descriptivo
    level_names = {
        1: "Nivel 1 - Grupo Anatómico Principal",
        2: "Nivel 2 - Subgrupo Terapéutico",
        3: "Nivel 3 - Subgrupo Farmacológico",
        4: "Nivel 4 - Subgrupo Químico",
        5: "Nivel 5 - Sustancia Química"
    }

    fig.update_layout(
        title={
            'text': f'Distribución por Clasificación ATC<br><sub>{level_names.get(current_level, "")}</sub>',
            'x': 0.5,
            'xanchor': 'center',
            'font': {'size': 18, 'color': '#2c3e50'}
        },
        height=600,
        margin={'t': 80, 'l': 10, 'r': 10, 'b': 10},
        paper_bgcolor='white'
    )

    return dcc.Graph(
        id="atc-distribution-treemap",
        figure=fig,
        config={
            'displayModeBar': True,
            'displaylogo': False,
            'modeBarButtonsToRemove': ['lasso2d', 'select2d'],
            'locale': 'es'
        }
    )
