# frontend/components/prescription/anomaly_alerts.py
"""
Componente para visualización de anomalías detectadas.

Issue #501: Fase 4 - Detector de Anomalías + Alertas.

Muestra:
- KPI card con contador de anomalías
- Tabla de alertas con detalles de cada anomalía
- Filtros de severidad y tipo
"""

from typing import Any, Dict, List

import dash_bootstrap_components as dbc
from dash import html
from dash_iconify import DashIconify

from styles.design_tokens import COLORS, SPACING


def create_anomaly_kpi_card(summary: Dict[str, Any]) -> dbc.Card:
    """
    Crea KPI card para el header del Tab 3.

    Args:
        summary: Diccionario con total_anomalies, spikes, drops, high_severity_count

    Returns:
        dbc.Card: KPI card de anomalías
    """
    total = summary.get("total_anomalies", 0)
    high_severity = summary.get("high_severity_count", 0)
    spikes = summary.get("spikes", 0)
    drops = summary.get("drops", 0)

    # Color según severidad
    if high_severity > 0:
        card_color = COLORS["danger"]
        icon_color = COLORS["danger"]
        status_text = f"{high_severity} severas"
    elif total > 0:
        card_color = COLORS["warning"]
        icon_color = COLORS["warning"]
        status_text = f"{spikes} picos, {drops} caídas"
    else:
        card_color = COLORS["success"]
        icon_color = COLORS["success"]
        status_text = "Sin anomalías"

    return dbc.Card(
        [
            dbc.CardBody(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:alert-octagon",
                                width=32,
                                color=icon_color,
                                className="me-2",
                            ),
                            html.Div(
                                [
                                    html.H4(
                                        f"{total}",
                                        className="mb-0",
                                        style={"color": card_color, "fontWeight": "700"},
                                    ),
                                    html.Small(
                                        "Anomalías (90d)",
                                        className="text-muted",
                                    ),
                                ],
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                    html.Hr(className="my-2"),
                    html.Small(
                        status_text,
                        style={"color": card_color},
                    ),
                ],
                className="py-2",
            ),
        ],
        className="h-100 shadow-sm",
        style={"borderLeft": f"4px solid {card_color}"},
    )


def create_anomaly_alerts_container(
    container_id: str = "anomaly-alerts-container",
) -> dbc.Card:
    """
    Crea el contenedor principal para alertas de anomalías.

    Args:
        container_id: ID del contenedor

    Returns:
        dbc.Card: Contenedor con controles y tabla
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.Div(
                        [
                            DashIconify(
                                icon="mdi:alert-circle-outline",
                                width=24,
                                color=COLORS["warning"],
                                className="me-2",
                            ),
                            html.Span(
                                "Alertas de Comportamiento Inusual",
                                style={"fontWeight": "600"},
                            ),
                        ],
                        className="d-flex align-items-center",
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}",
                },
            ),
            dbc.CardBody(
                [
                    # Row 1: Filtros y resumen
                    dbc.Row(
                        [
                            # Filtro de severidad
                            dbc.Col(
                                [
                                    html.Label(
                                        "Filtrar por severidad:",
                                        className="mb-2 small text-muted",
                                    ),
                                    dbc.ButtonGroup(
                                        [
                                            dbc.Button(
                                                "Todas",
                                                id="anomaly-filter-all",
                                                color="secondary",
                                                outline=True,
                                                size="sm",
                                                active=True,
                                            ),
                                            dbc.Button(
                                                "Alta",
                                                id="anomaly-filter-high",
                                                color="danger",
                                                outline=True,
                                                size="sm",
                                            ),
                                            dbc.Button(
                                                "Media",
                                                id="anomaly-filter-medium",
                                                color="warning",
                                                outline=True,
                                                size="sm",
                                            ),
                                        ],
                                        className="mb-2",
                                    ),
                                ],
                                width=12,
                                lg=4,
                            ),
                            # Resumen de anomalías
                            dbc.Col(
                                [
                                    html.Div(
                                        id="anomaly-summary-badges",
                                        className="d-flex flex-wrap gap-2 align-items-center",
                                    ),
                                ],
                                width=12,
                                lg=8,
                                className="d-flex align-items-end",
                            ),
                        ],
                        className="mb-3",
                    ),
                    # Row 2: Tabla de alertas
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Div(
                                        id="anomaly-alerts-table-container",
                                        style={
                                            "maxHeight": "400px",
                                            "overflowY": "auto",
                                        },
                                    ),
                                ],
                                width=12,
                            ),
                        ],
                    ),
                ],
                id=container_id,
            ),
        ],
        className="shadow-sm mb-4",
    )


def create_anomaly_summary_badges(summary: Dict[str, Any]) -> html.Div:
    """
    Crea badges con resumen de anomalías.

    Args:
        summary: Diccionario con estadísticas de anomalías

    Returns:
        html.Div: Contenedor con badges
    """
    if not summary:
        return html.Div("Sin datos")

    badges = []

    # Badge de días analizados
    days_analyzed = summary.get("days_analyzed", 0)
    if days_analyzed:
        badges.append(
            dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:calendar-check", width=14, className="me-1"),
                    f"{days_analyzed} días analizados",
                ]),
                color="secondary",
                className="me-2",
            )
        )

    # Badge de picos
    spikes = summary.get("spikes", 0)
    if spikes > 0:
        badges.append(
            dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:trending-up", width=14, className="me-1"),
                    f"{spikes} picos",
                ]),
                color="success",
                className="me-1",
            )
        )

    # Badge de caídas
    drops = summary.get("drops", 0)
    if drops > 0:
        badges.append(
            dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:trending-down", width=14, className="me-1"),
                    f"{drops} caídas",
                ]),
                color="danger",
                className="me-1",
            )
        )

    # Badge de alta severidad
    high_severity = summary.get("high_severity_count", 0)
    if high_severity > 0:
        badges.append(
            dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:alert", width=14, className="me-1"),
                    f"{high_severity} severas",
                ]),
                color="danger",
                className="me-1",
            )
        )

    # Promedio diario
    avg_sales = summary.get("avg_daily_sales", 0)
    if avg_sales > 0:
        badges.append(
            html.Small(
                f"(Media diaria: {avg_sales:,.0f}€)",
                className="text-muted ms-2",
            )
        )

    return html.Div(badges, className="d-flex flex-wrap gap-1 align-items-center")


def create_anomaly_alerts_table(anomalies: List[Dict[str, Any]]) -> dbc.Table:
    """
    Crea tabla con alertas de anomalías.

    Args:
        anomalies: Lista de anomalías detectadas

    Returns:
        dbc.Table: Tabla con datos de anomalías
    """
    if not anomalies:
        return html.Div(
            [
                DashIconify(
                    icon="mdi:check-circle-outline",
                    width=48,
                    color=COLORS["success"],
                    className="mb-2",
                ),
                html.P(
                    "No se detectaron anomalías en el período analizado.",
                    className="text-muted mb-1",
                ),
                html.Small(
                    "El comportamiento de ventas está dentro de lo esperado.",
                    className="text-muted",
                ),
            ],
            className="text-center py-4",
        )

    # Header de la tabla
    header = html.Thead(
        html.Tr(
            [
                html.Th("Fecha", style={"width": "120px"}),
                html.Th("Tipo", style={"width": "80px"}),
                html.Th("Severidad", style={"width": "90px"}),
                html.Th("Ventas", style={"width": "100px", "textAlign": "right"}),
                html.Th("Desviación", style={"width": "90px", "textAlign": "right"}),
                html.Th("Posibles causas"),
            ]
        ),
        className="sticky-top bg-white",
    )

    # Filas de la tabla
    rows = []
    for anomaly in anomalies:
        # Badge de tipo
        if anomaly.get("anomaly_type") == "PICO":
            type_badge = dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:arrow-up-bold", width=12, className="me-1"),
                    "PICO",
                ]),
                color="success",
            )
        else:
            type_badge = dbc.Badge(
                html.Span([
                    DashIconify(icon="mdi:arrow-down-bold", width=12, className="me-1"),
                    "CAÍDA",
                ]),
                color="danger",
            )

        # Badge de severidad
        severity = anomaly.get("severity", "BAJA")
        severity_color = anomaly.get("severity_color", "info")
        severity_badge = dbc.Badge(severity, color=severity_color)

        # Desviación con color
        deviation = anomaly.get("deviation_pct", 0)
        if deviation > 0:
            deviation_text = f"+{deviation:.0f}%"
            deviation_color = COLORS["success"]
        else:
            deviation_text = f"{deviation:.0f}%"
            deviation_color = COLORS["danger"]

        # Posibles causas
        causes = anomaly.get("possible_causes", [])
        causes_text = "; ".join(causes) if causes else "-"

        # Fecha formateada
        date_str = anomaly.get("date", "")
        weekday = anomaly.get("weekday_name", "")
        date_display = f"{date_str} ({weekday[:3]})" if weekday else date_str

        rows.append(
            html.Tr(
                [
                    html.Td(date_display, className="small"),
                    html.Td(type_badge),
                    html.Td(severity_badge),
                    html.Td(
                        f"{anomaly.get('observed_sales', 0):,.0f}€",
                        style={"textAlign": "right"},
                        className="small",
                    ),
                    html.Td(
                        deviation_text,
                        style={"textAlign": "right", "color": deviation_color, "fontWeight": "600"},
                    ),
                    html.Td(causes_text, className="small text-muted"),
                ],
                className=f"table-{severity_color}" if severity == "ALTA" else "",
            )
        )

    body = html.Tbody(rows)

    return dbc.Table(
        [header, body],
        bordered=True,
        hover=True,
        striped=False,
        responsive=True,
        size="sm",
        className="mb-0",
    )


def create_no_data_message() -> html.Div:
    """Mensaje cuando no hay datos suficientes para análisis."""
    return html.Div(
        [
            DashIconify(
                icon="mdi:chart-timeline-variant",
                width=64,
                color=COLORS["text_secondary"],
                className="mb-3",
            ),
            html.H5(
                "Datos insuficientes",
                className="text-muted mb-2",
            ),
            html.P(
                "Se necesitan al menos 14 días de datos para detectar anomalías.",
                className="text-muted mb-0",
                style={"maxWidth": "400px"},
            ),
        ],
        className="text-center py-5",
    )
