"""
Componente Gauge de Penetración de Partners para Issue #428.
Muestra la métrica de penetración: € partners / € total analizable.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html
import plotly.graph_objects as go
from typing import Dict, List, Optional

from components.base import BaseCard, Title
from styles.design_tokens import COLORS, SPACING


def PartnerPenetrationGauge():
    """
    Componente de gauge que muestra la penetración de partners.

    Métrica: % de ventas sustituibles cubiertas por partners seleccionados
    Rangos:
    - Rojo (<30%): Baja penetración
    - Amarillo (30-60%): Penetración media
    - Verde (>60%): Alta penetración

    Returns:
        dbc.Card: Componente completo con gauge y leyenda
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-tachometer-alt me-2", style={"color": COLORS["primary"]}),
                    html.Span(
                        "Penetración de Partners",
                        style={"fontWeight": "600", "fontSize": "1.1rem"}
                    ),
                ],
                style={"backgroundColor": COLORS["bg_tertiary"], "border": "none", "padding": SPACING["m"]}
            ),
            dbc.CardBody(
                [
                    # Loading wrapper
                    dcc.Loading(
                        id="penetration-gauge-loading",
                        type="default",
                        children=[
                            # Gauge chart
                            dcc.Graph(
                                id="partner-penetration-gauge-chart",
                                config={
                                    "displayModeBar": False,
                                    "responsive": True
                                },
                                style={"height": "250px"},
                                figure=create_empty_gauge()
                            ),
                            # Leyenda de partners debajo
                            html.Div(
                                id="penetration-gauge-legend",
                                className="mt-2",
                                children=_create_empty_legend()
                            ),
                        ]
                    ),
                ],
                style={"padding": SPACING["m"]}
            ),
        ],
        className="mb-4 shadow-sm h-100"
    )


def create_empty_gauge() -> go.Figure:
    """
    Crea un gauge vacío (estado inicial).

    Returns:
        go.Figure: Gauge con valor 0 y mensaje inicial
    """
    fig = go.Figure(
        go.Indicator(
            mode="gauge+number",
            value=0,
            number={"suffix": "%", "font": {"size": 36, "color": COLORS["text_primary"]}},
            gauge={
                "axis": {"range": [0, 100], "tickwidth": 1, "tickcolor": COLORS["border_light"]},
                "bar": {"color": COLORS["text_secondary"]},
                "bgcolor": "white",
                "borderwidth": 2,
                "bordercolor": COLORS["border_light"],
                "steps": [
                    {"range": [0, 30], "color": COLORS["danger_light"]},
                    {"range": [30, 60], "color": COLORS["warning_light"]},
                    {"range": [60, 100], "color": COLORS["success_light"]},
                ],
                "threshold": {
                    "line": {"color": COLORS["danger"], "width": 4},
                    "thickness": 0.75,
                    "value": 30
                },
            },
            title={"text": "Cargando datos...", "font": {"size": 14, "color": COLORS["text_secondary"]}}
        )
    )

    fig.update_layout(
        height=250,
        margin=dict(t=40, b=10, l=40, r=40),
        paper_bgcolor="white",
        font={"family": "Inter, sans-serif"}
    )

    return fig


def create_penetration_gauge(
    penetration_percentage: float,
    partners_amount: float,
    total_analyzable_amount: float,
    selected_partners: List[str]
) -> go.Figure:
    """
    Crea el gauge con datos reales de penetración.

    Args:
        penetration_percentage: Porcentaje de penetración (0-100)
        partners_amount: Monto total de ventas con partners (€)
        total_analyzable_amount: Monto total analizable (€)
        selected_partners: Lista de nombres de partners seleccionados

    Returns:
        go.Figure: Gauge configurado con datos reales
    """
    # Determinar color del indicador según rango
    if penetration_percentage < 30:
        bar_color = COLORS["danger"]
        status_text = "Baja penetración"
    elif penetration_percentage < 60:
        bar_color = COLORS["warning"]
        status_text = "Penetración media"
    else:
        bar_color = COLORS["success"]
        status_text = "Alta penetración"

    fig = go.Figure(
        go.Indicator(
            mode="gauge+number",
            value=penetration_percentage,
            number={
                "suffix": "%",
                "font": {"size": 42, "color": bar_color, "weight": "bold"}
            },
            gauge={
                "axis": {
                    "range": [0, 100],
                    "tickwidth": 1,
                    "tickcolor": COLORS["border_light"],
                    "tickmode": "linear",
                    "tick0": 0,
                    "dtick": 20
                },
                "bar": {"color": bar_color, "thickness": 0.75},
                "bgcolor": "white",
                "borderwidth": 2,
                "bordercolor": COLORS["border_light"],
                "steps": [
                    {"range": [0, 30], "color": COLORS["danger_light"]},
                    {"range": [30, 60], "color": COLORS["warning_light"]},
                    {"range": [60, 100], "color": COLORS["success_light"]},
                ],
                "threshold": {
                    "line": {"color": "rgba(0,0,0,0)", "width": 0},
                    "thickness": 0,
                    "value": 0
                },
            },
            title={
                "text": status_text,
                "font": {"size": 14, "color": COLORS["text_secondary"]}
            }
        )
    )

    fig.update_layout(
        height=250,
        margin=dict(t=40, b=10, l=40, r=40),
        paper_bgcolor="white",
        font={"family": "Inter, sans-serif"}
    )

    return fig


def create_partners_legend(selected_partners: List[str], partners_amount: float) -> html.Div:
    """
    Crea la leyenda simplificada debajo del gauge (solo ventas cubiertas).

    Args:
        selected_partners: Lista de nombres de partners seleccionados
        partners_amount: Monto total de ventas con partners (€)

    Returns:
        html.Div: Leyenda con monto de ventas cubiertas
    """
    if not selected_partners:
        return _create_empty_legend()

    # Formatear monto con separador de miles español
    from utils.helpers import format_currency

    return html.Div(
        [
            # Solo monto de ventas cubiertas (sin lista de laboratorios)
            html.Div(
                [
                    html.Small("Ventas cubiertas: ", className="text-muted"),
                    html.Strong(
                        format_currency(partners_amount),
                        style={"color": COLORS["success"], "fontSize": "1rem"}
                    ),
                ],
                className="text-center"
            ),
        ],
        className="p-2",
        style={
            "backgroundColor": COLORS["bg_tertiary"],
            "borderRadius": "4px",
            "border": f"1px solid {COLORS['border_light']}"
        }
    )


def _create_empty_legend() -> html.Div:
    """Crea leyenda vacía (estado inicial)."""
    return html.Div(
        [
            html.I(className="fas fa-info-circle me-2", style={"color": COLORS["info"]}),
            html.Small(
                "Selecciona partners para ver la métrica de penetración",
                className="text-muted"
            ),
        ],
        className="text-center p-3",
        style={
            "backgroundColor": COLORS["bg_tertiary"],
            "borderRadius": "4px",
            "border": f"1px solid {COLORS['border_light']}"
        }
    )
