"""
Componente de Partners Óptimos para Issue #415.
Versión simplificada: solo selector de cantidad + lista compacta de cobertura.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html
import plotly.graph_objects as go
from typing import List, Dict, Optional

from styles.design_tokens import COLORS, SPACING


def OptimalPartnersSection():
    """
    Sección simplificada de partners óptimos con:
    - Lista compacta de partners con cobertura individual
    - Total acumulado visible

    Returns:
        dbc.Card: Componente compacto de partners óptimos
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.I(className="fas fa-bullseye me-2", style={"color": COLORS["primary"]}),
                    html.Span(
                        "Partners Óptimos",
                        style={"fontWeight": "600"}
                    ),
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "border": "none",
                    "padding": f"{SPACING['s']} {SPACING['m']}"
                }
            ),
            dbc.CardBody(
                [
                    # Loading wrapper
                    dcc.Loading(
                        id="optimal-partners-loading",
                        type="default",
                        children=[
                            # Lista compacta de partners con cobertura
                            html.Div(
                                id="optimal-partners-compact-list",
                                children=_create_empty_list()
                            ),
                        ]
                    ),
                ],
                style={"padding": SPACING["m"]}
            ),
        ],
        className="h-100"
    )


def _create_empty_list():
    """Crea lista vacía (estado inicial)."""
    return html.Div(
        [
            html.I(className="fas fa-info-circle me-2", style={"color": COLORS["info"]}),
            html.Small(
                "Aplica filtros para ver partners óptimos",
                className="text-muted"
            ),
        ],
        className="text-center p-3",
        style={
            "backgroundColor": COLORS["bg_tertiary"],
            "borderRadius": "4px",
        }
    )


def create_compact_partners_list(optimal_partners_data: Dict) -> html.Div:
    """
    Crea lista compacta de partners óptimos con cobertura.

    Args:
        optimal_partners_data: Respuesta del endpoint /optimal-partners

    Returns:
        html.Div: Lista compacta de partners con barras de progreso
    """
    partners = optimal_partners_data.get("optimal_partners", [])

    if not partners:
        return _create_empty_list()

    # Calcular cobertura total acumulada
    # NOTA: Backend devuelve valores como decimales (0.57 = 57%), multiplicar por 100 para %
    raw_coverage = partners[-1]["cumulative_coverage"] if partners else 0.0
    # Si el valor es <= 1, es decimal y necesita multiplicarse por 100
    total_coverage = raw_coverage * 100 if raw_coverage <= 1 else raw_coverage

    # Crear lista de partners con barras de progreso
    partner_items = []
    for i, partner in enumerate(partners):
        # Convertir decimales a porcentajes (backend devuelve 0.009 = 0.9%)
        raw_incremental = partner['incremental_coverage']
        raw_cumulative = partner['cumulative_coverage']
        incremental_pct = raw_incremental * 100 if raw_incremental <= 1 else raw_incremental
        cumulative_pct = raw_cumulative * 100 if raw_cumulative <= 1 else raw_cumulative

        partner_items.append(
            html.Div(
                [
                    # Fila: posición + nombre + cobertura
                    html.Div(
                        [
                            # Posición (badge circular pequeño)
                            dbc.Badge(
                                str(i + 1),
                                color="primary",
                                className="me-2",
                                style={
                                    "width": "20px",
                                    "height": "20px",
                                    "borderRadius": "50%",
                                    "display": "inline-flex",
                                    "alignItems": "center",
                                    "justifyContent": "center",
                                    "fontSize": "0.7rem"
                                }
                            ),
                            # Nombre del partner
                            html.Span(
                                partner["partner_name"],
                                style={"fontSize": "0.85rem", "fontWeight": "500"}
                            ),
                            # Cobertura a la derecha
                            html.Span(
                                f"+{incremental_pct:.1f}%",
                                className="ms-auto text-success",
                                style={"fontSize": "0.85rem", "fontWeight": "600"}
                            ),
                        ],
                        className="d-flex align-items-center mb-1"
                    ),
                    # Barra de progreso pequeña
                    dbc.Progress(
                        value=cumulative_pct,
                        max=100,
                        style={"height": "4px"},
                        className="mb-2",
                        color="success" if cumulative_pct >= 60 else (
                            "warning" if cumulative_pct >= 30 else "danger"
                        )
                    ),
                ],
                className="mb-1"
            )
        )

    return html.Div(
        [
            # Lista de partners
            html.Div(partner_items),
            # Separador
            html.Hr(className="my-2"),
            # Total acumulado
            html.Div(
                [
                    html.Small("Cobertura total:", className="text-muted"),
                    html.Span(
                        f" {total_coverage:.1f}%",
                        style={
                            "fontWeight": "700",
                            "fontSize": "1.1rem",
                            "color": COLORS["success"] if total_coverage >= 60 else (
                                COLORS["warning"] if total_coverage >= 30 else COLORS["danger"]
                            )
                        }
                    ),
                ],
                className="text-center"
            ),
        ]
    )
