# frontend/components/navigation.py
"""
Componentes de navegación para la aplicación kaiFarma.
"""

from datetime import datetime
from typing import Dict, List, Optional

import dash_bootstrap_components as dbc
from dash import dcc, html
from utils.auth_helpers import extract_user_info
from utils.constants import COLORS, USER_ROLES


def create_navbar(pharmacy_name: Optional[str] = None, auth_state: Optional[Dict] = None, current_page: str = "dashboard") -> html.Div:
    """
    Navbar funcional con nombre de farmacia centrado y user menu dropdown.

    Issue #154: Mostrar nombre de farmacia permanentemente visible para mantener contexto.
    Issue #154 (parte 2): User menu dropdown en navbar.

    Args:
        pharmacy_name: Nombre de la farmacia a mostrar
        auth_state: Estado de autenticación (user name, role)
        current_page: Página actual (para futuras extensiones)

    Returns:
        Navbar de 60px con nombre de farmacia centrado y user menu
    """
    from styles.design_tokens import COLORS as DS_COLORS, SPACING, TYPOGRAPHY, SHADOWS, BORDER_RADIUS

    # Fallback si no hay nombre
    display_name = pharmacy_name if pharmacy_name else "Cargando farmacia..."

    # Extraer info del usuario usando helper (Issue #154 - Mejora de código)
    user_name, user_role = extract_user_info(auth_state)

    # Traducir rol a español usando constantes
    role_display = USER_ROLES.get(user_role, "Usuario")

    # User menu dropdown
    user_menu = dbc.DropdownMenu(
        [
            # Header con nombre y rol
            dbc.DropdownMenuItem(
                html.Div(
                    [
                        html.Div(
                            user_name,
                            style={
                                "fontWeight": "600",
                                "fontSize": "0.95rem",
                                "color": DS_COLORS["gray_900"],
                            }
                        ),
                        html.Div(
                            role_display,
                            style={
                                "fontSize": "0.75rem",
                                "color": DS_COLORS["gray_500"],
                            }
                        ),
                    ],
                    style={"padding": f"{SPACING['xs']} {SPACING['s']}"}
                ),
                header=True,
            ),
            dbc.DropdownMenuItem(divider=True),
            # Ajustes
            dbc.DropdownMenuItem(
                html.Span([
                    html.I(className="fas fa-cog me-2"),
                    "Ajustes"
                ]),
                href="/ajustes/perfil",
                style={
                    "fontSize": "0.9rem",
                    "padding": f"{SPACING['s']} {SPACING['m']}",
                }
            ),
            # Cerrar Sesión
            dbc.DropdownMenuItem(
                html.Span([
                    html.I(className="fas fa-sign-out-alt me-2"),
                    "Cerrar Sesión"
                ]),
                id="navbar-logout-button",
                n_clicks=0,
                style={
                    "fontSize": "0.9rem",
                    "color": DS_COLORS["danger"],
                    "padding": f"{SPACING['s']} {SPACING['m']}",
                }
            ),
        ],
        label=[
            html.I(className="fas fa-user-circle me-2", style={"fontSize": "1.2rem"}),
            html.Span(user_name.split()[0] if " " in user_name else user_name, className="d-none d-lg-inline"),  # Solo en desktop
        ],
        color="light",
        className="ms-2",
        direction="down",
        style={
            "borderRadius": BORDER_RADIUS["md"],
        }
    )

    return html.Div(
        [
            # Franja decorativa superior de 4px (mantener diseño existente)
            html.Div(
                style={
                    "height": "4px",
                    "background": f"linear-gradient(90deg, {DS_COLORS['primary']} 0%, {DS_COLORS['info']} 100%)",
                    "width": "100%",
                }
            ),
            # Navbar funcional
            dbc.Navbar(
                dbc.Container(
                    [
                        # Nombre de farmacia (izquierda)
                        dbc.Col(
                            html.Div(
                                html.Span(
                                    display_name,
                                    id="navbar-pharmacy-name",  # ID para updates dinámicos
                                    style={
                                        "fontSize": "1.1rem",
                                        "fontWeight": "600",
                                        "color": DS_COLORS["white"],
                                    }
                                )
                            ),
                            className="flex-grow-1",
                        ),
                        # User menu dropdown (derecha)
                        dbc.Col(
                            user_menu,
                            width="auto",
                        ),
                    ],
                    fluid=True,
                    className="d-flex align-items-center"
                ),
                color="dark",  # Gris oscuro (neutral, no compite con sidebar)
                dark=True,
                sticky="top",
                style={
                    "height": "60px",
                    "paddingTop": "0",
                    "paddingBottom": "0",
                    "boxShadow": SHADOWS["sm"],
                }
            ),
        ],
        style={
            "position": "fixed",
            "top": "0",
            "left": "280px",  # Empieza después del sidebar
            "right": "0",
            "zIndex": "1046",  # Mayor que sidebar (1045) para evitar que sidebar tape al navbar
        }
    )


def create_fallback_navbar() -> html.Div:
    """
    Navbar mínimo para estados de error o sesión expirada.

    Garantiza que el usuario SIEMPRE tenga acceso a navegación básica incluso
    cuando auth_state es inconsistente o la sesión expiró.

    Features:
    - Siempre visible (inmune a race conditions de auth)
    - Mensaje "Sesión expirada" para feedback claro
    - Botón "Iniciar Sesión" con id correcto para callbacks
    - UI consistente con navbar principal

    Returns:
        Navbar de 60px con logo y botón de login

    Note:
        Resuelve el problema de "pantalla vacía" cuando auth_guard hace logout
        pero navbar callback aún no se ejecutó (timing issue).
    """
    from styles.design_tokens import COLORS as DS_COLORS, SPACING, SHADOWS

    return html.Div(
        [
            dbc.Navbar(
                dbc.Container(
                    [
                        # Logo/Nombre de la app (izquierda)
                        html.Div(
                            [
                                html.I(
                                    className="fas fa-pills me-2",
                                    style={"fontSize": "1.5rem", "color": DS_COLORS["primary"]}
                                ),
                                html.Span(
                                    "kaiFarma",
                                    className="fw-bold text-white",
                                    style={"fontSize": "1.1rem"}
                                )
                            ],
                            className="d-flex align-items-center"
                        ),

                        # Mensaje + Botón login (derecha)
                        html.Div(
                            [
                                html.Span(
                                    [
                                        html.I(className="fas fa-exclamation-circle me-2"),
                                        "Sesión expirada"
                                    ],
                                    className="text-warning me-3 d-none d-sm-inline",
                                    style={"fontSize": "0.9rem"}
                                ),
                                # Botón de login en navbar fallback
                                # NOTA: NO usar id="navbar-logout-button" aquí porque tiene función diferente
                                # El skeleton global (app.py) tiene el placeholder para navbar-logout-button
                                dbc.Button(
                                    html.Span([
                                        html.I(className="fas fa-sign-in-alt me-2"),
                                        "Iniciar Sesión"
                                    ]),
                                    href="/auth/login",  # Fix: usar ruta canónica
                                    color="primary",
                                    size="sm",
                                    style={"whiteSpace": "nowrap"}
                                )
                            ],
                            className="d-flex align-items-center"
                        )
                    ],
                    fluid=True,
                    className="d-flex justify-content-between align-items-center"
                ),
                color="dark",
                dark=True,
                sticky="top",
                style={
                    "height": "60px",
                    "paddingTop": "0",
                    "paddingBottom": "0",
                    "boxShadow": SHADOWS["sm"],
                }
            )
        ],
        style={
            "position": "fixed",
            "top": "0",
            "left": "280px",  # Mismo offset que navbar principal
            "right": "0",
            "zIndex": "1046",
        }
    )


def create_sidebar(
    current_page: str = "dashboard",
    progress_data: Dict = None,
    catalog_status: Dict = None,
    auth_state: Dict = None,
    storage_usage: Dict = None,  # Issue #420: FREE tier storage indicator
) -> html.Div:
    """
    Crear barra lateral para navegación principal.

    Args:
        current_page: Página actual para resaltar
        progress_data: Datos de progreso de operaciones
        catalog_status: Estado del catálogo (last_update, status, is_syncing)
        auth_state: Estado de autenticación del usuario con rol
        storage_usage: Datos de uso de almacenamiento (Issue #420)

    Returns:
        Componente html.Div con sidebar
    """

    # Definir elementos del menú principal
    menu_items = [
        # ELIMINADO: "ESTADO DEL SISTEMA" - El logo ahora redirige a home (Issue #152)
        {
            "name": "VENTAS",
            "href": "/dashboard",
            "page": "dashboard",
            "icon": "fas fa-chart-line",
            "color": "text-info",
            "subtitle": "Análisis estratégico de ventas",
        },
        {
            "name": "PRESCRIPCIÓN",
            "href": "/prescription",
            "page": "prescription",
            "icon": "fas fa-prescription",
            "color": "text-success",
            "subtitle": "Medicamentos con receta",
        },
        {
            "name": "GENÉRICOS",
            "href": "/generics",
            "page": "generics",
            "icon": "fas fa-pills",
            "color": "text-warning",
            "subtitle": "Oportunidades inteligentes",
        },
        {
            "name": "VENTA LIBRE",
            "href": "/ventalibre",
            "page": "ventalibre",
            "icon": "fas fa-shopping-bag",
            "color": "text-danger",
            "subtitle": "Productos sin receta",
        },
        {
            "name": "CARGAR DATOS",
            "href": "/upload",
            "page": "upload",
            "icon": "fas fa-upload",
            "color": "text-secondary",
            "subtitle": "Upload simplificado",
        },
        {
            "name": "ADMINISTRACIÓN",
            "href": "/admin",
            "page": "admin",
            "icon": "fas fa-user-shield",
            "color": "text-danger",
            "subtitle": "Dashboard de salud y modo desarrollador",
            "admin_only": True,  # Marcar como solo para administradores
        },
    ]

    # Filtrar menú según rol del usuario
    user_role = None
    if auth_state and auth_state.get("authenticated") and auth_state.get("user"):
        user_role = auth_state["user"].get("role")

    # Si no es admin, filtrar items marcados como admin_only
    if user_role != "admin":
        menu_items = [item for item in menu_items if not item.get("admin_only", False)]

    # Crear elementos del menú
    menu_elements = []

    for item in menu_items:
        is_active = current_page == item["page"]

        # Crear el elemento del menú con estilo mejorado
        menu_elements.append(
            dcc.Link(
                html.Div(
                    [
                        html.Div(
                            [
                                html.I(
                                    className=f"{item['icon']} {item['color']}",
                                    style={"fontSize": "1.2rem", "width": "30px"},
                                ),
                                html.Div(
                                    [
                                        html.Div(item["name"], className="fw-bold", style={"fontSize": "0.85rem"}),
                                        html.Small(
                                            item["subtitle"], className="text-muted", style={"fontSize": "0.7rem"}
                                        ),
                                    ],
                                    className="ms-2",
                                ),
                            ],
                            className="d-flex align-items-center",
                        ),
                    ],
                    className=f"sidebar-item p-3 {'active' if is_active else ''}",
                    style={
                        "borderLeft": f"4px solid {COLORS['primary']}" if is_active else "4px solid transparent",
                        "backgroundColor": "rgba(0, 123, 255, 0.05)" if is_active else "transparent",
                        "transition": "all 0.3s ease",
                        "cursor": "pointer",
                    },
                ),
                href=item["href"],
                style={"textDecoration": "none", "color": "inherit"},
            )
        )

    # Preparar badge de estado del catálogo
    catalog_badge = None
    if catalog_status:
        # Determinar estado basado en última actualización
        if catalog_status.get("is_syncing"):
            badge_color = "info"
            badge_icon = "fas fa-sync-alt fa-spin"
            badge_text = "Sincronizando..."
            tooltip_text = "El catálogo se está actualizando en este momento"
        else:
            last_update = catalog_status.get("last_update")
            if last_update:
                try:
                    last_update_dt = datetime.fromisoformat(last_update)
                    days_since_update = (datetime.now() - last_update_dt).days

                    if days_since_update < 5:
                        badge_color = "success"
                        badge_icon = "fas fa-check-circle"
                        badge_text = "Actualizado"
                        tooltip_text = f"Última actualización: hace {days_since_update} días"
                    elif days_since_update < 10:
                        badge_color = "warning"
                        badge_icon = "fas fa-exclamation-circle"
                        badge_text = f"{days_since_update} días"
                        tooltip_text = f"Catálogo desactualizado: {days_since_update} días sin actualizar"
                    else:
                        badge_color = "danger"
                        badge_icon = "fas fa-times-circle"
                        badge_text = f"{days_since_update} días"
                        tooltip_text = f"Catálogo muy desactualizado: {days_since_update} días sin actualizar"
                except:
                    badge_color = "secondary"
                    badge_icon = "fas fa-question-circle"
                    badge_text = "Sin datos"
                    tooltip_text = "No hay información de actualización disponible"
            else:
                badge_color = "secondary"
                badge_icon = "fas fa-question-circle"
                badge_text = "Sin datos"
                tooltip_text = "El catálogo no ha sido inicializado"

        catalog_badge = dbc.Badge(
            [html.I(className=f"{badge_icon} me-1", style={"fontSize": "0.7rem"}), badge_text],
            color=badge_color,
            pill=True,
            className="ms-auto",
            id="catalog-status-badge",
            style={"fontSize": "0.7rem", "padding": "0.25rem 0.5rem"},
        )

    return html.Div(
        [
            # Header del sidebar con logo CLICKEABLE → Home (Issue #152)
            dcc.Link(
                html.Div(
                    [
                        html.I(className="fas fa-pills me-2", style={"fontSize": "1.5rem"}),
                        html.Strong("kaiFarma", className="fs-4"),
                    ],
                    className="d-flex align-items-center justify-content-center text-white py-4",
                    style={
                        "backgroundColor": COLORS["primary"],
                        "cursor": "pointer",  # Indicar que es clickeable
                        "transition": "background-color 0.3s ease",
                    },
                ),
                href="/home",  # Redirige a homepage
                style={"textDecoration": "none", "color": "inherit"},
                className="sidebar-logo-link",
            ),
            # Estado del catálogo (si está disponible)
            (
                html.Div(
                    [
                        html.Div(
                            [
                                html.Div(
                                    [
                                        html.I(
                                            className="fas fa-database me-2 text-muted", style={"fontSize": "0.9rem"}
                                        ),
                                        html.Span("Catálogo", className="text-muted", style={"fontSize": "0.85rem"}),
                                        catalog_badge,
                                    ],
                                    className="d-flex align-items-center justify-content-between",
                                )
                            ],
                            className="px-3 py-2",
                            style={
                                "backgroundColor": "#f8f9fa",
                                "borderLeft": f"4px solid {COLORS[badge_color] if catalog_badge else '#dee2e6'}",
                                "margin": "0 12px 12px 12px",
                                "borderRadius": "0 4px 4px 0",
                            },
                        ),
                        (
                            dbc.Tooltip(tooltip_text, target="catalog-status-badge", placement="right")
                            if catalog_badge
                            else None
                        ),
                    ]
                )
                if catalog_status
                else None
            ),
            # Storage indicator movido al footer (Issue #420)
            # Menú de navegación
            html.Nav(menu_elements, className="sidebar-nav py-3"),
            # Sección de progreso - SOLO mostrar si hay tarea activa real
            (
                html.Div(
                    [
                        html.Hr(className="mx-3"),
                        html.Div(
                            [
                                html.Div(
                                    [
                                        html.I(className="bi bi-arrow-repeat bi-spin me-2"),
                                        html.Small(progress_data.get("message", "Procesando..."), className="fw-bold"),
                                    ],
                                    className="d-flex align-items-center mb-2",
                                ),
                                dbc.Progress(
                                    id="update-progress-bar",
                                    value=0,
                                    animated=True,
                                    striped=True,
                                    color="info",
                                    className="mb-2",
                                    style={"height": "8px"},
                                ),
                                html.Small(
                                    id="update-progress-text",
                                    children="Iniciando...",
                                    className="text-muted d-block text-center",
                                ),
                            ],
                            className="px-3 py-2",
                            style={
                                "backgroundColor": "rgba(0, 123, 255, 0.05)",
                                "borderRadius": "8px",
                                "margin": "0 12px",
                            },
                        ),
                        html.Hr(className="mx-3"),
                    ],
                    id="progress-section",
                )
                if progress_data and progress_data.get("active") == True
                else html.Div(id="progress-section", style={"display": "none"})
            ),
            # Footer del sidebar con storage + versión (Issue #154, #420)
            html.Div(
                [
                    html.Hr(className="mx-3"),
                    # Issue #420: Storage indicator siempre visible para FREE users
                    (
                        html.Div(
                            [
                                html.Div(
                                    [
                                        html.I(
                                            className="fas fa-hdd me-2 text-muted",
                                            style={"fontSize": "0.85rem"},
                                        ),
                                        html.Small(
                                            "Almacenamiento",
                                            className="text-muted",
                                            style={"fontWeight": "500"},
                                        ),
                                    ],
                                    className="d-flex align-items-center mb-2",
                                ),
                                dbc.Progress(
                                    value=float(storage_usage.get("percentage", 0) or 0) if storage_usage else 0,
                                    color=_get_storage_color(float(storage_usage.get("percentage", 0) or 0) if storage_usage else 0),
                                    className="mb-1",
                                    style={"height": "5px"},
                                ),
                                html.Small(
                                    f"{float(storage_usage.get('total_used_mb', 0) or 0):.0f} / "
                                    f"{int(storage_usage.get('limit_mb', 100) or 100)} MB"
                                    if storage_usage else "Cargando...",
                                    className="text-muted d-block",
                                    style={"fontSize": "0.7rem"},
                                ),
                                html.Small(
                                    "Plan FREE",
                                    className="d-block mt-1",
                                    style={
                                        "fontSize": "0.65rem",
                                        "color": COLORS["info"],
                                        "fontWeight": "500",
                                    },
                                ),
                            ],
                            className="px-3 py-2 mx-3 mb-3",
                            style={
                                "backgroundColor": "#f8f9fa",
                                "borderRadius": "6px",
                                "border": f"1px solid {COLORS['muted']}",  # Issue #442: Fix KeyError 'border_light'
                            },
                        )
                        if storage_usage is not None  # Mostrar si storage_usage fue calculado (incluye {} vacío para FREE)
                        else None
                    ),
                    # Versión
                    html.Div(
                        [
                            html.Small("v1.0.0", className="text-muted"),
                            " | ",
                            html.Small("© 2026 kaiFarma", className="text-muted"),
                        ],
                        className="text-center pb-3",
                    ),
                    # REMOVED: logout-button placeholder ahora está en skeleton global (app.py línea 138-142)
                    # Razón: Evitar IDs duplicados cuando sidebar se renderiza
                ],
                className="mt-auto",
            ),
        ],
        className="sidebar d-flex flex-column",
        id="sidebar",
        style={"backgroundColor": "#ffffff", "borderRight": "1px solid #dee2e6", "overflowY": "auto"},
    )


def _get_storage_color(percentage: float) -> str:
    """
    Determinar color de progress bar según porcentaje de uso (Issue #420).

    Args:
        percentage: Porcentaje de almacenamiento usado (0-100)

    Returns:
        Color Bootstrap: 'success', 'warning', 'danger'
    """
    if percentage >= 80:
        return "danger"
    elif percentage >= 50:
        return "warning"
    return "success"


def _get_storage_border_color(percentage: float) -> str:
    """
    Determinar color de borde izquierdo según porcentaje (Issue #420).

    Args:
        percentage: Porcentaje de almacenamiento usado (0-100)

    Returns:
        Color hex para borderLeft
    """
    if percentage >= 80:
        return COLORS["danger"]
    elif percentage >= 50:
        return COLORS["warning"]
    return COLORS["success"]


def create_breadcrumbs(pages: List[Dict[str, str]]) -> html.Nav:
    """
    Crear breadcrumbs de navegación usando HTML + Bootstrap.

    Args:
        pages: Lista de páginas con 'label' y 'href'

    Returns:
        Componente html.Nav con breadcrumbs
    """

    items = []
    for i, page in enumerate(pages):
        is_last = i == len(pages) - 1

        if is_last:
            # Página actual - sin enlace
            items.append(html.Li(page["label"], className="breadcrumb-item active", **{"aria-current": "page"}))
        else:
            # Páginas anteriores - con enlace
            items.append(
                html.Li(
                    dcc.Link(page["label"], href=page.get("href", "#"), className="text-decoration-none"),
                    className="breadcrumb-item",
                )
            )

    return html.Nav(html.Ol(items, className="breadcrumb bg-transparent px-0 mb-0"), **{"aria-label": "breadcrumb"})


def create_page_header(
    title: str,
    subtitle: Optional[str] = None,
    breadcrumbs: Optional[List[Dict[str, str]]] = None,
    actions: Optional[List] = None,
) -> html.Div:
    """
    Crear header de página con título, breadcrumbs y acciones.

    Args:
        title: Título de la página
        subtitle: Subtítulo opcional
        breadcrumbs: Lista de breadcrumbs
        actions: Lista de botones/acciones

    Returns:
        Componente html.Div con header
    """

    elements = []

    # Breadcrumbs
    if breadcrumbs:
        elements.append(create_breadcrumbs(breadcrumbs))

    # Título y acciones
    title_row = [
        html.Div(
            [html.H1(title, className="h3 mb-0"), html.P(subtitle, className="text-muted mb-0") if subtitle else None],
            className="col",
        )
    ]

    # Añadir acciones si las hay
    if actions:
        title_row.append(html.Div(actions, className="col-auto d-flex gap-2"))

    elements.append(dbc.Row(title_row, className="align-items-center"))

    return html.Div(elements, className="page-header pb-3 mb-4 border-bottom")


def create_mobile_nav(current_page: str = "dashboard") -> html.Div:
    """
    Crear navegación inferior para móviles.

    Args:
        current_page: Página actual

    Returns:
        Componente html.Div con navegación móvil
    """

    nav_items = [
        {"name": "Dashboard", "href": "/", "page": "dashboard", "icon": "fas fa-chart-dashboard"},
        {"name": "Datos", "href": "/upload", "page": "upload", "icon": "fas fa-upload"},
        {"name": "Análisis", "href": "/analysis", "page": "analysis", "icon": "fas fa-chart-line"},
        {"name": "Más", "href": "/config", "page": "config", "icon": "fas fa-ellipsis"},
    ]

    nav_buttons = []
    for item in nav_items:
        is_active = current_page == item["page"]

        nav_buttons.append(
            html.A(
                [html.I(className=f"{item['icon']} d-block"), html.Small(item["name"])],
                href=item["href"],
                className=f"mobile-nav-item {'active' if is_active else ''}",
                style={
                    "flex": "1",
                    "textAlign": "center",
                    "padding": "8px 4px",
                    "textDecoration": "none",
                    "color": COLORS["primary"] if is_active else COLORS["secondary"],
                },
            )
        )

    return html.Div(
        nav_buttons,
        className="mobile-nav d-flex d-lg-none fixed-bottom bg-white shadow-lg border-top",
        style={"height": "60px", "alignItems": "center", "zIndex": "1000"},
    )
