"""
Herramientas de mantenimiento seguras para administradores.
Operaciones de limpieza y verificación que no afectan datos críticos.
"""

from datetime import datetime
from typing import Dict, Literal

import dash_bootstrap_components as dbc
from dash import Input, Output, State, ctx, dcc, html

from ..utils.api_client import APIClient
from .confirmation_modal import create_confirmation_modal
from .toast_manager import error_toast, success_toast

MaintenanceOperation = Literal[
    "clear_cache", "verify_connectivity", "refresh_system_status", "reenrich_sales"
]


class MaintenanceTools:
    """Gestor de herramientas de mantenimiento seguras."""

    # Configuración de operaciones
    OPERATIONS_CONFIG = {
        "clear_cache": {
            "title": "Limpiar Cache",
            "description": "Elimina datos temporales en cache para mejorar rendimiento",
            "icon": "fas fa-broom",
            "color": "warning",
            "risk_level": "low",
            "estimated_time": "5-10 segundos",
            "requires_confirmation": True,
            "endpoint": "/api/v1/admin-tools/maintenance/clear-cache",
            "method": "POST",
        },
        "verify_connectivity": {
            "title": "Verificar Conectividad",
            "description": "Prueba conexiones a servicios externos (CIMA, base de datos, etc.)",
            "icon": "fas fa-network-wired",
            "color": "info",
            "risk_level": "none",
            "estimated_time": "10-15 segundos",
            "requires_confirmation": False,
            "endpoint": "/api/v1/admin-tools/maintenance/verify-connectivity",
            "method": "POST",
        },
        "refresh_system_status": {
            "title": "Actualizar Estado",
            "description": "Fuerza actualización inmediata del estado del sistema",
            "icon": "fas fa-sync-alt",
            "color": "primary",
            "risk_level": "none",
            "estimated_time": "2-5 segundos",
            "requires_confirmation": False,
            "endpoint": "/api/v1/system-unified/health-summary",
            "method": "GET",
            "params": {"force_refresh": True},
        },
        "reenrich_sales": {
            "title": "Re-enriquecer Ventas",
            "description": "Busca ventas en manual_review que ahora tienen match en catálogo y las enriquece",
            "icon": "fas fa-magic",
            "color": "info",
            "risk_level": "low",
            "estimated_time": "30-120 segundos",
            "requires_confirmation": True,
            "endpoint": "/api/v1/reenrichment/execute-all",
            "method": "POST",
        },
    }

    # Niveles de riesgo y sus indicadores
    RISK_INDICATORS = {
        "none": {"badge": "success", "text": "Sin riesgo", "icon": "fas fa-check-circle"},
        "low": {"badge": "warning", "text": "Riesgo bajo", "icon": "fas fa-exclamation-triangle"},
        "medium": {"badge": "danger", "text": "Riesgo medio", "icon": "fas fa-exclamation-circle"},
        "high": {"badge": "danger", "text": "Riesgo alto", "icon": "fas fa-ban"},
    }


def create_maintenance_panel(panel_id: str = "maintenance-tools-panel") -> html.Div:
    """
    Crea el panel principal de herramientas de mantenimiento.

    Args:
        panel_id: ID único del panel
    """
    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H5(
                        [html.I(className="fas fa-tools me-2 text-primary"), "Herramientas de Mantenimiento"],
                        className="mb-0",
                    ),
                    html.Small("Operaciones seguras de limpieza y verificación", className="text-muted"),
                ]
            ),
            dbc.CardBody(
                [
                    # Stores para gestión de estado
                    dcc.Store(id=f"{panel_id}-operation-store", data={}),
                    dcc.Store(id=f"{panel_id}-results-store", data={}),
                    # Modal de confirmación
                    create_confirmation_modal(f"{panel_id}-confirmation-modal", "warning"),
                    # Contenido principal
                    html.Div(id=f"{panel_id}-content", children=[create_maintenance_tools_grid()]),
                ]
            ),
        ],
        className="shadow-sm",
    )


def create_maintenance_tools_grid() -> html.Div:
    """Crea la grilla de herramientas de mantenimiento."""
    tools = []

    for operation_id, config in MaintenanceTools.OPERATIONS_CONFIG.items():
        risk_info = MaintenanceTools.RISK_INDICATORS.get(config["risk_level"], MaintenanceTools.RISK_INDICATORS["none"])

        tool_card = dbc.Col(
            [
                dbc.Card(
                    [
                        dbc.CardBody(
                            [
                                # Header con icono y riesgo
                                html.Div(
                                    [
                                        html.I(className=f"{config['icon']} fa-2x text-{config['color']}"),
                                        dbc.Badge(
                                            html.Div(
                                                [html.I(className=f"{risk_info['icon']} me-1"), risk_info["text"]]
                                            ),
                                            color=risk_info["badge"],
                                            className="ms-auto",
                                        ),
                                    ],
                                    className="d-flex justify-content-between align-items-start mb-3",
                                ),
                                # Título y descripción
                                html.H6(config["title"], className="card-title"),
                                html.P(config["description"], className="card-text text-muted small"),
                                # Información adicional
                                html.Div(
                                    [
                                        html.Small(
                                            [
                                                html.I(className="fas fa-clock me-1"),
                                                f"Tiempo estimado: {config['estimated_time']}",
                                            ],
                                            className="text-muted d-block mb-2",
                                        )
                                    ]
                                ),
                                # Botón de acción
                                dbc.Button(
                                    html.Div([html.I(className=f"{config['icon']} me-2"), "Ejecutar"]),
                                    id={"type": "maintenance-tool-btn", "operation": operation_id},
                                    color=config["color"],
                                    outline=True,
                                    size="sm",
                                    className="w-100",
                                ),
                            ]
                        )
                    ],
                    className="h-100",
                )
            ],
            width=12,
            md=6,
            lg=4,
            className="mb-3",
        )

        tools.append(tool_card)

    return html.Div(
        [
            dbc.Row(tools),
            # Panel de resultados
            html.Hr(),
            html.Div(id="maintenance-results-panel", className="mt-3"),
        ]
    )


def create_operation_result_card(operation_id: str, result_data: Dict, show_details: bool = True) -> dbc.Card:
    """
    Crea una tarjeta con el resultado de una operación de mantenimiento.

    Args:
        operation_id: ID de la operación ejecutada
        result_data: Datos del resultado
        show_details: Si mostrar detalles técnicos
    """
    config = MaintenanceTools.OPERATIONS_CONFIG.get(operation_id, {})
    success = result_data.get("success", False)

    # Determinar color y icono según resultado
    if success:
        color = "success"
        icon = "fas fa-check-circle"
        title = f"✅ {config.get('title', 'Operación')} Completada"
    else:
        color = "danger"
        icon = "fas fa-times-circle"
        title = f"❌ Error en {config.get('title', 'Operación')}"

    # Contenido básico
    content = [html.P(result_data.get("message", "Operación completada"), className="mb-3")]

    # Añadir detalles si están disponibles y se solicitan
    if show_details and result_data.get("details"):
        details = result_data["details"]

        if isinstance(details, dict):
            detail_items = []
            for key, value in details.items():
                detail_items.append(html.Li([html.Strong(f"{key.replace('_', ' ').title()}: "), str(value)]))

            content.append(html.Ul(detail_items, className="small"))
        else:
            content.append(dbc.Alert(str(details), color="info", className="small"))

    # Timestamp
    if result_data.get("timestamp"):
        content.append(html.Small(f"Ejecutado: {result_data['timestamp']}", className="text-muted"))

    return dbc.Alert(
        [html.H6([html.I(className=f"{icon} me-2"), title], className="alert-heading"), html.Div(content)],
        color=color,
        dismissable=True,
        className="mb-3",
    )


def create_connectivity_test_results(results: Dict) -> html.Div:
    """Crea visualización específica para resultados de pruebas de conectividad."""
    if not results or not results.get("connectivity_tests"):
        return html.Div()

    tests = results["connectivity_tests"]
    test_items = []

    for test_name, test_result in tests.items():
        success = test_result.get("success", False)
        response_time = test_result.get("response_time_ms")

        # Icono y color según resultado
        if success:
            icon = "fas fa-check-circle text-success"
            status = "Conectado"
        else:
            icon = "fas fa-times-circle text-danger"
            status = "Error"

        # Información adicional
        details = []
        if response_time is not None:
            details.append(f"{response_time}ms")

        if test_result.get("error"):
            details.append(test_result["error"])

        test_items.append(
            html.Li(
                [
                    html.Div(
                        [
                            html.I(className=icon),
                            html.Span(f" {test_name.replace('_', ' ').title()}", className="ms-2"),
                            html.Span(f" - {status}", className="text-muted"),
                            html.Div(" | ".join(details) if details else "", className="small text-muted"),
                        ]
                    )
                ],
                className="list-group-item border-0 py-2",
            )
        )

    return html.Div(
        [html.H6("Resultados de Conectividad"), html.Ul(test_items, className="list-group list-group-flush")]
    )


# Callbacks del componente
def register_maintenance_tools_callbacks(app):
    """Registra los callbacks para las herramientas de mantenimiento."""

    @app.callback(
        [
            Output("maintenance-tools-panel-confirmation-modal", "is_open"),
            Output("maintenance-tools-panel-operation-store", "data"),
        ],
        [
            Input({"type": "maintenance-tool-btn", "operation": ALL}, "n_clicks"),
            Input("maintenance-tools-panel-confirmation-modal-confirm", "n_clicks"),
            Input("maintenance-tools-panel-confirmation-modal-cancel", "n_clicks"),
        ],
        [
            State("maintenance-tools-panel-confirmation-modal", "is_open"),
            State("maintenance-tools-panel-operation-store", "data"),
        ],
        prevent_initial_call=True,
    )
    def handle_maintenance_modal_consolidated(n_clicks_list, confirm_clicks, cancel_clicks, modal_open, operation_data):
        """
        Callback consolidado que maneja apertura/cierre del modal de confirmación.

        Cumple REGLA #11: ONE INPUT ONE CALLBACK
        - Múltiples Inputs: botones de herramientas, confirm, cancel
        - Múltiples Outputs: modal state, operation store

        Usa ctx.triggered para identificar qué Input disparó el callback.
        """
        if not ctx.triggered:
            return modal_open, operation_data or {}

        trigger_id = ctx.triggered[0]["prop_id"]

        # CASO 1: Click en botón de herramienta → Abrir modal o ejecutar inmediatamente
        if "maintenance-tool-btn" in trigger_id:
            if not any(n_clicks_list):
                return modal_open, operation_data or {}

            # Obtener la operación que se clickeó
            button_id = trigger_id.split(".")[0]
            operation_id = eval(button_id)["operation"]

            config = MaintenanceTools.OPERATIONS_CONFIG.get(operation_id)
            if not config:
                return modal_open, operation_data or {}

            new_operation_data = {
                "operation_id": operation_id,
                "config": config,
                "timestamp": datetime.now().isoformat(),
            }

            # Si requiere confirmación, abrir modal
            if config.get("requires_confirmation", False):
                return True, new_operation_data
            else:
                # Ejecutar inmediatamente si no requiere confirmación
                return modal_open, {**new_operation_data, "execute": True}

        # CASO 2: Click en Confirm → Confirmar ejecución y cerrar modal
        elif trigger_id.endswith("confirm.n_clicks"):
            if not operation_data:
                return False, {}
            # Confirmar ejecución
            return False, {**operation_data, "execute": True}

        # CASO 3: Click en Cancel → Cerrar modal sin ejecutar
        elif trigger_id.endswith("cancel.n_clicks"):
            return False, {}

        return modal_open, operation_data or {}

    @app.callback(
        [
            Output("maintenance-results-panel", "children"),
            Output("toast-trigger-store", "data", allow_duplicate=True),
            Output("maintenance-tools-panel-confirmation-modal-title", "children"),
            Output("maintenance-tools-panel-confirmation-modal-message", "children"),
        ],
        Input("maintenance-tools-panel-operation-store", "data"),
        prevent_initial_call=True,
    )
    def handle_maintenance_operation_unified(operation_data):
        """
        Callback unificado que maneja tanto la actualización del modal de confirmación
        como la ejecución de operaciones de mantenimiento.

        Cumple REGLA #11: ONE INPUT ONE CALLBACK
        - Un solo Input: maintenance-tools-panel-operation-store
        - Múltiples Outputs: results, toast, modal title, modal message

        Flujo:
        1. Si operation_data vacío o sin 'execute' flag:
           - Actualizar solo modal (preparar confirmación)
           - NO ejecutar operación
        2. Si operation_data tiene 'execute': True:
           - Ejecutar operación
           - Actualizar resultados y toast
           - Modal no se actualiza (ya está cerrado)
        """
        # Valores por defecto para outputs
        default_results = []
        default_toast = {}
        default_modal_title = "Confirmar Operación"
        default_modal_message = "¿Está seguro?"

        # Si no hay datos, retornar valores por defecto
        if not operation_data:
            return default_results, default_toast, default_modal_title, default_modal_message

        operation_id = operation_data.get("operation_id")
        config = operation_data.get("config", {})
        execute_flag = operation_data.get("execute", False)

        # CASO 1: Preparar modal de confirmación (execute=False o ausente)
        if not execute_flag:
            # Solo actualizar modal, no ejecutar operación
            title = f"Confirmar: {config.get('title', 'Operación')}"

            message = html.Div(
                [
                    html.P(config.get("description", "¿Desea ejecutar esta operación?")),
                    dbc.Alert(
                        [
                            html.Strong("Información:"),
                            html.Ul(
                                [
                                    html.Li(f"Tiempo estimado: {config.get('estimated_time', 'Variable')}"),
                                    html.Li(f"Nivel de riesgo: {config.get('risk_level', 'No especificado')}"),
                                    html.Li(
                                        "Esta operación es reversible"
                                        if config.get("risk_level") == "none"
                                        else "Verifique antes de continuar"
                                    ),
                                ]
                            ),
                        ],
                        color="info",
                        className="mt-3",
                    ),
                ]
            )

            return default_results, default_toast, title, message

        # CASO 2: Ejecutar operación (execute=True)
        if not operation_id or not config:
            error_toast_data = error_toast("Error: Operación no válida")
            return [], error_toast_data, default_modal_title, default_modal_message

        try:
            client = APIClient()

            # Ejecutar la operación según configuración
            endpoint = config.get("endpoint")
            method = config.get("method", "GET").upper()
            params = config.get("params", {})

            if method == "POST":
                response = client.post(endpoint, json=params)
            else:
                response = client.get(endpoint, params=params)

            # Procesar respuesta
            if response and response.status_code == 200:
                result_data = response.json()

                # Crear visualización del resultado
                result_card = create_operation_result_card(operation_id, result_data, show_details=True)

                # Añadir visualización específica para conectividad
                result_content = [result_card]
                if operation_id == "verify_connectivity":
                    connectivity_results = create_connectivity_test_results(result_data)
                    if connectivity_results:
                        result_content.append(connectivity_results)

                toast_data = success_toast(f"{config['title']} completada exitosamente", "Operación Exitosa")

                return result_content, toast_data, default_modal_title, default_modal_message

            else:
                # Error de API
                error_data = {
                    "success": False,
                    "message": f"Error del servidor: {response.status_code if response else 'Sin respuesta'}",
                    "timestamp": datetime.now().strftime("%H:%M:%S"),
                }

                result_card = create_operation_result_card(operation_id, error_data)
                toast_data = error_toast(f"Error al ejecutar {config['title']}", "Error de Operación")

                return [result_card], toast_data, default_modal_title, default_modal_message

        except Exception as e:
            # Error de conexión o ejecución
            error_data = {
                "success": False,
                "message": f"Error de conexión: {str(e)}",
                "timestamp": datetime.now().strftime("%H:%M:%S"),
            }

            result_card = create_operation_result_card(operation_id, error_data)
            toast_data = error_toast(f"Error al conectar para {config['title']}", "Error de Conexión")

            return [result_card], toast_data, default_modal_title, default_modal_message


# Funciones de utilidad
def get_maintenance_tools_component(component_id: str = "main-maintenance") -> html.Div:
    """
    Función principal para obtener el componente de herramientas de mantenimiento.

    Args:
        component_id: ID único del componente
    """
    return create_maintenance_panel(component_id)


def create_quick_actions_toolbar() -> html.Div:
    """Crea una barra de herramientas con acciones rápidas."""
    return dbc.ButtonGroup(
        [
            dbc.Button(
                html.Div([html.I(className="fas fa-sync-alt me-1"), "Actualizar"]),
                id="quick-refresh-btn",
                color="primary",
                outline=True,
                size="sm",
            ),
            dbc.Button(
                html.Div([html.I(className="fas fa-broom me-1"), "Limpiar Cache"]),
                id="quick-cache-btn",
                color="warning",
                outline=True,
                size="sm",
            ),
            dbc.Button(
                html.Div([html.I(className="fas fa-network-wired me-1"), "Test Red"]),
                id="quick-network-btn",
                color="info",
                outline=True,
                size="sm",
            ),
        ],
        className="mb-3",
    )
