# frontend/components/lock_screen.py
"""
Lock Screen Component for kaiFarma Local (Pivot 2026).

PIN-based terminal lock screen for local Windows installation.
Displays when terminal is locked, hides when unlocked.

Usage in app.py:
    from components.lock_screen import create_lock_screen_modal, create_lock_screen_stores

    app.layout = html.Div([
        *create_lock_screen_stores(),
        create_lock_screen_modal(),
        # ... rest of layout
    ])
"""

import dash_bootstrap_components as dbc
from dash import dcc, html


def create_lock_screen_modal() -> dbc.Modal:
    """
    Create the lock screen modal component.

    Features:
    - Centered modal with backdrop (can't click outside)
    - PIN input with password masking
    - Error message display
    - Pharmacy name display
    - Default PIN hint for first-time users
    """
    return dbc.Modal(
        [
            dbc.ModalHeader(
                html.Div(
                    [
                        html.I(className="fas fa-lock me-2", style={"fontSize": "1.2rem"}),
                        html.Span("kaiFarma", style={"fontWeight": "bold"}),
                    ],
                    className="d-flex align-items-center",
                ),
                close_button=False,
                className="border-0 pb-0",
            ),
            dbc.ModalBody(
                [
                    # Pharmacy name
                    html.Div(
                        id="lock-pharmacy-name",
                        className="text-center mb-4 fs-5 text-muted",
                    ),
                    # Lock icon
                    html.Div(
                        html.I(
                            className="fas fa-user-lock",
                            style={"fontSize": "3rem", "color": "#6c757d"},
                        ),
                        className="text-center mb-4",
                    ),
                    # PIN label
                    html.Label(
                        "Introduce tu PIN para continuar",
                        className="form-label text-center d-block mb-2",
                    ),
                    # PIN input
                    dbc.Input(
                        id="pin-input",
                        type="password",
                        placeholder="* * * *",
                        maxLength=6,
                        className="text-center fs-3 mb-3",
                        style={
                            "letterSpacing": "0.5em",
                            "fontFamily": "monospace",
                            "maxWidth": "200px",
                            "margin": "0 auto",
                        },
                        autoComplete="off",
                    ),
                    # Error message
                    html.Div(
                        id="pin-error",
                        className="text-danger text-center mb-3",
                        style={"minHeight": "1.5rem"},
                    ),
                    # Unlock button
                    dbc.Button(
                        [
                            html.I(className="fas fa-unlock me-2"),
                            "Desbloquear",
                        ],
                        id="btn-unlock",
                        color="primary",
                        className="w-100",
                        size="lg",
                    ),
                ],
                className="p-4",
            ),
            dbc.ModalFooter(
                html.Small(
                    "PIN por defecto: 0000",
                    className="text-muted",
                    id="pin-hint",
                ),
                className="justify-content-center border-0 pt-0",
            ),
        ],
        id="lock-screen-modal",
        is_open=True,  # Start locked
        centered=True,
        backdrop="static",  # Can't click outside to close
        keyboard=False,  # Can't press Esc to close
        size="sm",
        className="lock-screen-modal",
    )


def create_lock_screen_stores() -> list:
    """
    Create the stores and interval needed for lock screen functionality.

    Returns:
        List of dcc components for the skeleton layout
    """
    return [
        # Polling interval for status checks (every 5 seconds)
        dcc.Interval(
            id="security-pulse",
            interval=5000,  # 5 seconds
            n_intervals=0,
        ),
        # Local auth state store
        dcc.Store(
            id="local-auth-state",
            storage_type="memory",  # Memory is fine - resets on page refresh anyway
            data={
                "unlocked": False,
                "pharmacy": "",
                "initialized": False,
            },
        ),
    ]


def create_lock_button() -> dbc.Button:
    """
    Create a lock button for the navbar (optional).

    Can be added to the navbar to allow manual locking.
    """
    return dbc.Button(
        [
            html.I(className="fas fa-lock me-1"),
            "Bloquear",
        ],
        id="btn-manual-lock",
        color="outline-secondary",
        size="sm",
        className="ms-2",
    )
