# frontend/components/kpi_cards.py
"""
Componentes de tarjetas KPI para mostrar métricas clave.
"""

from typing import Any, Dict, List, Optional, Union

import dash_bootstrap_components as dbc
from dash import html
from utils.constants import COLORS, KPI_CONFIG
from utils.helpers import format_currency, format_number, format_percentage, get_trend_color, get_trend_icon


def create_kpi_card(
    title: str,
    value: Union[int, float, str],
    previous_value: Optional[Union[int, float]] = None,
    format_type: str = "number",
    icon: str = "fas fa-chart-line",
    color: str = COLORS["primary"],
    subtitle: Optional[str] = None,
    loading: bool = False,
) -> dbc.Card:
    """
    Crear una tarjeta KPI con valor, tendencia e icono.

    Args:
        title: Título de la métrica
        value: Valor actual de la métrica
        previous_value: Valor anterior para mostrar tendencia
        format_type: Tipo de formato ('currency', 'percentage', 'number')
        icon: Clase CSS del icono FontAwesome
        color: Color del tema
        subtitle: Subtítulo opcional
        loading: Si mostrar estado de carga

    Returns:
        Componente dbc.Card con la información KPI
    """

    # Formatear valor según el tipo
    if loading:
        formatted_value = "..."
    else:
        if format_type == "currency":
            formatted_value = format_currency(value)
        elif format_type == "percentage":
            formatted_value = format_percentage(value)
        elif format_type == "number":
            formatted_value = format_number(value)
        else:
            formatted_value = str(value) if value is not None else "0"

    # Calcular tendencia si hay valor anterior
    trend_elements = []
    if previous_value is not None and not loading and value is not None:
        trend_icon = get_trend_icon(value, previous_value)
        trend_color = get_trend_color(value, previous_value)

        # Calcular porcentaje de cambio
        if previous_value != 0:
            change_percent = ((value - previous_value) / previous_value) * 100
        else:
            change_percent = 0

        trend_elements = [
            html.Div(
                [
                    html.I(className=f"{trend_icon} me-1 {trend_color}"),
                    html.Span(
                        format_percentage(abs(change_percent) / 100, decimals=1), className=f"small {trend_color}"
                    ),
                ],
                className="mt-1",
            )
        ]

    # Crear contenido de la tarjeta
    card_content = [
        # Header con icono
        html.Div(
            [
                html.Div(
                    [
                        html.I(className=f"{icon} fa-2x", style={"color": color}),
                    ],
                    className="col-auto",
                ),
                html.Div(
                    [
                        html.H6(title, className="card-title text-muted mb-1"),
                        html.H3(formatted_value, className="mb-0 fw-bold"),
                        *trend_elements,
                        html.Small(subtitle, className="text-muted") if subtitle else None,
                    ],
                    className="col",
                ),
            ],
            className="row align-items-center",
        )
    ]

    # Añadir spinner si está cargando
    if loading:
        card_content.append(dbc.Spinner(size="sm", color="primary", spinnerClassName="mt-2"))

    return dbc.Card(
        [dbc.CardBody(card_content)],
        className="h-100 shadow-sm border-start border-4",
        style={"border-left-color": f"{color} !important"},
    )


def create_kpi_row(kpis: List[Dict[str, Any]], loading: bool = False) -> dbc.Row:
    """
    Crear una fila con múltiples KPIs.

    Args:
        kpis: Lista de diccionarios con configuración de KPIs
              Cada KPI debe tener: title, value, y opcionalmente:
              previous_value, format_type, icon, color, subtitle
        loading: Si mostrar estado de carga

    Returns:
        Componente dbc.Row con las tarjetas KPI
    """

    cols = []
    for kpi in kpis:
        # Obtener configuración por defecto si existe
        kpi_key = kpi.get("key")
        default_config = KPI_CONFIG.get(kpi_key, {})

        # Mezclar configuración por defecto con la proporcionada
        config = {
            "title": kpi.get("title", default_config.get("title", "Métrica")),
            "value": kpi.get("value", 0),
            "previous_value": kpi.get("previous_value"),
            "format_type": kpi.get("format_type", default_config.get("format", "number")),
            "icon": kpi.get("icon", default_config.get("icon", "fas fa-chart-line")),
            "color": kpi.get("color", default_config.get("color", COLORS["primary"])),
            "subtitle": kpi.get("subtitle"),
            "loading": loading,
        }

        # Crear columna con KPI
        col_width = 12 // len(kpis) if len(kpis) <= 4 else 3
        cols.append(dbc.Col([create_kpi_card(**config)], width=col_width, className="mb-4"))

    return dbc.Row(cols)


def create_summary_kpi_card(
    title: str, items: List[Dict[str, Any]], color: str = COLORS["primary"], loading: bool = False
) -> dbc.Card:
    """
    Crear tarjeta KPI de resumen con múltiples elementos.

    Args:
        title: Título de la tarjeta
        items: Lista de items con 'label' y 'value'
        color: Color del tema
        loading: Si mostrar estado de carga

    Returns:
        Componente dbc.Card con resumen
    """

    if loading:
        content = [dbc.Spinner(color="primary", size="lg", spinnerClassName="my-4")]
    else:
        content = []
        for item in items:
            content.append(
                html.Div(
                    [
                        html.Span(item["label"], className="text-muted small"),
                        html.Div(item["value"], className="fw-bold"),
                    ],
                    className="mb-2",
                )
            )

    return dbc.Card(
        [dbc.CardHeader([html.H6(title, className="mb-0 fw-bold", style={"color": color})]), dbc.CardBody(content)],
        className="h-100 shadow-sm",
    )


def create_mini_kpi(
    label: str, value: Union[int, float, str], format_type: str = "number", trend: Optional[str] = None
) -> html.Div:
    """
    Crear KPI mini para uso en otras tarjetas.

    Args:
        label: Etiqueta del KPI
        value: Valor a mostrar
        format_type: Tipo de formato
        trend: Tendencia ('up', 'down', 'neutral')

    Returns:
        Componente html.Div con KPI mini
    """

    # Formatear valor
    if format_type == "currency":
        formatted_value = format_currency(value)
    elif format_type == "percentage":
        formatted_value = format_percentage(value)
    elif format_type == "number":
        formatted_value = format_number(value)
    else:
        formatted_value = str(value) if value is not None else "0"

    # Icono de tendencia
    trend_icon = ""
    trend_class = ""
    if trend == "up":
        trend_icon = "fas fa-arrow-up"
        trend_class = "text-success"
    elif trend == "down":
        trend_icon = "fas fa-arrow-down"
        trend_class = "text-danger"
    elif trend == "neutral":
        trend_icon = "fas fa-minus"
        trend_class = "text-muted"

    return html.Div(
        [
            html.Div(
                [
                    html.Span(label, className="text-muted small d-block"),
                    html.Div(
                        [
                            html.Span(formatted_value, className="fw-bold me-2"),
                            html.I(className=f"{trend_icon} small {trend_class}") if trend_icon else None,
                        ]
                    ),
                ]
            )
        ],
        className="text-center p-2",
    )
