# frontend/components/inventory/tab_layout.py
"""
Layout reutilizable para pestaña de Inventario.

Issue #471: Combina KPIs, ABC y tabla de rotación en un layout cohesivo.
Se usa en /prescription y /ventalibre con diferentes filtros product_type.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from .abc_chart import create_abc_treemap
from .kpi_cards import create_inventory_kpis
from .rotation_table import create_rotation_table


def create_inventory_tab(
    id_prefix: str,
    product_type: str,
    title: str = "Análisis de Inventario",
    description: str = None,
) -> html.Div:
    """
    Crear layout completo para pestaña de inventario.

    Args:
        id_prefix: Prefijo para IDs únicos (ej: "prescription", "ventalibre")
        product_type: Tipo de producto para filtrar ("medicamento", "venta_libre")
        title: Título de la sección (opcional)
        description: Descripción de la sección (opcional)

    Returns:
        html.Div con layout completo de inventario
    """
    if description is None:
        if product_type == "medicamento":
            description = "Análisis de rotación y rentabilidad del inventario de medicamentos"
        else:
            description = "Análisis de rotación y rentabilidad del inventario de parafarmacia"

    return html.Div([
        # NOTA: Store para product_type está en app.py (REGLA #0.5: dcc.Store ÚNICO global)
        # Los callbacks usan State(f"{id_prefix}-inv-product-type", "data")

        # ===================================================================
        # HEADER SECTION
        # ===================================================================
        html.Div([
            html.H5([
                html.I(className="fas fa-boxes me-2"),
                title,
            ], className="mb-2"),
            html.P(description, className="text-muted small mb-4"),
        ]),

        # ===================================================================
        # ALERTA SI NO HAY DATOS DE INVENTARIO
        # ===================================================================
        dbc.Alert(
            html.Span([
                html.I(className="fas fa-info-circle me-2"),
                "Para ver el análisis de inventario, ",
                html.A(
                    "sube un fichero de inventario",
                    href="/upload",
                    className="alert-link",
                ),
                " desde tu ERP (Farmanager, Farmatic, etc.)",
            ]),
            id=f"{id_prefix}-inv-no-data-alert", color="info", is_open=False, className="mb-4"
        ),

        # ===================================================================
        # KPIs ROW
        # ===================================================================
        create_inventory_kpis(id_prefix),

        # ===================================================================
        # ABC + TABLA ROW
        # ===================================================================
        dbc.Row([
            # Clasificación ABC (izquierda)
            dbc.Col([
                create_abc_treemap(id_prefix),
            ], width=12, lg=6, className="mb-4"),

            # Tabla de rotación (derecha)
            dbc.Col([
                create_rotation_table(id_prefix),
            ], width=12, lg=6, className="mb-4"),
        ]),

        # ===================================================================
        # MÉTRICAS ADICIONALES (cobertura días, stock crítico)
        # ===================================================================
        dbc.Row([
            # Cobertura de Stock
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(className="fas fa-calendar-alt me-2"),
                        "Cobertura de Stock",
                    ]),
                    dbc.CardBody([
                        dcc.Loading([
                            html.Div(id=f"{id_prefix}-inv-coverage-chart"),
                        ], type="circle"),
                    ]),
                ], className="shadow-sm"),
            ], width=12, lg=6, className="mb-4"),

            # Alertas de Stock
            dbc.Col([
                dbc.Card([
                    dbc.CardHeader([
                        html.I(className="fas fa-exclamation-circle me-2"),
                        "Alertas de Stock",
                        dbc.Badge(
                            id=f"{id_prefix}-inv-alerts-count",
                            children="0",
                            color="danger",
                            className="ms-2",
                        ),
                    ]),
                    dbc.CardBody([
                        # Filtro por tipo de alerta
                        dcc.Dropdown(
                            id=f"{id_prefix}-inv-alerts-filter",
                            options=[
                                {"label": "Todas las alertas", "value": "all"},
                                {"label": "Stock muerto (sin ventas 180+ días)", "value": "dead_stock"},
                                {"label": "Stock bajo (cobertura < 7 días)", "value": "low_stock"},
                                {"label": "Sobrestock (cobertura > 90 días)", "value": "overstock"},
                            ],
                            value="all",
                            clearable=False,
                            className="mb-3",
                        ),
                        dcc.Loading([
                            html.Div(
                                id=f"{id_prefix}-inv-alerts-list",
                                style={"maxHeight": "280px", "overflowY": "auto"},
                            ),
                        ], type="circle"),
                    ]),
                ], className="shadow-sm"),
            ], width=12, lg=6, className="mb-4"),
        ]),

        # ===================================================================
        # CAJA EXPLICATIVA DE MÉTRICAS (Issue #470)
        # ===================================================================
        dbc.Card([
            dbc.CardHeader([
                html.I(className="fas fa-info-circle me-2"),
                "Guía de Métricas de Inventario",
            ], className="bg-light"),
            dbc.CardBody([
                dbc.Row([
                    # Columna 1: Rotación y Días Inventario
                    dbc.Col([
                        html.H6([
                            html.I(className="fas fa-sync-alt me-2 text-primary"),
                            "Índice de Rotación",
                        ], className="mb-2"),
                        html.P([
                            html.Strong("Fórmula: "),
                            "Coste de Ventas (12 meses) / Stock Promedio a Coste",
                        ], className="small mb-1"),
                        html.P([
                            html.Strong("Interpretación: "),
                        ], className="small mb-1"),
                        html.Ul([
                            html.Li("≥6x: Excelente rotación", className="text-success small"),
                            html.Li("4-6x: Buena rotación", className="text-success small"),
                            html.Li("2-4x: Rotación moderada", className="text-warning small"),
                            html.Li("<2x: Rotación lenta - Revisar", className="text-danger small"),
                        ], className="mb-3 ps-3"),

                        html.H6([
                            html.I(className="fas fa-calendar-day me-2 text-info"),
                            "Días en Inventario",
                        ], className="mb-2"),
                        html.P([
                            html.Strong("Fórmula: "),
                            "365 / Índice de Rotación",
                        ], className="small mb-1"),
                        html.P([
                            "Indica cuántos días tarda en venderse el stock promedio.",
                        ], className="small text-muted"),
                    ], width=12, md=4, className="mb-3 mb-md-0"),

                    # Columna 2: GMROI
                    dbc.Col([
                        html.H6([
                            html.I(className="fas fa-chart-line me-2 text-success"),
                            "GMROI (Rentabilidad del Inventario)",
                        ], className="mb-2"),
                        html.P([
                            html.Strong("Fórmula: "),
                            "Margen Bruto / Stock a Coste",
                        ], className="small mb-1"),
                        html.P([
                            html.Strong("Interpretación: "),
                            "Euros de margen generados por cada euro invertido en stock.",
                        ], className="small mb-2"),
                        html.Ul([
                            html.Li("≥3.0: GMROI excelente", className="text-success small"),
                            html.Li("2.0-3.0: GMROI muy bueno", className="text-success small"),
                            html.Li("1.5-2.0: GMROI saludable", className="text-warning small"),
                            html.Li("<1.5: GMROI bajo - Optimizar", className="text-danger small"),
                        ], className="mb-3 ps-3"),
                        html.P([
                            html.I(className="fas fa-lightbulb me-2 text-warning"),
                            html.Strong("¿Por qué varía el GMROI entre productos?"),
                        ], className="small mb-1"),
                        html.P([
                            "Aunque el margen (%) es similar, cada producto tiene diferente ",
                            html.Strong("relación ventas/stock"),
                            ". Un producto con €10.000 ventas y €1.000 stock tiene GMROI=3.0, "
                            "mientras que otro con las mismas ventas pero €5.000 stock tiene GMROI=0.6.",
                        ], className="small text-muted"),
                    ], width=12, md=4, className="mb-3 mb-md-0"),

                    # Columna 3: Clasificación ABC
                    dbc.Col([
                        html.H6([
                            html.I(className="fas fa-layer-group me-2 text-warning"),
                            "Clasificación ABC (Pareto)",
                        ], className="mb-2"),
                        html.P([
                            "Clasifica productos según su contribución a las ventas:",
                        ], className="small mb-2"),
                        html.Div([
                            html.Div([
                                dbc.Badge("A", color="success", className="me-2"),
                                html.Span("Top 80% ventas (~20% productos)", className="small"),
                            ], className="mb-1"),
                            html.Div([
                                dbc.Badge("B", color="warning", className="me-2"),
                                html.Span("Siguiente 15% ventas (~30% productos)", className="small"),
                            ], className="mb-1"),
                            html.Div([
                                dbc.Badge("C", color="secondary", className="me-2"),
                                html.Span("Último 5% ventas (~50% productos)", className="small"),
                            ], className="mb-1"),
                        ], className="mb-3"),
                        html.P([
                            html.I(className="fas fa-bullseye me-2 text-primary"),
                            html.Strong("Acción recomendada:"),
                        ], className="small mb-1"),
                        html.P([
                            "Productos A: Mantener stock óptimo. "
                            "Productos C con bajo GMROI: Reducir stock o eliminar.",
                        ], className="small text-muted"),
                    ], width=12, md=4),
                ]),
            ]),
        ], className="shadow-sm border-info mb-4"),

    ], className="pt-3")


def create_inventory_tab_placeholder(id_prefix: str) -> html.Div:
    """
    Crear placeholder para pestaña de inventario (cuando no hay datos).

    Args:
        id_prefix: Prefijo para IDs únicos

    Returns:
        html.Div con mensaje de estado vacío
    """
    return html.Div([
        html.Div([
            html.I(className="fas fa-warehouse fa-4x text-muted mb-4"),
            html.H4("Sin datos de inventario", className="text-muted"),
            html.P(
                html.Span([
                    "Para analizar la rotación y rentabilidad del inventario, ",
                    "primero debes subir un fichero de inventario desde tu ERP.",
                ]),
                className="text-muted mb-4"
            ),
            dbc.Button(
                html.Span([html.I(className="fas fa-upload me-2"), "Subir fichero de inventario"]),
                href="/upload", color="primary", size="lg"
            ),
        ], className="text-center py-5"),
    ])
