# frontend/components/inventory/rotation_table.py
"""
Tabla de productos por rotación/rentabilidad.

Issue #471: Tabla con Top/Bottom productos.
Consume medidas: rotation_index, gmroi, days_inventory.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from styles.design_tokens import COLORS


def create_rotation_table(id_prefix: str) -> dbc.Card:
    """
    Crear card con tabla de productos ordenados por rotación/rentabilidad.

    Los datos se actualizan vía callback usando el ID:
    - {id_prefix}-inv-rotation-table

    Args:
        id_prefix: Prefijo para IDs únicos

    Returns:
        dbc.Card con tabla de productos
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-sort-amount-down me-2"),
            "Productos por Rotación",
            # Selector Top/Bottom
            dbc.ButtonGroup([
                dbc.Button(
                    "Top 10",
                    id=f"{id_prefix}-inv-table-top",
                    color="success",
                    size="sm",
                    outline=True,
                    active=True,
                ),
                dbc.Button(
                    "Bottom 10",
                    id=f"{id_prefix}-inv-table-bottom",
                    color="danger",
                    size="sm",
                    outline=True,
                ),
            ], className="float-end", size="sm"),
        ]),
        dbc.CardBody([
            # Ordenar por
            dbc.Row([
                dbc.Col([
                    html.Label("Ordenar por:", className="small text-muted"),
                    dcc.Dropdown(
                        id=f"{id_prefix}-inv-table-sort",
                        options=[
                            {"label": "Rotación (veces/año)", "value": "rotation"},
                            {"label": "Días en inventario", "value": "days"},
                            {"label": "GMROI", "value": "gmroi"},
                            {"label": "Ventas (€)", "value": "sales"},
                            {"label": "Stock muerto (días)", "value": "dead_days"},
                        ],
                        value="rotation",
                        clearable=False,
                        className="mb-3",
                    ),
                ], width=12, md=6),
                dbc.Col([
                    html.Label("Clase ABC:", className="small text-muted"),
                    dcc.Dropdown(
                        id=f"{id_prefix}-inv-table-abc-filter",
                        options=[
                            {"label": "Todas", "value": "all"},
                            {"label": "Clase A", "value": "A"},
                            {"label": "Clase B", "value": "B"},
                            {"label": "Clase C", "value": "C"},
                        ],
                        value="all",
                        clearable=False,
                        className="mb-3",
                    ),
                ], width=12, md=6),
            ]),

            # Tabla de productos
            dcc.Loading([
                html.Div(
                    id=f"{id_prefix}-inv-rotation-table",
                    style={"maxHeight": "400px", "overflowY": "auto"},
                ),
            ], type="circle"),
        ]),
    ], className="h-100 shadow-sm")


def create_product_row(
    product_name: str,
    codigo_nacional: str,
    rotation: float = None,
    days_inventory: int = None,
    gmroi: float = None,
    sales: float = 0,
    abc_class: str = "A",
) -> html.Tr:
    """
    Crear fila de tabla para un producto.

    Args:
        product_name: Nombre del producto
        codigo_nacional: Código nacional
        rotation: Índice de rotación (None = no disponible)
        days_inventory: Días en inventario (None = no disponible)
        gmroi: GMROI del producto (None = no disponible)
        sales: Ventas totales
        abc_class: Clasificación ABC (A, B, C)

    Returns:
        html.Tr con datos del producto
    """
    # Color según clase ABC
    abc_colors = {
        "A": "success",
        "B": "warning",
        "C": "danger",
    }

    # Color según rotación (si disponible)
    rotation_color = "text-muted"
    if rotation is not None:
        if rotation >= 6:
            rotation_color = "text-success"
        elif rotation >= 2:
            rotation_color = "text-warning"
        else:
            rotation_color = "text-danger"

    # Formatear valores (N/D si no disponible)
    rotation_text = f"{rotation:.1f}x" if rotation is not None else "N/D"
    days_text = f"{days_inventory} días" if days_inventory is not None else "N/D"
    gmroi_text = f"{gmroi:.2f}" if gmroi is not None else "N/D"

    return html.Tr([
        # Producto
        html.Td([
            html.Div(product_name[:40] + "..." if len(product_name) > 40 else product_name,
                     className="fw-medium"),
            html.Small(codigo_nacional, className="text-muted"),
        ]),
        # Rotación
        html.Td([
            html.Span(rotation_text, className=rotation_color),
        ], className="text-center"),
        # Días
        html.Td([
            html.Span(days_text, className="text-muted" if days_inventory is None else ""),
        ], className="text-center"),
        # GMROI
        html.Td([
            html.Span(gmroi_text, className="text-muted" if gmroi is None else ""),
        ], className="text-center"),
        # Ventas
        html.Td([
            html.Span(f"€{sales:,.0f}"),
        ], className="text-end"),
        # Clase ABC
        html.Td([
            dbc.Badge(abc_class, color=abc_colors.get(abc_class, "secondary")),
        ], className="text-center"),
    ])


def create_rotation_table_header() -> html.Thead:
    """Crear header de la tabla de rotación."""
    return html.Thead([
        html.Tr([
            html.Th("Producto", style={"width": "35%"}),
            html.Th("Rotación", className="text-center", style={"width": "12%"}),
            html.Th("Días Inv.", className="text-center", style={"width": "12%"}),
            html.Th("GMROI", className="text-center", style={"width": "12%"}),
            html.Th("Ventas", className="text-end", style={"width": "15%"}),
            html.Th("ABC", className="text-center", style={"width": "10%"}),
        ], className="table-light"),
    ])


def create_empty_rotation_table(message: str = "No hay datos de inventario") -> html.Div:
    """
    Crear estado vacío para la tabla.

    Args:
        message: Mensaje a mostrar

    Returns:
        html.Div con mensaje de estado vacío
    """
    return html.Div([
        html.I(className="fas fa-box-open fa-3x text-muted mb-3"),
        html.P(message, className="text-muted"),
        html.Small(
            "Sube un fichero de inventario para ver el análisis",
            className="text-muted",
        ),
    ], className="text-center py-5")
