# frontend/components/inventory/kpi_cards.py
"""
Componentes KPI para análisis de inventario.

Issue #471: KPIs de inventario reutilizables.
Issue #487: Añadido KPI "Cobertura" con color dinámico según alert_level.

Consumen medidas del backend: stock_value, rotation_index, gmroi, stock_coverage_days, dead_stock_value.
"""

import dash_bootstrap_components as dbc
from dash import html

from styles.design_tokens import COLORS


def create_inventory_kpis(id_prefix: str) -> dbc.Row:
    """
    Crear fila de 5 KPIs de inventario.

    Los valores se actualizan vía callback usando los IDs:
    - {id_prefix}-inv-kpi-stock-value
    - {id_prefix}-inv-kpi-rotation
    - {id_prefix}-inv-kpi-gmroi
    - {id_prefix}-inv-kpi-coverage (Issue #487)
    - {id_prefix}-inv-kpi-coverage-card (Issue #487: para color dinámico)
    - {id_prefix}-inv-kpi-dead-stock

    Args:
        id_prefix: Prefijo para IDs únicos (ej: "prescription", "ventalibre")

    Returns:
        dbc.Row con 5 KPI cards (ancho igual en pantallas lg+)
    """
    return dbc.Row([
        # KPI 1: Valor de Stock
        dbc.Col([
            _create_inventory_kpi_card(
                kpi_id=f"{id_prefix}-inv-kpi-stock-value",
                title="Valor Stock",
                icon="fas fa-warehouse",
                color=COLORS.get("info", "#17a2b8"),
            )
        ], width=6, md=4, lg=True, className="mb-3"),

        # KPI 2: Índice de Rotación
        dbc.Col([
            _create_inventory_kpi_card(
                kpi_id=f"{id_prefix}-inv-kpi-rotation",
                title="Rotación",
                icon="fas fa-sync-alt",
                color=COLORS.get("success", "#28a745"),
                subtitle="veces/año",
            )
        ], width=6, md=4, lg=True, className="mb-3"),

        # KPI 3: GMROI
        dbc.Col([
            _create_inventory_kpi_card(
                kpi_id=f"{id_prefix}-inv-kpi-gmroi",
                title="GMROI",
                icon="fas fa-chart-line",
                color=COLORS.get("primary", "#007bff"),
                subtitle="margen/inversión",
            )
        ], width=6, md=4, lg=True, className="mb-3"),

        # KPI 4: Días de Cobertura (Issue #487) - El más importante para compras
        dbc.Col([
            _create_inventory_kpi_card(
                kpi_id=f"{id_prefix}-inv-kpi-coverage",
                title="Cobertura",
                icon="fas fa-calendar-check",
                color=COLORS.get("secondary", "#6c757d"),  # Color base, dinámico vía callback
                subtitle="días de stock",
                card_id=f"{id_prefix}-inv-kpi-coverage-card",  # Issue #487: Para borde dinámico
            )
        ], width=6, md=6, lg=True, className="mb-3"),

        # KPI 5: Stock Muerto
        dbc.Col([
            _create_inventory_kpi_card(
                kpi_id=f"{id_prefix}-inv-kpi-dead-stock",
                title="Stock Muerto",
                icon="fas fa-exclamation-triangle",
                color=COLORS.get("warning", "#ffc107"),
                subtitle=">180 días",
            )
        ], width=6, md=6, lg=True, className="mb-3"),
    ])


def _create_inventory_kpi_card(
    kpi_id: str,
    title: str,
    icon: str,
    color: str,
    subtitle: str = None,
    card_id: str = None,
) -> dbc.Card:
    """
    Crear tarjeta KPI individual para inventario.

    Args:
        kpi_id: ID único del componente (para callbacks)
        title: Título del KPI
        icon: Clase CSS FontAwesome del icono
        color: Color del icono
        subtitle: Subtítulo opcional
        card_id: ID opcional del Card (para estilos dinámicos vía callback)

    Returns:
        dbc.Card con estructura KPI
    """
    card_props = {
        "className": "h-100 shadow-sm",
    }
    if card_id:
        card_props["id"] = card_id

    return dbc.Card([
        dbc.CardBody([
            # Icono
            html.Div([
                html.I(className=f"{icon} fa-2x", style={"color": color}),
            ], className="text-center mb-2"),

            # Valor (actualizado por callback)
            html.H3(
                id=kpi_id,
                children="--",
                className="text-center mb-1",
            ),

            # Título
            html.P(title, className="text-muted text-center small mb-0"),

            # Subtítulo opcional
            html.Small(
                subtitle,
                className="text-muted text-center d-block",
            ) if subtitle else None,

            # Interpretación (actualizada por callback)
            html.Small(
                id=f"{kpi_id}-interpretation",
                children="",
                className="text-muted text-center d-block fst-italic mt-1",
                style={"fontSize": "0.75rem"},
            ),
        ], className="py-3"),
    ], **card_props)


def create_inventory_kpi_loading_state(id_prefix: str) -> dict:
    """
    Crear diccionario de outputs para estado de carga.

    Args:
        id_prefix: Prefijo de IDs

    Returns:
        Diccionario con valores de carga para cada KPI
    """
    return {
        f"{id_prefix}-inv-kpi-stock-value": "...",
        f"{id_prefix}-inv-kpi-rotation": "...",
        f"{id_prefix}-inv-kpi-gmroi": "...",
        f"{id_prefix}-inv-kpi-coverage": "...",  # Issue #487
        f"{id_prefix}-inv-kpi-dead-stock": "...",
    }
