# frontend/components/inventory/abc_chart.py
"""
Visualización de Clasificación ABC para inventario.

Issue #471: Treemap/Barras para clasificación Pareto.
Consume medida: abc_classification del backend.
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from styles.design_tokens import COLORS


def create_abc_treemap(id_prefix: str) -> dbc.Card:
    """
    Crear card con visualización ABC Classification.

    El gráfico se actualiza vía callback usando el ID:
    - {id_prefix}-inv-abc-chart

    Args:
        id_prefix: Prefijo para IDs únicos

    Returns:
        dbc.Card con treemap/bar chart ABC
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-layer-group me-2"),
            "Clasificación ABC",
            # Legend badges
            html.Span([
                dbc.Badge("A", color="success", className="ms-2", title="80% ventas, ~20% productos"),
                dbc.Badge("B", color="warning", className="ms-1", title="15% ventas, ~30% productos"),
                dbc.Badge("C", color="danger", className="ms-1", title="5% ventas, ~50% productos"),
            ], className="float-end"),
        ]),
        dbc.CardBody([
            # Selector de tipo de visualización
            dbc.ButtonGroup([
                dbc.Button(
                    html.Span([html.I(className="fas fa-chart-bar me-1"), "Barras"]),
                    id=f"{id_prefix}-inv-abc-view-bar",
                    color="primary",
                    size="sm",
                    outline=True,
                    active=True,
                ),
                dbc.Button(
                    html.Span([html.I(className="fas fa-th-large me-1"), "Treemap"]),
                    id=f"{id_prefix}-inv-abc-view-treemap",
                    color="primary",
                    size="sm",
                    outline=True,
                ),
            ], className="mb-3"),

            # Container del gráfico
            dcc.Loading([
                html.Div(id=f"{id_prefix}-inv-abc-chart"),
            ], type="circle"),

            # Resumen estadístico
            html.Hr(className="my-3"),
            dbc.Row([
                dbc.Col([
                    html.Div([
                        html.Span("Clase A", className="fw-bold text-success"),
                        html.Div(id=f"{id_prefix}-inv-abc-summary-a", children="--"),
                    ], className="text-center"),
                ], width=4),
                dbc.Col([
                    html.Div([
                        html.Span("Clase B", className="fw-bold text-warning"),
                        html.Div(id=f"{id_prefix}-inv-abc-summary-b", children="--"),
                    ], className="text-center"),
                ], width=4),
                dbc.Col([
                    html.Div([
                        html.Span("Clase C", className="fw-bold text-danger"),
                        html.Div(id=f"{id_prefix}-inv-abc-summary-c", children="--"),
                    ], className="text-center"),
                ], width=4),
            ]),
        ]),
    ], className="h-100 shadow-sm")


def create_abc_summary_card(id_prefix: str) -> dbc.Card:
    """
    Crear card compacta con resumen ABC (alternativa al treemap).

    Args:
        id_prefix: Prefijo para IDs únicos

    Returns:
        dbc.Card con resumen ABC compacto
    """
    return dbc.Card([
        dbc.CardHeader([
            html.I(className="fas fa-layer-group me-2"),
            "Resumen ABC",
        ]),
        dbc.CardBody([
            # Clase A
            dbc.Row([
                dbc.Col([
                    html.Div([
                        dbc.Badge("A", color="success", className="me-2"),
                        html.Span("Alta rotación"),
                    ]),
                ], width=6),
                dbc.Col([
                    html.Span(id=f"{id_prefix}-inv-abc-a-products", children="-- prods"),
                    html.Span(" | ", className="text-muted"),
                    html.Span(id=f"{id_prefix}-inv-abc-a-sales", children="-- €"),
                ], width=6, className="text-end"),
            ], className="mb-2"),

            # Clase B
            dbc.Row([
                dbc.Col([
                    html.Div([
                        dbc.Badge("B", color="warning", className="me-2"),
                        html.Span("Media rotación"),
                    ]),
                ], width=6),
                dbc.Col([
                    html.Span(id=f"{id_prefix}-inv-abc-b-products", children="-- prods"),
                    html.Span(" | ", className="text-muted"),
                    html.Span(id=f"{id_prefix}-inv-abc-b-sales", children="-- €"),
                ], width=6, className="text-end"),
            ], className="mb-2"),

            # Clase C
            dbc.Row([
                dbc.Col([
                    html.Div([
                        dbc.Badge("C", color="danger", className="me-2"),
                        html.Span("Baja rotación"),
                    ]),
                ], width=6),
                dbc.Col([
                    html.Span(id=f"{id_prefix}-inv-abc-c-products", children="-- prods"),
                    html.Span(" | ", className="text-muted"),
                    html.Span(id=f"{id_prefix}-inv-abc-c-sales", children="-- €"),
                ], width=6, className="text-end"),
            ]),
        ]),
    ], className="shadow-sm")
