# frontend/components/insights/insights_panel.py
"""
InsightsPanel Component - Insight Engine v2.0 (Issue #506)

Panel completo de insights que incluye:
- Header: Total oportunidad + badges por severidad
- Filtros: Por categoría/severidad
- Lista de InsightCards
- Estados: Loading, empty, error

Uso:
    from components.insights import InsightsPanel

    InsightsPanel(id_prefix="ventalibre")
"""

import dash_bootstrap_components as dbc
from dash import dcc, html

from components.skeleton_loader import create_card_skeleton
from styles.design_tokens import BORDER_RADIUS, COLORS, SHADOWS, SPACING

from .insight_card import InsightCard


def InsightsPanel(id_prefix: str = "insights") -> html.Div:
    """
    Crea el panel completo de insights.

    Args:
        id_prefix: Prefijo para IDs únicos (para evitar colisiones)

    Returns:
        html.Div con el panel de insights
    """
    return html.Div(
        id=f"{id_prefix}-insights-panel",
        children=[
            # NOTA: Stores e Interval definidos en app.py skeleton (REGLA #0.5)
            # - {id_prefix}-insights-data-store
            # - {id_prefix}-insights-filter-store
            # - {id_prefix}-insights-refresh-interval

            # Header con estadísticas
            _create_header(id_prefix),

            # Contenedor principal
            html.Div(
                id=f"{id_prefix}-insights-container",
                children=[_create_loading_state()],
                style={"marginTop": SPACING["m"]},
            ),
        ],
        style={
            "padding": SPACING["m"],
            "backgroundColor": COLORS["bg_secondary"],
            "borderRadius": BORDER_RADIUS["md"],
            "minHeight": "400px",
        },
    )


def _create_header(id_prefix: str) -> html.Div:
    """Crea el header del panel con estadísticas y filtros."""
    return html.Div(
        children=[
            # Título y total
            html.Div(
                children=[
                    html.Div(
                        children=[
                            html.I(className="bi bi-lightbulb-fill me-2", style={"color": COLORS["warning"]}),
                            html.H5("Insights Automáticos", className="mb-0 d-inline"),
                        ],
                        className="d-flex align-items-center",
                    ),
                    html.Div(
                        id=f"{id_prefix}-insights-total-opportunity",
                        children=[
                            html.Span("Oportunidad total: ", className="text-muted"),
                            html.Strong("--", id=f"{id_prefix}-insights-total-value"),
                        ],
                        className="mt-1",
                        style={"fontSize": "0.9rem"},
                    ),
                ],
                className="mb-3",
            ),

            # Badges de severidad y conteo
            html.Div(
                id=f"{id_prefix}-insights-severity-badges",
                children=[
                    _create_severity_badge("high", 0, id_prefix),
                    _create_severity_badge("medium", 0, id_prefix),
                    _create_severity_badge("low", 0, id_prefix),
                    html.Span(
                        id=f"{id_prefix}-insights-suppressed-count",
                        children="",
                        className="text-muted ms-3",
                        style={"fontSize": "0.8rem"},
                    ),
                ],
                className="d-flex align-items-center flex-wrap gap-2 mb-3",
            ),

            # Filtros
            html.Div(
                children=[
                    dbc.ButtonGroup(
                        children=[
                            dbc.Button(
                                "Todos",
                                id={"type": f"{id_prefix}-insight-filter", "value": "all"},
                                color="primary",
                                outline=True,
                                size="sm",
                                active=True,
                                n_clicks=0,
                            ),
                            dbc.Button(
                                "Stock",
                                id={"type": f"{id_prefix}-insight-filter", "value": "stock"},
                                color="primary",
                                outline=True,
                                size="sm",
                                n_clicks=0,
                            ),
                            dbc.Button(
                                "Margen",
                                id={"type": f"{id_prefix}-insight-filter", "value": "margin"},
                                color="primary",
                                outline=True,
                                size="sm",
                                n_clicks=0,
                            ),
                            dbc.Button(
                                "Concentración",
                                id={"type": f"{id_prefix}-insight-filter", "value": "hhi"},
                                color="primary",
                                outline=True,
                                size="sm",
                                n_clicks=0,
                            ),
                            dbc.Button(
                                "Tendencia",
                                id={"type": f"{id_prefix}-insight-filter", "value": "trend"},
                                color="primary",
                                outline=True,
                                size="sm",
                                n_clicks=0,
                            ),
                            dbc.Button(
                                "Surtido",
                                id={"type": f"{id_prefix}-insight-filter", "value": "surtido"},
                                color="primary",
                                outline=True,
                                size="sm",
                                n_clicks=0,
                            ),
                        ],
                        size="sm",
                    ),
                    # Botón refresh
                    dbc.Button(
                        children=[
                            html.I(className="bi bi-arrow-clockwise"),
                        ],
                        id=f"{id_prefix}-insights-refresh-btn",
                        color="secondary",
                        outline=True,
                        size="sm",
                        className="ms-3",
                        title="Actualizar insights",
                        n_clicks=0,
                    ),
                ],
                className="d-flex align-items-center",
            ),
        ],
        style={
            "backgroundColor": COLORS["bg_primary"],
            "padding": SPACING["m"],
            "borderRadius": BORDER_RADIUS["md"],
            "boxShadow": SHADOWS["sm"],
        },
    )


def _create_severity_badge(severity: str, count: int, id_prefix: str) -> dbc.Badge:
    """Crea un badge de conteo por severidad."""
    config = {
        "high": {"label": "Alta", "color": "danger", "icon": "bi bi-exclamation-triangle-fill"},
        "medium": {"label": "Media", "color": "warning", "icon": "bi bi-exclamation-circle-fill"},
        "low": {"label": "Baja", "color": "info", "icon": "bi bi-info-circle-fill"},
    }
    cfg = config.get(severity, config["medium"])

    return dbc.Badge(
        children=[
            html.I(className=f"{cfg['icon']} me-1"),
            html.Span(f"{cfg['label']}: ", style={"fontWeight": "normal"}),
            html.Strong(str(count), id=f"{id_prefix}-insights-count-{severity}"),
        ],
        id=f"{id_prefix}-insights-badge-{severity}",
        color=cfg["color"],
        className="me-2",
        style={"fontSize": "0.75rem"},
    )


def _create_loading_state() -> html.Div:
    """Crea el estado de carga con skeletons."""
    return html.Div(
        children=[
            create_card_skeleton(),
            create_card_skeleton(),
            create_card_skeleton(),
        ],
        className="p-3",
    )


def create_empty_state() -> html.Div:
    """Crea el estado vacío cuando no hay insights."""
    return html.Div(
        children=[
            html.Div(
                children=[
                    html.I(
                        className="bi bi-check-circle-fill",
                        style={
                            "fontSize": "3rem",
                            "color": COLORS["success"],
                        },
                    ),
                    html.H5(
                        "Todo bajo control",
                        className="mt-3 mb-2",
                        style={"color": COLORS["text_primary"]},
                    ),
                    html.P(
                        "No se detectaron oportunidades de mejora en este momento. "
                        "El sistema analiza automáticamente stock, márgenes, tendencias y surtido.",
                        className="text-muted mb-0",
                        style={"maxWidth": "400px"},
                    ),
                ],
                className="text-center py-5",
            ),
        ],
        style={
            "backgroundColor": COLORS["bg_primary"],
            "borderRadius": BORDER_RADIUS["md"],
            "boxShadow": SHADOWS["sm"],
        },
    )


def create_error_state(error_message: str = "Error al cargar insights", id_prefix: str = "insights") -> html.Div:
    """Crea el estado de error."""
    return html.Div(
        children=[
            html.Div(
                children=[
                    html.I(
                        className="bi bi-exclamation-triangle-fill",
                        style={
                            "fontSize": "2.5rem",
                            "color": COLORS["danger"],
                        },
                    ),
                    html.H6(
                        "Error al cargar insights",
                        className="mt-3 mb-2",
                        style={"color": COLORS["text_primary"]},
                    ),
                    html.P(
                        error_message,
                        className="text-muted mb-3",
                    ),
                    dbc.Button(
                        children=html.Span([
                            html.I(className="bi bi-arrow-clockwise me-1"),
                            "Reintentar",
                        ]),
                        id=f"{id_prefix}-insights-retry-btn",
                        color="primary",
                        size="sm",
                    ),
                ],
                className="text-center py-4",
            ),
        ],
        style={
            "backgroundColor": COLORS["bg_primary"],
            "borderRadius": BORDER_RADIUS["md"],
            "boxShadow": SHADOWS["sm"],
        },
    )


def render_insights_list(insights: list, id_prefix: str = "insights") -> html.Div:
    """
    Renderiza la lista de insights como InsightCards.

    Args:
        insights: Lista de insights (InsightItem schema)
        id_prefix: Prefijo para IDs

    Returns:
        html.Div con las cards o estado vacío
    """
    if not insights:
        return create_empty_state()

    return html.Div(
        children=[
            InsightCard(insight=insight, index=i)
            for i, insight in enumerate(insights)
        ],
    )
