# frontend/components/insights/insight_card.py
"""
InsightCard Component - Insight Engine v2.0 (Issue #506)

Card individual para mostrar un insight con:
- Header: Severity badge + título + impacto económico
- Body: Descripción + productos afectados (opcional)
- Footer: Botones de acción (Deeplink, Posponer, Descartar)

Uso:
    from components.insights import InsightCard

    InsightCard(
        insight={
            "rule_code": "STOCK_001",
            "category": "stock",
            "severity": "high",
            "title": "Stock Crítico: 12 productos",
            "description": "Tienes productos con alta demanda...",
            "economic_impact": "Est. +1.200€/mes",
            "economic_value": 1200.0,
            "action_label": "Ver productos",
            "deeplink": "/ventalibre/inventario?alert=low_stock",
            "affected_items": [...],
            "insight_hash": "abc123",
        }
    )
"""

import dash_bootstrap_components as dbc
from dash import html

from styles.design_tokens import BORDER_RADIUS, COLORS, SHADOWS, SPACING


# Mapeo de severidad a colores y labels
SEVERITY_CONFIG = {
    "high": {
        "color": COLORS["danger"],
        "bg": COLORS["danger_light"],
        "label": "Alta",
        "icon": "bi bi-exclamation-triangle-fill",
    },
    "medium": {
        "color": COLORS["warning_dark"],
        "bg": COLORS["warning_light"],
        "label": "Media",
        "icon": "bi bi-exclamation-circle-fill",
    },
    "low": {
        "color": COLORS["info"],
        "bg": COLORS["info_light"],
        "label": "Baja",
        "icon": "bi bi-info-circle-fill",
    },
}

# Mapeo de categoría a iconos
CATEGORY_ICONS = {
    "stock": "bi bi-box-seam",
    "margin": "bi bi-percent",
    "hhi": "bi bi-pie-chart",
    "trend": "bi bi-graph-down",
    "surtido": "bi bi-grid-3x3-gap",
}


def InsightCard(insight: dict, show_details: bool = True, index: int = 0) -> dbc.Card:
    """
    Crea una tarjeta de insight individual.

    Args:
        insight: Diccionario con datos del insight (InsightItem schema)
        show_details: Si mostrar productos afectados
        index: Índice para IDs únicos

    Returns:
        dbc.Card con el insight renderizado
    """
    severity = insight.get("severity", "medium")
    category = insight.get("category", "stock")
    severity_cfg = SEVERITY_CONFIG.get(severity, SEVERITY_CONFIG["medium"])
    category_icon = CATEGORY_ICONS.get(category, "bi bi-lightbulb")

    # Header: Badge + Título + Impacto
    header = dbc.CardHeader(
        children=[
            html.Div(
                children=[
                    # Severity badge
                    dbc.Badge(
                        children=[
                            html.I(className=f"{severity_cfg['icon']} me-1"),
                            severity_cfg["label"],
                        ],
                        color="",
                        className="me-2",
                        style={
                            "backgroundColor": severity_cfg["bg"],
                            "color": severity_cfg["color"],
                            "padding": "4px 8px",
                            "fontSize": "0.75rem",
                            "fontWeight": "600",
                        },
                    ),
                    # Category icon
                    html.I(
                        className=f"{category_icon} me-2",
                        style={"color": COLORS["secondary"]},
                    ),
                    # Rule code
                    html.Small(
                        insight.get("rule_code", ""),
                        className="text-muted",
                    ),
                ],
                className="d-flex align-items-center mb-2",
            ),
            html.Div(
                children=[
                    # Título
                    html.H6(
                        insight.get("title", "Insight"),
                        className="mb-0 flex-grow-1",
                        style={
                            "color": COLORS["text_primary"],
                            "fontWeight": "600",
                        },
                    ),
                    # Impacto económico
                    html.Span(
                        insight.get("economic_impact", ""),
                        className="ms-2",
                        style={
                            "color": COLORS["success"],
                            "fontWeight": "700",
                            "fontSize": "0.95rem",
                            "whiteSpace": "nowrap",
                        },
                    ),
                ],
                className="d-flex align-items-center justify-content-between",
            ),
        ],
        style={
            "backgroundColor": COLORS["bg_secondary"],
            "borderBottom": f"1px solid {COLORS['border_light']}",
            "padding": SPACING["m"],
        },
    )

    # Body: Descripción + Afectados
    body_children = [
        html.P(
            insight.get("description", ""),
            className="mb-3",
            style={
                "color": COLORS["text_secondary"],
                "fontSize": "0.875rem",
                "lineHeight": "1.5",
            },
        ),
    ]

    # Añadir lista de afectados si hay y show_details
    affected = insight.get("affected_items", [])
    if show_details and affected and len(affected) > 0:
        # Mostrar primeros 3 items
        items_to_show = affected[:3]
        body_children.append(
            html.Div(
                children=[
                    html.Small(
                        f"{len(affected)} productos afectados:",
                        className="text-muted d-block mb-2",
                        style={"fontWeight": "500"},
                    ),
                    html.Ul(
                        children=[
                            html.Li(
                                _format_affected_item(item),
                                style={"fontSize": "0.8rem"},
                            )
                            for item in items_to_show
                        ],
                        className="mb-0 ps-3",
                        style={"listStyle": "none"},
                    ),
                    html.Small(
                        f"+ {len(affected) - 3} más..." if len(affected) > 3 else "",
                        className="text-muted",
                    ) if len(affected) > 3 else None,
                ],
                style={
                    "backgroundColor": COLORS["bg_tertiary"],
                    "borderRadius": BORDER_RADIUS["sm"],
                    "padding": SPACING["s"],
                },
            )
        )

    body = dbc.CardBody(
        children=body_children,
        style={"padding": SPACING["m"]},
    )

    # Footer: Botones de acción
    insight_hash = insight.get("insight_hash", "")
    rule_code = insight.get("rule_code", "")

    footer = dbc.CardFooter(
        children=[
            html.Div(
                children=[
                    # Botón principal (deeplink)
                    dbc.Button(
                        children=html.Span([
                            html.I(className="bi bi-arrow-right-circle me-1"),
                            insight.get("action_label", "Ver detalles"),
                        ]),
                        id={"type": "insight-action-btn", "index": index},
                        color="primary",
                        size="sm",
                        className="me-2",
                        n_clicks=0,
                    ),
                    # Botón posponer
                    dbc.Button(
                        children=html.Span([
                            html.I(className="bi bi-clock me-1"),
                            "Posponer",
                        ]),
                        id={"type": "insight-snooze-btn", "index": index, "hash": insight_hash, "rule": rule_code},
                        color="secondary",
                        outline=True,
                        size="sm",
                        className="me-2",
                        n_clicks=0,
                    ),
                    # Botón descartar
                    dbc.Button(
                        children=html.Span([
                            html.I(className="bi bi-x-circle me-1"),
                            "Descartar",
                        ]),
                        id={"type": "insight-dismiss-btn", "index": index, "hash": insight_hash, "rule": rule_code},
                        color="secondary",
                        outline=True,
                        size="sm",
                        n_clicks=0,
                    ),
                ],
                className="d-flex align-items-center",
            ),
            # Store para deeplink
            html.Div(
                insight.get("deeplink", ""),
                id={"type": "insight-deeplink", "index": index},
                style={"display": "none"},
            ),
        ],
        style={
            "backgroundColor": COLORS["bg_primary"],
            "borderTop": f"1px solid {COLORS['border_light']}",
            "padding": SPACING["s"],
        },
    )

    # Card completa
    return dbc.Card(
        children=[header, body, footer],
        className="mb-3",
        style={
            "borderRadius": BORDER_RADIUS["md"],
            "boxShadow": SHADOWS["sm"],
            "border": f"1px solid {COLORS['border_light']}",
            "borderLeft": f"4px solid {severity_cfg['color']}",
        },
    )


def _format_affected_item(item: dict) -> str:
    """Formatea un item afectado para mostrar en la lista."""
    name = item.get("product_name", item.get("name", "Producto"))
    if len(name) > 40:
        name = name[:37] + "..."

    # Buscar valor relevante
    if "sales_lost_estimate" in item:
        return f"{name} (-{item['sales_lost_estimate']:.0f}€)"
    elif "excess_value" in item:
        return f"{name} ({item['excess_value']:.0f}€ exceso)"
    elif "stock_value" in item:
        return f"{name} ({item['stock_value']:.0f}€)"
    elif "coverage_days" in item:
        return f"{name} ({item['coverage_days']} días)"
    elif "erosion_pp" in item:
        return f"{item.get('category', name)} (-{item['erosion_pp']:.1f}pp)"
    elif "potential_gain" in item:
        return f"{name} (+{item['potential_gain']:.0f}€ potencial)"
    else:
        return name
