# frontend/components/generics/tables.py
"""
Table components for generics analysis.

This module contains table visualization components extracted from
frontend/callbacks/generics.py for better organization.

Functions:
- create_products_detail_table: Detailed product table for homogeneous groups
- create_product_row_with_bar: Product row with visual sales bar
- create_product_row: Simple product row
- create_presentation_item: Presentation item for matrix
- create_traditional_table: Fallback traditional table
- create_partner_refs_table: Sortable partner references table
"""

import logging
from typing import List

import dash_bootstrap_components as dbc
from components.common import create_alert
from dash import dash_table, html
from styles.design_tokens import COLORS as DS_COLORS
from styles.design_tokens import SPACING, TYPOGRAPHY
from utils.helpers import format_compact_number, format_currency

logger = logging.getLogger(__name__)


def create_products_detail_table(products: List[dict], group_summary: dict, discount_percentage: float = 0) -> html.Div:
    """
    Crear tabla detallada de productos de un conjunto homogéneo.
    Separa productos Partners (verde) de No-Partners/Oportunidad (amarillo).

    Args:
        products: Lista de productos del conjunto
        group_summary: Resumen del grupo
        discount_percentage: Porcentaje de descuento a aplicar (0-100)
    """
    if not products:
        return create_alert("No hay productos en este conjunto", "info")

    # Separar productos por tipo
    partner_products = [p for p in products if p.get("is_partner", False)]
    opportunity_products = [p for p in products if not p.get("is_partner", False)]

    # Totales del resumen
    total_products = group_summary.get("total_products", len(products))
    partner_penetration = group_summary.get("partner_penetration", 0)

    # Calcular máximo revenue para barras proporcionales
    max_revenue = max([p.get("revenue_sold", 0) for p in products]) if products else 1

    return html.Div(
        [
            # Resumen del grupo con comparación visual
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-info-circle me-2 text-primary"),
                                    html.Strong(f"{total_products} productos", className="me-3"),
                                    html.Span(
                                        f"Penetración Partners: {partner_penetration:.1f}%", className="text-muted"
                                    ),
                                ]
                            )
                        ],
                        width=12,
                        className="mb-3",
                    )
                ]
            ),
            # OPORTUNIDADES PRIMERO - Productos apilados con barras visuales
            html.Div(
                [
                    html.H6(
                        [
                            html.I(className="fas fa-bullseye me-2 text-warning"),
                            f"🎯 Productos Oportunidad ({len(opportunity_products)})",
                        ],
                        className="text-warning mb-3",
                        style={"fontSize": "1.1rem", "fontWeight": "600"},
                    ),
                    (
                        html.Div(
                            [
                                create_product_row_with_bar(
                                    product,
                                    is_partner=False,
                                    max_revenue=max_revenue,
                                    discount_percentage=discount_percentage,
                                )
                                for product in opportunity_products[:10]
                            ]
                        )
                        if opportunity_products
                        else html.Small("No hay productos de oportunidad", className="text-muted")
                    ),
                ],
                className="mb-4",
            ),
            # PARTNERS DESPUÉS - Productos apilados con barras visuales
            html.Div(
                [
                    html.H6(
                        [
                            html.I(className="fas fa-handshake me-2 text-success"),
                            f"✓ Productos Partners ({len(partner_products)})",
                        ],
                        className="text-success mb-3",
                        style={"fontSize": "1.1rem", "fontWeight": "600"},
                    ),
                    (
                        html.Div(
                            [
                                create_product_row_with_bar(
                                    product,
                                    is_partner=True,
                                    max_revenue=max_revenue,
                                    discount_percentage=discount_percentage,
                                )
                                for product in partner_products[:10]
                            ]
                        )
                        if partner_products
                        else html.Small("No hay productos de partners", className="text-muted")
                    ),
                ]
            ),
        ]
    )


def create_product_row_with_bar(
    product: dict, is_partner: bool, max_revenue: float, discount_percentage: float = 0
) -> html.Div:
    """
    Crear fila de producto con barra visual comparativa de ventas.

    Args:
        product: Datos del producto
        is_partner: Si es producto partner o no
        max_revenue: Máximo revenue del grupo para calcular barras proporcionales
        discount_percentage: Porcentaje de descuento a aplicar (0-100)
    """
    national_code = product.get("national_code", "")
    product_name = product.get("product_name", "")
    laboratory = product.get("laboratory", "")
    units_sold = product.get("units_sold", 0)
    revenue_sold = product.get("revenue_sold", 0)
    individual_savings_base = product.get("individual_savings_base", 0)

    # ✅ CORRECCIÓN: Aplicar descuento al ahorro
    # El ahorro base es: cantidad * PVL (sin descuento)
    # El ahorro real es: ahorro_base * (descuento / 100)
    individual_savings = individual_savings_base * (discount_percentage / 100)

    # Calcular porcentaje de barra visual
    bar_percentage = (revenue_sold / max_revenue * 100) if max_revenue > 0 else 0

    # Colores diferenciados
    if is_partner:
        border_color = "#28a745"
        bar_color = "#28a745"
        bg_color = "rgba(40, 167, 69, 0.1)"
    else:
        border_color = "#ffc107"
        bar_color = "#ffc107"
        bg_color = "rgba(255, 193, 7, 0.1)"

    return html.Div(
        [
            dbc.Row(
                [
                    # Columna izquierda: Info del producto (INVERTIDO: nombre arriba, laboratorio abajo)
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Strong(
                                        national_code,
                                        className="text-primary" if is_partner else "text-warning",
                                        style={"fontSize": "0.95rem"},
                                    ),
                                    html.Span(" - ", className="text-muted mx-1"),
                                    html.Span(
                                        product_name[:60] + ("..." if len(product_name) > 60 else ""),
                                        style={"fontSize": "0.9rem"},
                                    ),
                                ],
                                className="mb-1",
                            ),
                            html.Small(
                                [html.I(className="fas fa-flask me-1"), laboratory],
                                className="text-success" if is_partner else "text-muted",
                            ),
                        ],
                        width=12,
                        lg=6,
                    ),
                    # Columna derecha: Barra visual + métricas
                    dbc.Col(
                        [
                            # Barra visual de ventas
                            html.Div(
                                [
                                    html.Div(
                                        [
                                            html.Span(
                                                format_currency(revenue_sold),
                                                style={"fontSize": "0.9rem", "fontWeight": "600", "color": "#fff"},
                                            )
                                        ],
                                        style={
                                            "width": f"{bar_percentage}%",
                                            "backgroundColor": bar_color,
                                            "padding": "4px 8px",
                                            "borderRadius": "4px",
                                            "transition": "width 0.3s ease",
                                            "minWidth": "80px",
                                            "textAlign": "center",
                                        },
                                    )
                                ],
                                className="mb-1",
                            ),
                            # Métricas adicionales
                            html.Div(
                                [
                                    html.Small(
                                        [
                                            html.I(className="fas fa-box me-1"),
                                            f"{format_compact_number(units_sold)} uds",
                                        ],
                                        className="me-3",
                                    ),
                                    (
                                        html.Small(
                                            [
                                                html.I(className="fas fa-piggy-bank me-1 text-warning"),
                                                f"Ahorro: {format_currency(individual_savings)}",
                                            ]
                                        )
                                        if not is_partner and individual_savings > 0
                                        else None
                                    ),
                                ],
                                className="text-muted",
                            ),
                        ],
                        width=12,
                        lg=6,
                    ),
                ],
                className="align-items-center",
            )
        ],
        className="border-bottom py-2 mb-1",
        style={
            "borderLeft": f"4px solid {border_color}",
            "backgroundColor": bg_color,
            "borderRadius": "4px",
            "transition": "all 0.2s ease",
        },
    )


def create_product_row(product: dict, is_partner: bool) -> html.Div:
    """
    Crear fila de producto individual con border-left para mejor contraste.
    """
    national_code = product.get("national_code", "")
    product_name = product.get("product_name", "")
    laboratory = product.get("laboratory", "")
    units_sold = product.get("units_sold", 0)
    revenue_sold = product.get("revenue_sold", 0)
    individual_savings = product.get("individual_savings_base", 0)

    # Colores y estilos diferenciados
    if is_partner:
        bg_color = "bg-success-subtle"
        border_color = "#28a745"
        border_width = "4px"
    else:
        bg_color = "bg-warning-subtle"
        border_color = "#ffc107"
        border_width = "4px"

    return html.Div(
        [
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Strong(
                                        national_code, className="text-primary" if is_partner else "text-warning"
                                    ),
                                    html.Br(),
                                    html.Small(
                                        product_name[:60] + ("..." if len(product_name) > 60 else ""),
                                        className="text-muted",
                                    ),
                                ]
                            )
                        ],
                        width=12,
                        className="mb-1",
                    ),
                    dbc.Col(
                        [
                            html.Small(
                                [html.I(className="fas fa-flask me-1"), html.Strong(laboratory)],
                                className="text-success" if is_partner else "text-dark",
                            )
                        ],
                        width=12,
                        className="mb-1",
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Small(
                                        [
                                            html.Span(f"{format_compact_number(units_sold)} uds", className="me-2"),
                                            html.Strong(format_currency(revenue_sold), className="text-success"),
                                        ]
                                    ),
                                    (
                                        html.Small(
                                            [
                                                html.I(className="fas fa-piggy-bank me-1 text-warning"),
                                                f"Ahorro: {format_currency(individual_savings)}",
                                            ],
                                            className="ms-2",
                                        )
                                        if not is_partner and individual_savings > 0
                                        else None
                                    ),
                                ]
                            )
                        ],
                        width=12,
                    ),
                ]
            )
        ],
        className=f"border-bottom py-2 {bg_color}",
        style={"borderLeft": f"{border_width} solid {border_color}"},
    )


def create_presentation_item(presentation: dict, is_partner: bool) -> html.Div:
    """
    Crear item de presentación para matriz.
    """
    lab_name = presentation.get("laboratory_name", "")
    units = presentation.get("units", 0)
    sales_amount = presentation.get("sales_amount", 0)
    potential_savings = presentation.get("potential_savings", 0)

    if is_partner:
        return html.Div(
            [
                html.Div(
                    [
                        html.Small(lab_name, className="fw-bold text-success"),
                        html.Small(
                            f" • {format_compact_number(units)} uds • {format_currency(sales_amount)}",
                            className="text-muted ms-1",
                        ),
                    ]
                )
            ],
            className="mb-1",
        )
    else:
        return html.Div(
            [
                html.Div(
                    [
                        html.Small(lab_name, className="fw-bold text-dark"),
                        html.Small(
                            f" • {format_compact_number(units)} uds • {format_currency(sales_amount)}",
                            className="text-muted ms-1",
                        ),
                        html.Small(
                            f" → Ahorro: {format_currency(potential_savings)}", className="text-success fw-bold ms-1"
                        ),
                    ]
                )
            ],
            className="mb-1",
        )


def create_traditional_table(data: List[dict]) -> html.Div:
    """
    Crear tabla tradicional como fallback.
    """
    if not data:
        return create_alert("No hay datos disponibles", "info")

    table_data = []
    for group in data:
        table_data.append(
            {
                "conjunto": group.get("homogeneous_name", ""),
                "ventas": format_currency(group.get("total_revenue", 0)),
                "unidades": format_compact_number(group.get("total_units", 0)),
                "ahorro": format_currency(group.get("savings_base", 0)),
                "porcentaje": f"{group.get('partner_penetration', 0):.1f}%",
            }
        )

    return dash_table.DataTable(
        data=table_data,
        columns=[
            {"name": "Conjunto", "id": "conjunto"},
            {"name": "Ventas", "id": "ventas"},
            {"name": "Unidades", "id": "unidades"},
            {"name": "Ahorro", "id": "ahorro"},
            {"name": "% Partners", "id": "porcentaje"},
        ],
        style_cell={"textAlign": "left", "padding": "10px"},
        style_header={"backgroundColor": "#f8f9fa", "fontWeight": "bold"},
        page_size=10,
    )


def create_partner_refs_table(partner_refs: List[dict]) -> html.Div:
    """
    Crear tabla ordenable de referencias partners usando DataTable.
    """
    if not partner_refs:
        return create_alert("No hay referencias disponibles", "info")

    # Preparar datos para DataTable
    table_data = []
    for ref in partner_refs:
        table_data.append(
            {
                "Código Nacional": str(ref.get("national_code", "")),  # Forzar como string
                "Descripción": ref.get("product_name", ""),
                "Laboratorio": ref.get("laboratory", ""),
                "Unidades": ref.get("units_sold", 0),
                "Ventas (€)": ref.get("total_sales", 0),
            }
        )

    # Crear DataTable con anchos optimizados
    return dash_table.DataTable(
        data=table_data,
        columns=[
            {"name": "Código Nacional", "id": "Código Nacional", "type": "text"},
            {"name": "Descripción", "id": "Descripción", "type": "text"},
            {"name": "Laboratorio", "id": "Laboratorio", "type": "text"},
            {
                "name": "Unidades",
                "id": "Unidades",
                "type": "numeric",
                "format": {"specifier": ",.0f", "locale": {"decimal": ",", "group": "."}},
            },
            {
                "name": "Ventas (€)",
                "id": "Ventas (€)",
                "type": "numeric",
                "format": {"specifier": "$,.2f", "locale": {"decimal": ",", "group": ".", "symbol": ["", " €"]}},
            },
        ],
        sort_action="native",
        filter_action="native",
        page_size=20,
        page_action="native",
        tooltip_data=[{"Descripción": {"value": str(row["Descripción"]), "type": "markdown"}} for row in table_data],
        tooltip_duration=None,
        style_table={"overflowX": "auto", "maxHeight": "500px", "overflowY": "auto", "width": "100%"},
        style_cell={
            "textAlign": "left",
            "padding": SPACING["s"],
            "fontFamily": "system-ui, -apple-system, sans-serif",
            "fontSize": TYPOGRAPHY["body_small"]["size"],
        },
        style_cell_conditional=[
            {"if": {"column_id": "Código Nacional"}, "width": "100px", "maxWidth": "100px", "minWidth": "100px"},
            {
                "if": {"column_id": "Unidades"},
                "width": "80px",
                "maxWidth": "80px",
                "minWidth": "80px",
                "textAlign": "right",
            },
            {
                "if": {"column_id": "Ventas (€)"},
                "width": "120px",
                "maxWidth": "120px",
                "minWidth": "120px",
                "textAlign": "right",
            },
            {"if": {"column_id": "Laboratorio"}, "width": "150px", "maxWidth": "150px", "minWidth": "150px"},
        ],
        style_header={
            "backgroundColor": DS_COLORS["gray_100"],
            "fontWeight": "600",
            "color": DS_COLORS["gray_700"],
            "borderBottom": f"2px solid {DS_COLORS['gray_200']}",
            "fontSize": TYPOGRAPHY["body_small"]["size"],
        },
        style_data_conditional=[
            {"if": {"row_index": "odd"}, "backgroundColor": "#f8f9fa"},
            {"if": {"column_id": "Código Nacional"}, "fontWeight": "600", "color": "#0d6efd"},
        ],
    )
