# frontend/components/generics/matrix.py
"""
PowerBI-style matrix components for generics analysis.

This module contains matrix visualization components extracted from
frontend/callbacks/generics.py for better organization.

Functions:
- create_powerbi_matrix: Create main PowerBI-style matrix with expandable rows
- create_matrix_group_row: Create individual group row in matrix
- create_matrix_detail_content: Create detail content for expanded row
"""

import logging
from typing import List, Optional

import dash_bootstrap_components as dbc
from components.common import create_alert
from dash import dcc, html
from utils.helpers import format_compact_number, format_currency

logger = logging.getLogger(__name__)


def create_powerbi_matrix(
    data: List[dict], expand_all_clicks: Optional[int] = None, discount_percentage: float = 0, sort_state: dict = None
) -> html.Div:
    """
    Crear matriz PowerBI-style con jerarquía expandible.

    Args:
        data: Lista de grupos homogéneos
        expand_all_clicks: Número de clicks en expandir todo
        discount_percentage: Porcentaje de descuento a aplicar (0-100)
        sort_state: Estado de ordenación {column, direction}
    """
    if not data:
        return create_alert("No hay datos para mostrar en la matriz", "info")

    matrix_rows = []

    # Estado de ordenación por defecto
    if not sort_state:
        sort_state = {"column": "savings_base", "direction": "desc"}

    current_column = sort_state.get("column", "savings_base")
    current_direction = sort_state.get("direction", "desc")

    # Función helper para crear header clickeable
    def create_sortable_header(text: str, column: str, lg: int, align: str = "left"):
        is_active = current_column == column
        icon = (
            "fas fa-sort-down"
            if (is_active and current_direction == "desc")
            else "fas fa-sort-up" if (is_active and current_direction == "asc") else "fas fa-sort"
        )
        icon_color = "#0d6efd" if is_active else "#6c757d"

        return dbc.Col(
            [
                dbc.Button(
                    [
                        html.Strong(text, style={"color": "#495057" if not is_active else "#0d6efd"}),
                        html.I(className=f"{icon} ms-1", style={"fontSize": "0.8rem", "color": icon_color}),
                    ],
                    id={"type": "matrix-header", "column": column},
                    color="light" if not is_active else "primary",
                    outline=True,
                    size="sm",
                    className=f"w-100 text-{align} border-0",
                    style={"backgroundColor": "transparent" if not is_active else "rgba(13, 110, 253, 0.1)"},
                )
            ],
            lg=lg,
        )

    # Encabezado de matriz con ordenación clickeable
    header_row = html.Div(
        [
            dbc.Row(
                [
                    create_sortable_header("Conjunto Homogéneo", "name", 4, "left"),
                    create_sortable_header("Ventas Totales", "total_revenue", 2, "center"),
                    create_sortable_header("Unidades", "total_units", 2, "center"),
                    create_sortable_header("% Partners", "partner_penetration", 2, "center"),
                    create_sortable_header("Ahorro Potencial", "savings_base", 2, "center"),
                ],
                className="py-2 border-bottom",
                style={"backgroundColor": "#f8f9fa"},
            )
        ]
    )
    matrix_rows.append(header_row)

    # Filas de datos con jerarquía y descuento aplicado
    for i, group in enumerate(data[:20]):  # Limitar a 20 para performance
        matrix_rows.append(create_matrix_group_row(group, i, expand_all_clicks, discount_percentage))

    return html.Div(
        [
            html.Div(matrix_rows, className="matrix-container"),
            (
                html.Div(
                    [
                        html.Small(
                            f"Mostrando los primeros 20 de {len(data)} conjuntos",
                            className="text-muted text-center d-block mt-3",
                        )
                    ]
                )
                if len(data) > 20
                else None
            ),
        ]
    )


def create_matrix_group_row(
    group: dict, index: int, expand_all_clicks: Optional[int], discount_percentage: float = 0
) -> html.Div:
    """
    Crear fila de grupo en matriz PowerBI.

    Args:
        group: Datos del grupo homogéneo
        index: Índice de la fila
        expand_all_clicks: Número de clicks en expandir todo
        discount_percentage: Porcentaje de descuento a aplicar (0-100)
    """
    group_id = f"matrix-group-{index}"
    homogeneous_name = group.get("homogeneous_name", "")
    # Issue #346: Convertir código a string para evitar problemas con IDs pattern-matching
    homogeneous_code = str(group.get("homogeneous_code", ""))
    total_sales = group.get("total_revenue", 0)
    total_units = group.get("total_units", 0)
    partner_savings_base = group.get("savings_base", 0)
    partner_percentage = group.get("partner_penetration", 0)

    # ✅ CORRECCIÓN: Aplicar descuento al ahorro
    partner_savings = partner_savings_base * (discount_percentage / 100)

    # Colores basados en performance (usar ahorro con descuento)
    savings_color = "#28a745" if partner_savings > 100 else "#ffc107" if partner_savings > 50 else "#dc3545"
    percentage_color = "#28a745" if partner_percentage > 50 else "#ffc107" if partner_percentage > 20 else "#dc3545"

    return html.Div(
        [
            # Fila principal del grupo (% Partners antes que Ahorro)
            dbc.Row(
                [
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    # Botón de selección (Issue #346)
                                    dbc.Button(
                                        html.I(className="fas fa-chart-line"),
                                        id={"type": "select-homogeneous-btn", "code": homogeneous_code},
                                        size="sm",
                                        color="light",
                                        outline=True,
                                        className="me-2",
                                        style={"fontSize": "0.7rem"},  # Reducido de 0.8rem
                                        title="Seleccionar para ver evolución temporal",
                                    ),
                                    # Botón de expansión
                                    dbc.Button(
                                        html.I(
                                            className="fas fa-plus-square", id={"type": "expand-icon", "index": index}
                                        ),
                                        id={"type": "expand-btn", "index": index},
                                        size="sm",
                                        color="light",
                                        className="me-2 border-0",
                                        style={"fontSize": "0.9rem"},
                                    ),
                                    html.Span(
                                        [
                                            html.Strong(homogeneous_name[:50], style={"color": "#212529"}),
                                            html.Small(
                                                f" ({group.get('total_units', 0)} uds • {group.get('partner_penetration', 0):.0f}% partners)",
                                                className="text-muted ms-1",
                                            ),
                                        ]
                                    ),
                                ],
                                className="d-flex align-items-center",
                            )
                        ],
                        lg=4,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Span(
                                        format_currency(total_sales), style={"fontWeight": "600", "color": "#495057"}
                                    )
                                ],
                                className="text-center",
                            )
                        ],
                        lg=2,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Span(
                                        format_compact_number(total_units),
                                        style={"fontWeight": "600", "color": "#495057"},
                                    )
                                ],
                                className="text-center",
                            )
                        ],
                        lg=2,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    dbc.Progress(
                                        value=min(partner_percentage, 100),
                                        color=(
                                            "success"
                                            if partner_percentage > 50
                                            else "warning" if partner_percentage > 20 else "danger"
                                        ),
                                        style={"height": "18px"},
                                        children=f"{partner_percentage:.1f}%",
                                        className="text-center",
                                    )
                                ],
                                className="d-flex align-items-center",
                            )
                        ],
                        lg=2,
                    ),
                    dbc.Col(
                        [
                            html.Div(
                                [
                                    html.Span(
                                        format_currency(partner_savings),
                                        style={
                                            "fontWeight": "600",
                                            "color": "#fff" if partner_savings > 100 else savings_color,
                                        },
                                    )
                                ],
                                className="text-center px-2 py-1 rounded",
                                style={
                                    "backgroundColor": savings_color if partner_savings > 100 else "transparent",
                                    "display": "inline-block",
                                    "minWidth": "100px",
                                },
                            )
                        ],
                        lg=2,
                        className="text-center",
                    ),
                ],
                className="py-2 border-bottom hover-row",
                style={"cursor": "pointer"},
            ),
            # Fila de detalles (inicialmente oculta) - full width
            html.Div(
                create_matrix_detail_content(group),
                id={"type": "detail-row", "index": index},
                style={"display": "none", "marginLeft": "0", "marginRight": "0"},
                className="detail-row bg-light border-top border-3 border-primary py-3 mt-1",
            ),
        ],
        className="matrix-row",
    )


def create_matrix_detail_content(group: dict) -> html.Div:
    """
    Crear contenido detallado de matriz PowerBI con carga dinámica.
    El contenido real se carga via callback cuando se expande la fila.
    """
    homogeneous_code = group.get("homogeneous_code", "")

    # Crear contenedor con ID único para carga dinámica - full width con padding
    # El callback "load_homogeneous_detail" poblará este contenedor
    return html.Div(
        [
            html.Div(
                id={"type": "detail-content", "code": homogeneous_code},
                children=[
                    dcc.Loading(
                        [
                            html.Div(
                                [
                                    html.I(className="fas fa-info-circle me-2 text-primary"),
                                    "Haz click en el botón (+) para cargar el detalle de productos...",
                                ],
                                className="text-muted text-center py-3",
                            )
                        ],
                        type="default",
                    )
                ],
                className="px-3",
            )
        ]
    )
