# frontend/components/generics/charts.py
"""
Chart components for generics analysis.

This module contains chart visualization components extracted from
frontend/callbacks/generics.py for better organization.

Functions:
- create_stacked_bar_chart: Create 100% stacked bar chart for Partners vs No-Partners
- create_empty_chart: Create empty placeholder chart with message
"""

import logging
from typing import List

import pandas as pd
import plotly.graph_objects as go

logger = logging.getLogger(__name__)

# Colores consistentes para Partners vs No-Partners
PARTNER_COLOR = "#0d6efd"  # Azul para partners
NO_PARTNER_COLOR = "#ffc107"  # Amarillo para no-partners


def create_stacked_bar_chart(data: List[dict], value_field: str, title: str, level: str) -> go.Figure:
    """
    Crear gráfico de barras 100% apiladas para Partners vs No-Partners con porcentajes dentro de las barras.
    """
    if not data:
        return create_empty_chart("No hay datos disponibles")

    df = pd.DataFrame(data)

    # Validar que existan las columnas requeridas
    required_cols = ["period_label", "partner_units", "opportunity_units", "total_revenue", "savings_base"]
    missing_cols = [col for col in required_cols if col not in df.columns]
    if missing_cols:
        logger.error(f"Columnas faltantes en datos temporales: {missing_cols}")
        return create_empty_chart(f"Datos incompletos: faltan {', '.join(missing_cols)}")

    # Calcular totales y porcentajes
    if value_field == "units":
        partners_values = df["partner_units"]
        no_partners_values = df["opportunity_units"]
    else:
        partners_values = df["total_revenue"] - df["savings_base"]
        no_partners_values = df["savings_base"]

    totals = partners_values + no_partners_values
    partners_pct = (partners_values / totals * 100).fillna(0)
    no_partners_pct = (no_partners_values / totals * 100).fillna(0)

    fig = go.Figure()

    # Barras para Partners con porcentajes dentro
    fig.add_trace(
        go.Bar(
            name="Partners",
            x=df["period_label"],
            y=partners_values,
            marker_color=PARTNER_COLOR,
            text=[f"{pct:.1f}%" if pct >= 5 else "" for pct in partners_pct],  # Solo mostrar % si >= 5%
            textposition="inside",
            textfont=dict(color="white", size=12, family="Arial Black"),
            hovertemplate=f"<b>Partners</b><br>Período: %{{x}}<br>{title}: %{{y:,.0f}} (%{{text}})<extra></extra>",
        )
    )

    # Barras para No-Partners con porcentajes dentro
    fig.add_trace(
        go.Bar(
            name="No-Partners",
            x=df["period_label"],
            y=no_partners_values,
            marker_color=NO_PARTNER_COLOR,
            text=[f"{pct:.1f}%" if pct >= 5 else "" for pct in no_partners_pct],  # Solo mostrar % si >= 5%
            textposition="inside",
            textfont=dict(color="black", size=12, family="Arial Black"),
            hovertemplate=f"<b>No-Partners</b><br>Período: %{{x}}<br>{title}: %{{y:,.0f}} (%{{text}})<extra></extra>",
        )
    )

    fig.update_layout(
        barmode="stack",
        title=None,
        xaxis_title=f"Período ({level})",
        yaxis_title=title,
        showlegend=True,
        height=280,
        margin=dict(l=10, r=10, t=10, b=40),
        legend=dict(orientation="h", yanchor="bottom", y=1.02, xanchor="right", x=1),
        # Mejorar contraste de texto
        font=dict(color="#333333"),
        bargap=0.2,  # Espacio entre barras para mejor legibilidad
    )

    return fig


def create_empty_chart(message: str) -> go.Figure:
    """
    Crear gráfico vacío con mensaje.
    """
    fig = go.Figure()
    fig.add_annotation(
        text=message, xref="paper", yref="paper", x=0.5, y=0.5, showarrow=False, font=dict(size=14, color="gray")
    )
    fig.update_layout(
        showlegend=False,
        height=280,
        margin=dict(l=10, r=10, t=10, b=10),
        xaxis=dict(visible=False),
        yaxis=dict(visible=False),
    )
    return fig
