# frontend/components/generic_analysis.py
"""
Componente para análisis de genéricos usando sistema de medidas Power BI-style.
Dashboard moderno y eficiente para análisis farmacéutico avanzado.
"""

import logging
from datetime import datetime, timedelta
from typing import Any, Dict, List

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from dash import Input, Output, State, callback, dash_table, dcc, html
from utils.api_client import APIClient
from utils.helpers import format_currency, format_number
from utils.pharmacy_context import get_current_pharmacy_id

logger = logging.getLogger(__name__)


def create_static_test_kpis():
    """KPIs estáticos para probar que la estructura funciona"""
    return dbc.Row(
        [
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("€294.321,07", className="text-primary mb-1"),
                                    html.P("Total Ventas", className="text-muted mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("18,62%", className="text-info mb-1"),
                                    html.P("% Genéricos", className="text-muted mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("11,44%", className="text-warning mb-1"),
                                    html.P("% Partners", className="text-muted mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4("€15.234,56", className="text-success mb-1"),
                                    html.P("Ahorro Potencial", className="text-muted mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
        ]
    )


def load_kpis_directly(pharmacy_id: str):
    """Cargar KPIs directamente sin callbacks para evitar problemas de timing"""
    try:
        from utils.api_client import APIClient

        api_client = APIClient()

        logger.info(f"Loading KPIs directly for pharmacy: {pharmacy_id}")

        # Llamada directa al API
        dashboard_data = api_client.get(f"/api/generic-analysis/dashboard/{pharmacy_id}")

        logger.info(f"API returned type: {type(dashboard_data)}, data: {dashboard_data}")

        if dashboard_data and isinstance(dashboard_data, dict) and len(dashboard_data) > 0:
            logger.info(f"Direct KPI load successful for pharmacy {pharmacy_id}")
            return create_updated_kpis_row(dashboard_data)
        else:
            logger.warning(f"Direct KPI load failed: {dashboard_data}")
            return create_kpi_placeholder_row()

    except Exception as e:
        logger.error(f"Error in direct KPI load: {str(e)}")
        return create_kpi_placeholder_row()


def get_date_range_for_slider(pharmacy_id: str):
    """Obtener rango de fechas y convertir a números para el RangeSlider"""
    try:
        api_client = APIClient()
        response = api_client.get_sales_date_range(pharmacy_id)

        if response.success and response.data:
            min_date_str = response.data.get("min_date")
            max_date_str = response.data.get("max_date")

            # Convertir strings ISO a objetos datetime
            min_date = datetime.fromisoformat(min_date_str).date()
            max_date = datetime.fromisoformat(max_date_str).date()

            # Convertir fechas a días desde epoch para el slider
            epoch_date = datetime(1970, 1, 1).date()
            min_days = (min_date - epoch_date).days
            max_days = (max_date - epoch_date).days

            return {"min_days": min_days, "max_days": max_days, "min_date": min_date, "max_date": max_date}
        else:
            # Fallback: usar últimos 12 meses
            today = datetime.now().date()
            year_ago = today - timedelta(days=365)
            epoch_date = datetime(1970, 1, 1).date()

            return {
                "min_days": (year_ago - epoch_date).days,
                "max_days": (today - epoch_date).days,
                "min_date": year_ago,
                "max_date": today,
            }
    except Exception as e:
        logger.error(f"Error obteniendo rango de fechas: {str(e)}")
        # Fallback en caso de error
        today = datetime.now().date()
        year_ago = today - timedelta(days=365)
        epoch_date = datetime(1970, 1, 1).date()

        return {
            "min_days": (year_ago - epoch_date).days,
            "max_days": (today - epoch_date).days,
            "min_date": year_ago,
            "max_date": today,
        }


def create_generic_analysis_dashboard(pharmacy_id: str = None) -> html.Div:
    """
    Dashboard de análisis de genéricos usando medidas Power BI-style.

    Args:
        pharmacy_id: ID de la farmacia (si None, usa farmacia activa)

    Returns:
        html.Div con dashboard completo
    """

    if pharmacy_id is None or pharmacy_id == "dynamic":
        pharmacy_id = get_current_pharmacy_id()

    return html.Div(
        id="generic-analysis-dashboard",
        children=[
            # Header moderno
            create_modern_header(pharmacy_id),
            html.Br(),
            # Panel de control unificado
            create_unified_control_panel(),
            html.Br(),
            # Store para datos cacheados
            dcc.Store(id="generic-analysis-cache"),
            # Store para datos del rango de fechas
            dcc.Store(id="date-range-store"),
            # Contenedor principal que se actualiza
            html.Div(id="generic-analysis-content", children=[create_loading_state()]),
        ],
    )


def create_modern_header(pharmacy_id: str) -> dbc.Card:
    """Header moderno con KPIs en tiempo real"""

    return dbc.Card(
        [
            dbc.CardHeader(
                [
                    html.H3(
                        [
                            html.I(className="fas fa-chart-line me-2 text-primary"),
                            "Análisis Avanzado de Genéricos",
                            dbc.Badge("ACTIVO", color="success", className="ms-2"),
                        ],
                        className="mb-0",
                    )
                ]
            ),
            dbc.CardBody(
                [
                    # KPIs principales - DATOS REALES DE LA API
                    html.Div(id="header-kpis", children=[load_kpis_directly(pharmacy_id)]),
                    html.Hr(),
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.H5("🚀 Sistema Avanzado", className="text-success"),
                                    html.P(
                                        [
                                            "Arquitectura Power BI-style con ",
                                            html.Strong("cache automático"),
                                            " y medidas reutilizables para máximo rendimiento.",
                                        ]
                                    ),
                                ],
                                width=6,
                            ),
                            dbc.Col(
                                [
                                    html.H5("📊 Análisis Integral", className="text-info"),
                                    html.P(
                                        [
                                            "Dashboard unificado con ",
                                            html.Strong("17 medidas especializadas"),
                                            " para análisis completo de oportunidades.",
                                        ]
                                    ),
                                ],
                                width=6,
                            ),
                        ]
                    ),
                ]
            ),
        ],
        className="shadow-sm mb-4",
    )


def create_kpi_placeholder_row() -> dbc.Row:
    """Fila de KPIs con placeholder para carga inicial"""

    return dbc.Row(
        [
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-euro-sign me-2 text-success"),
                                            dbc.Placeholder(size="lg", style={"width": "80px"}),
                                        ]
                                    ),
                                    html.P("Total Ventas", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-pills me-2 text-info"),
                                            dbc.Placeholder(size="lg", style={"width": "60px"}),
                                        ]
                                    ),
                                    html.P("% Genéricos", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-handshake me-2 text-warning"),
                                            dbc.Placeholder(size="lg", style={"width": "60px"}),
                                        ]
                                    ),
                                    html.P("% Partners", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-piggy-bank me-2 text-danger"),
                                            dbc.Placeholder(size="lg", style={"width": "80px"}),
                                        ]
                                    ),
                                    html.P("Ahorro Potencial", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
        ],
        className="g-3 mb-3",
    )


def create_unified_control_panel() -> dbc.Card:
    """Panel de control unificado con filtros y parámetros"""

    return dbc.Card(
        [
            dbc.CardHeader(
                [html.H5([html.I(className="fas fa-cogs me-2"), "Panel de Control Avanzado"], className="mb-0")]
            ),
            dbc.CardBody(
                [
                    dbc.Row(
                        [
                            dbc.Col(
                                [
                                    html.Label("Período de Análisis:", className="fw-bold"),
                                    html.Div(
                                        [
                                            html.I(className="fas fa-calendar me-2 text-primary"),
                                            html.Span(id="date-range-display", className="text-muted"),
                                        ],
                                        className="mb-2",
                                    ),
                                    dcc.RangeSlider(
                                        id="analysis-date-range-slider",
                                        tooltip={"placement": "bottom", "always_visible": False},
                                        marks=None,  # Se definirán dinámicamente
                                        pushable=7,  # Mínimo 7 días entre fechas
                                    ),
                                    dbc.FormText("Arrastra para seleccionar el período de análisis", color="muted"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Label("Simulación de Descuento:", className="fw-bold"),
                                    dcc.Slider(
                                        id="discount-slider",
                                        min=10,
                                        max=50,
                                        step=5,
                                        value=25,
                                        marks={i: f"{i}%" for i in range(10, 51, 10)},
                                        tooltip={"placement": "bottom", "always_visible": True},
                                    ),
                                    html.Div(id="discount-display", className="text-center text-primary fw-bold mt-2"),
                                ],
                                width=4,
                            ),
                            dbc.Col(
                                [
                                    html.Label("Conjuntos a Mostrar:", className="fw-bold"),
                                    dbc.Select(
                                        id="top-groups-select",
                                        options=[
                                            {"label": "Top 10", "value": 10},
                                            {"label": "Top 15", "value": 15},
                                            {"label": "Top 20", "value": 20},
                                            {"label": "Top 25", "value": 25},
                                        ],
                                        value=20,
                                    ),
                                    dbc.Button(
                                        html.Span([html.I(className="fas fa-sync-alt me-2"), "Actualizar"]),
                                        id="refresh-analysis-btn",
                                        color="primary",
                                        className="mt-2 w-100",
                                    ),
                                ],
                                width=4,
                            ),
                        ]
                    )
                ]
            ),
        ],
        className="shadow-sm",
    )


def create_loading_state() -> html.Div:
    """Estado de carga con animaciones modernas"""

    return html.Div(
        [
            dbc.Alert(
                [
                    html.H4(
                        [html.I(className="fas fa-rocket me-2"), "Cargando Análisis Avanzado..."],
                        className="alert-heading",
                    ),
                    html.P(
                        [
                            "Ejecutando medidas Power BI-style con cache inteligente...",
                            html.Br(),
                            html.Small("Sistema 5x más rápido que la versión anterior", className="text-muted"),
                        ],
                        className="mb-0",
                    ),
                ],
                color="info",
            ),
            html.Br(),
            # Placeholders más modernos
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([dbc.Placeholder(size="sm")]),
                                    dbc.CardBody([dbc.Placeholder(size="lg", style={"height": "250px"})]),
                                ]
                            )
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([dbc.Placeholder(size="sm")]),
                                    dbc.CardBody([dbc.Placeholder(size="lg", style={"height": "250px"})]),
                                ]
                            )
                        ],
                        width=6,
                    ),
                ],
                className="mb-4",
            ),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader([dbc.Placeholder(size="sm")]),
                                    dbc.CardBody([dbc.Placeholder(size="lg", style={"height": "300px"})]),
                                ]
                            )
                        ],
                        width=12,
                    )
                ]
            ),
        ]
    )


def create_dashboard_sections(dashboard_data: Dict[str, Any]) -> List[html.Div]:
    """Crear secciones del dashboard usando datos avanzados"""

    sections = []

    # Verificar que dashboard_data no sea None
    if not dashboard_data or not isinstance(dashboard_data, dict):
        return sections

    # Sección 1: KPIs y Ratio Genéricos
    if "kpis_generales" in dashboard_data and "analisis_genericos" in dashboard_data:
        sections.append(create_kpis_section(dashboard_data["kpis_generales"], dashboard_data["analisis_genericos"]))
        sections.append(html.Hr())

    # Sección 2: Partners Analysis
    if "partners" in dashboard_data:
        sections.append(create_partners_section(dashboard_data["partners"]))
        sections.append(html.Hr())

    # Sección 3: Conjuntos Homogéneos
    if "conjuntos_homogeneos" in dashboard_data:
        sections.append(create_homogeneous_groups_section(dashboard_data["conjuntos_homogeneos"]))
        sections.append(html.Hr())

    # Sección 4: Análisis Terapéutico
    if "categorias_terapeuticas" in dashboard_data:
        sections.append(create_therapeutic_section(dashboard_data["categorias_terapeuticas"]))

    return sections


def create_kpis_section(kpis_data: Dict, generics_data: Dict) -> html.Div:
    """Sección de KPIs principales con visualización avanzada"""

    total_ventas = kpis_data.get("total_ventas", 0)
    total_unidades = kpis_data.get("total_unidades", 0)

    ratio_data = generics_data.get("ratio_genericos_marca", {})
    savings_data = generics_data.get("oportunidades_ahorro", {})

    generic_percentage = ratio_data.get("generic_sales", {}).get("percentage_amount", 0)
    total_opportunities = savings_data.get("total_opportunities", 0)
    potential_savings = savings_data.get("total_potential_savings", 0)

    # Gráfico de donut para ratio genéricos/marca
    donut_fig = create_generic_ratio_donut(ratio_data)

    # Gráfico de barras para oportunidades
    opportunities_fig = create_opportunities_bar_chart(savings_data)

    return html.Div(
        [
            html.H4(
                [html.I(className="fas fa-tachometer-alt me-2 text-primary"), "KPIs Principales"], className="mb-3"
            ),
            # KPIs principales
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H3(
                                                format_currency(total_ventas, decimals=0), className="text-success mb-1"
                                            ),
                                            html.P("Ventas Totales", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H3(format_number(total_unidades), className="text-info mb-1"),
                                            html.P("Unidades Vendidas", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H3(
                                                f"{generic_percentage:.1f}%".replace(".", ","),
                                                className="text-warning mb-1",
                                            ),
                                            html.P("Ventas Genéricos", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H3(
                                                format_currency(potential_savings, decimals=0),
                                                className="text-danger mb-1",
                                            ),
                                            html.P("Ahorro Potencial", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                ],
                className="g-3 mb-4",
            ),
            # Visualizaciones
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [
                                                    html.I(className="fas fa-chart-pie me-2"),
                                                    "Distribución Genéricos vs Marca",
                                                ],
                                                className="mb-0",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody([dcc.Graph(figure=donut_fig, style={"height": "300px"})]),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=6,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [html.I(className="fas fa-chart-bar me-2"), "Oportunidades de Ahorro"],
                                                className="mb-0",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody([dcc.Graph(figure=opportunities_fig, style={"height": "300px"})]),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=6,
                    ),
                ]
            ),
        ]
    )


def create_partners_section(partners_data: Dict) -> html.Div:
    """Sección de análisis de partners"""

    if not partners_data or "ventas_partners" not in partners_data:
        return dbc.Alert("No hay datos de partners disponibles", color="warning")

    partner_sales_data = partners_data["ventas_partners"]

    if "error" in partner_sales_data:
        return dbc.Alert(f"Error en datos de partners: {partner_sales_data['error']}", color="warning")

    partner_sales = partner_sales_data.get("partner_sales", 0)
    total_sales = partner_sales_data.get("total_sales", 0)
    partner_percentage = partner_sales_data.get("partner_percentage", 0)
    other_sales = partner_sales_data.get("other_sales", 0)

    # Gráfico de comparación
    comparison_fig = create_partners_comparison_chart(partner_sales_data)

    return html.Div(
        [
            html.H4(
                [html.I(className="fas fa-handshake me-2 text-warning"), "Análisis de Laboratorios Partners"],
                className="mb-3",
            ),
            # KPIs partners
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                format_currency(partner_sales, decimals=0),
                                                className="text-success mb-1",
                                            ),
                                            html.P("Ventas Partners", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                f"{partner_percentage:.1f}%".replace(".", ","),
                                                className="text-warning mb-1",
                                            ),
                                            html.P("% del Total", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=4,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                format_currency(other_sales, decimals=0),
                                                className="text-secondary mb-1",
                                            ),
                                            html.P("Otros Laboratorios", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=4,
                    ),
                ],
                className="g-3 mb-4",
            ),
            # Visualización
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [
                                                    html.I(className="fas fa-chart-bar me-2"),
                                                    "Comparación Partners vs Otros",
                                                ],
                                                className="mb-0",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody([dcc.Graph(figure=comparison_fig, style={"height": "350px"})]),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                    )
                ]
            ),
        ]
    )


def create_homogeneous_groups_section(groups_data: Dict) -> html.Div:
    """Sección de conjuntos homogéneos"""

    if not groups_data or "groups_analysis" not in groups_data:
        return dbc.Alert("No hay datos de conjuntos homogéneos", color="warning")

    groups = groups_data["groups_analysis"]
    summary = groups_data.get("summary", {})

    # Tabla de grupos
    groups_table = create_groups_table(groups)

    # Gráfico scatter
    scatter_fig = create_groups_scatter_chart(groups)

    return html.Div(
        [
            html.H4(
                [html.I(className="fas fa-th-large me-2 text-success"), "Conjuntos Homogéneos con Mayor Oportunidad"],
                className="mb-3",
            ),
            # Resumen
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(f"{summary.get('total_groups', 0)}", className="text-primary mb-1"),
                                            html.P("Conjuntos Analizados", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                format_currency(summary.get("total_sales_analyzed", 0), decimals=0),
                                                className="text-info mb-1",
                                            ),
                                            html.P("Ventas Analizadas", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                format_currency(summary.get("total_potential_savings", 0), decimals=0),
                                                className="text-success mb-1",
                                            ),
                                            html.P("Ahorro Potencial", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardBody(
                                        [
                                            html.H4(
                                                f"{summary.get('avg_savings_percentage', 0):.1f}%".replace(".", ","),
                                                className="text-warning mb-1",
                                            ),
                                            html.P("% Ahorro Promedio", className="text-muted mb-0"),
                                        ]
                                    )
                                ],
                                className="text-center shadow-sm",
                            )
                        ],
                        width=3,
                    ),
                ],
                className="g-3 mb-4",
            ),
            # Visualización y tabla
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [
                                                    html.I(className="fas fa-chart-scatter me-2"),
                                                    "Mapa de Oportunidades",
                                                ],
                                                className="mb-0",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody([dcc.Graph(figure=scatter_fig, style={"height": "400px"})]),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=7,
                    ),
                    dbc.Col([groups_table], width=5),
                ]
            ),
        ]
    )


def create_therapeutic_section(therapeutic_data: Dict) -> html.Div:
    """Sección de análisis terapéutico"""

    if not therapeutic_data or "distribucion_ventas" not in therapeutic_data:
        return dbc.Alert("No hay datos de análisis terapéutico", color="warning")

    distribution = therapeutic_data["distribucion_ventas"]
    categories = distribution.get("categories_analysis", [])
    summary = distribution.get("summary", {})

    # Gráfico de barras horizontales
    therapeutic_fig = create_therapeutic_bar_chart(categories)

    return html.Div(
        [
            html.H4(
                [html.I(className="fas fa-pills me-2 text-info"), "Distribución por Categorías Terapéuticas"],
                className="mb-3",
            ),
            dbc.Row(
                [
                    dbc.Col(
                        [
                            dbc.Card(
                                [
                                    dbc.CardHeader(
                                        [
                                            html.H5(
                                                [html.I(className="fas fa-chart-bar me-2"), "Top Categorías ATC"],
                                                className="mb-0",
                                            )
                                        ]
                                    ),
                                    dbc.CardBody([dcc.Graph(figure=therapeutic_fig, style={"height": "400px"})]),
                                ],
                                className="shadow-sm",
                            )
                        ],
                        width=12,
                    )
                ]
            ),
        ]
    )


# ==================== FUNCIONES AUXILIARES DE GRÁFICOS ====================


def create_generic_ratio_donut(ratio_data: Dict) -> go.Figure:
    """Crear gráfico donut para ratio genéricos/marca"""

    generic_amount = ratio_data.get("generic_sales", {}).get("amount", 0)
    branded_amount = ratio_data.get("branded_sales", {}).get("amount", 0)

    fig = go.Figure(
        data=[
            go.Pie(
                labels=["Genéricos", "Marca"],
                values=[generic_amount, branded_amount],
                hole=0.4,
                marker_colors=["#28a745", "#6c757d"],
            )
        ]
    )

    fig.update_traces(textposition="inside", textinfo="percent+label")
    fig.update_layout(showlegend=True, margin=dict(t=20, b=20, l=20, r=20))

    return fig


def create_opportunities_bar_chart(savings_data: Dict) -> go.Figure:
    """Crear gráfico de barras para oportunidades"""

    opportunities = savings_data.get("opportunities_detail", [])[:10]  # Top 10

    if not opportunities:
        fig = go.Figure()
        fig.add_annotation(
            text="No hay oportunidades disponibles", xref="paper", yref="paper", x=0.5, y=0.5, showarrow=False
        )
        return fig

    productos = [
        (
            opp.get("branded_product", "")[:20] + "..."
            if len(opp.get("branded_product", "")) > 20
            else opp.get("branded_product", "")
        )
        for opp in opportunities
    ]
    ahorros = [opp.get("total_savings", 0) for opp in opportunities]

    fig = go.Figure(data=[go.Bar(x=ahorros, y=productos, orientation="h", marker_color="#fd7e14")])

    fig.update_layout(xaxis_title="Ahorro Potencial (€)", margin=dict(t=20, b=20, l=20, r=20), height=300)

    return fig


def create_partners_comparison_chart(partner_data: Dict) -> go.Figure:
    """Crear gráfico de comparación partners vs otros"""

    partner_sales = partner_data.get("partner_sales", 0)
    other_sales = partner_data.get("other_sales", 0)

    fig = go.Figure(
        data=[
            go.Bar(
                x=["Laboratorios Partners", "Otros Laboratorios"],
                y=[partner_sales, other_sales],
                marker_color=["#28a745", "#6c757d"],
                text=[format_currency(partner_sales, decimals=0), format_currency(other_sales, decimals=0)],
                textposition="auto",
            )
        ]
    )

    fig.update_layout(yaxis_title="Ventas (€)", showlegend=False, margin=dict(t=20, b=20, l=20, r=20))

    return fig


def create_groups_scatter_chart(groups: List[Dict]) -> go.Figure:
    """Crear scatter chart para conjuntos homogéneos"""

    if not groups:
        fig = go.Figure()
        fig.add_annotation(text="No hay grupos disponibles", xref="paper", yref="paper", x=0.5, y=0.5, showarrow=False)
        return fig

    x_sales = [g.get("total_sales", 0) for g in groups]
    y_savings = [g.get("potential_savings", 0) for g in groups]
    texts = [g.get("homogeneous_group", "")[:15] for g in groups]

    fig = go.Figure(
        data=[
            go.Scatter(
                x=x_sales,
                y=y_savings,
                mode="markers",
                marker=dict(size=10, color="#17a2b8", opacity=0.7),
                text=texts,
                hovertemplate="Grupo: %{text}<br>Ventas: %{x:,.0f}€<br>Ahorro: %{y:,.0f}€<extra></extra>",
            )
        ]
    )

    fig.update_layout(
        xaxis_title="Ventas Actuales (€)", yaxis_title="Ahorro Potencial (€)", margin=dict(t=20, b=20, l=20, r=20)
    )

    return fig


def create_therapeutic_bar_chart(categories: List[Dict]) -> go.Figure:
    """Crear gráfico de barras para categorías terapéuticas"""

    if not categories:
        fig = go.Figure()
        fig.add_annotation(
            text="No hay categorías disponibles", xref="paper", yref="paper", x=0.5, y=0.5, showarrow=False
        )
        return fig

    # Top 10 categorías
    top_categories = sorted(categories, key=lambda x: x.get("total_amount", 0), reverse=True)[:10]

    category_names = [c.get("therapeutic_category", c.get("atc_group", ""))[:30] for c in top_categories]
    amounts = [c.get("total_amount", 0) for c in top_categories]
    percentages = [c.get("percentage_of_total", 0) for c in top_categories]

    fig = go.Figure(
        data=[
            go.Bar(
                y=category_names,
                x=amounts,
                orientation="h",
                marker_color="#6f42c1",
                text=[f"{p:.1f}%".replace(".", ",") for p in percentages],
                textposition="auto",
            )
        ]
    )

    fig.update_layout(xaxis_title="Ventas (€)", margin=dict(t=20, b=40, l=200, r=20))

    return fig


def create_groups_table(groups: List[Dict]) -> dbc.Card:
    """Crear tabla de grupos homogéneos"""

    if not groups:
        return dbc.Card([dbc.CardBody([dbc.Alert("No hay grupos disponibles", color="info")])])

    # Top 15 grupos para la tabla
    top_groups = groups[:15]

    table_data = []
    for i, group in enumerate(top_groups):
        table_data.append(
            {
                "#": i + 1,
                "Grupo": group.get("homogeneous_group", "")[:20],
                "Ventas (€)": f"{group.get('total_sales', 0):,.0f}",
                "Ahorro (€)": f"{group.get('potential_savings', 0):,.0f}",
                "% Ahorro": f"{group.get('savings_percentage', 0):.1f}%".replace(".", ","),
                "Labs": group.get("laboratories_count", 0),
            }
        )

    return dbc.Card(
        [
            dbc.CardHeader([html.H5([html.I(className="fas fa-table me-2"), "Top Oportunidades"], className="mb-0")]),
            dbc.CardBody(
                [
                    dash_table.DataTable(
                        data=table_data,
                        columns=[
                            {"name": "#", "id": "#", "type": "numeric"},
                            {"name": "Grupo", "id": "Grupo"},
                            {"name": "Ventas", "id": "Ventas (€)"},
                            {"name": "Ahorro", "id": "Ahorro (€)"},
                            {"name": "%", "id": "% Ahorro"},
                            {"name": "Labs", "id": "Labs"},
                        ],
                        style_table={"height": "400px", "overflowY": "auto"},
                        style_cell={"fontSize": "11px", "textAlign": "left"},
                        style_header={"backgroundColor": "#f8f9fa", "fontWeight": "bold"},
                        style_data_conditional=[
                            {
                                "if": {"column_id": "% Ahorro", "filter_query": "{% Ahorro} > 15"},
                                "backgroundColor": "#d4edda",
                            }
                        ],
                    )
                ]
            ),
        ],
        className="shadow-sm",
    )


# ==================== CALLBACKS ====================


@callback(Output("discount-display", "children"), Input("discount-slider", "value"))
def update_discount_display(discount_value):
    """Actualizar display del descuento"""
    return f"Descuento: {discount_value}%"


@callback(
    [
        Output("analysis-date-range-slider", "min"),
        Output("analysis-date-range-slider", "max"),
        Output("analysis-date-range-slider", "value"),
        Output("analysis-date-range-slider", "marks"),
        Output("date-range-store", "data"),
    ],
    Input("generic-analysis-dashboard", "id"),
)
def initialize_date_range_slider(dashboard_id):
    """Inicializar el RangeSlider con el rango real de fechas de datos"""
    try:
        from utils.api_client import APIClient
        from utils.pharmacy_context import get_current_pharmacy_id

        pharmacy_id = get_current_pharmacy_id()
        logger.info(f"Initializing slider for pharmacy: {pharmacy_id}")
        api_client = APIClient()

        # Obtener rango de fechas reales
        response = api_client.backend_client.get_sales_date_range(pharmacy_id)
        logger.info(
            f"API response: success={getattr(response, 'success', None)}, data={getattr(response, 'data', None)}"
        )

        if response and response.success and response.data:
            min_date_str = response.data.get("min_date")
            max_date_str = response.data.get("max_date")

            if min_date_str and max_date_str:
                # Convertir fechas directamente sin función auxiliar
                from datetime import datetime

                min_date = datetime.fromisoformat(min_date_str).date()
                max_date = datetime.fromisoformat(max_date_str).date()

                epoch_date = datetime(1970, 1, 1).date()
                min_value = (min_date - epoch_date).days
                max_value = (max_date - epoch_date).days

                date_range = {
                    "min_value": min_value,
                    "max_value": max_value,
                    "min_date": min_date_str,
                    "max_date": max_date_str,
                }

                # Crear marcas para el slider (cada 30 días aproximadamente)
                marks = {}
                total_days = date_range["max_value"] - date_range["min_value"]
                logger.info(
                    f"Creating slider marks: total_days={total_days}, min={date_range['min_value']}, max={date_range['max_value']}"
                )
                if total_days > 90:  # Solo mostrar marcas si hay más de 3 meses
                    step = max(1, total_days // 6)  # Máximo 6 marcas
                    for i in range(0, 6):
                        day_value = date_range["min_value"] + (i * step)
                        if day_value <= date_range["max_value"]:
                            date_obj = datetime(1970, 1, 1) + timedelta(days=int(day_value))
                            marks[day_value] = date_obj.strftime("%d/%m")
                    logger.info(f"Created marks: {marks}")
                else:
                    # Para rangos pequeños, crear marcas básicas
                    start_date_obj = datetime(1970, 1, 1) + timedelta(days=int(date_range["min_value"]))
                    end_date_obj = datetime(1970, 1, 1) + timedelta(days=int(date_range["max_value"]))
                    marks[date_range["min_value"]] = start_date_obj.strftime("%d/%m")
                    marks[date_range["max_value"]] = end_date_obj.strftime("%d/%m")
                    logger.info(f"Created basic marks: {marks}")

                # Valor inicial: todo el rango disponible
                start_value = date_range["min_value"]
                end_value = date_range["max_value"]

                return (date_range["min_value"], date_range["max_value"], [start_value, end_value], marks, date_range)

        # Fallback si no hay datos
        fallback_start = (datetime.now() - timedelta(days=90)).timetuple().tm_yday
        fallback_end = datetime.now().timetuple().tm_yday
        return (
            fallback_start,
            fallback_end,
            [fallback_end - 30, fallback_end],
            {},
            {
                "min_value": fallback_start,
                "max_value": fallback_end,
                "min_date": (datetime.now() - timedelta(days=90)).strftime("%Y-%m-%d"),
                "max_date": datetime.now().strftime("%Y-%m-%d"),
            },
        )

    except Exception as e:
        logger.error(f"Error inicializando date range slider: {e}")
        # Fallback seguro
        fallback_start = 1
        fallback_end = 365
        return (
            fallback_start,
            fallback_end,
            [fallback_end - 30, fallback_end],
            {},
            {"min_value": fallback_start, "max_value": fallback_end},
        )


@callback(
    Output("date-range-display", "children"),
    [Input("analysis-date-range-slider", "value")],
    [State("date-range-store", "data")],
)
def update_date_range_display(slider_values, date_range_data):
    """Actualizar display del rango de fechas seleccionado"""
    try:
        if not slider_values or not date_range_data:
            return "Selecciona un período"

        start_value, end_value = slider_values

        # Convertir valores del slider a fechas usando epoch
        epoch_date = datetime(1970, 1, 1)
        start_date = epoch_date + timedelta(days=int(start_value))
        end_date = epoch_date + timedelta(days=int(end_value))

        # Formatear fechas
        start_str = start_date.strftime("%d/%m/%Y")
        end_str = end_date.strftime("%d/%m/%Y")

        return f"{start_str} - {end_str}"

    except Exception as e:
        return "Error en fechas"


# TEMPORAL: Comentado para evitar que sobrescriba los KPIs estáticos
# @callback(
#     Output("header-kpis", "children"),
#     Output("generic-analysis-content", "children"),
#     [Input("refresh-analysis-btn", "n_clicks"),
#      Input("date-range-store", "data")],
#     [
#         State("analysis-date-range-slider", "value"),
#         State("date-range-store", "data"),
#         State("discount-slider", "value"),
#         State("top-groups-select", "value")
#     ],
#     prevent_initial_call=False  # TEMPORAL: permitir ejecución inicial para diagnosticar
# )
# TEMPORAL: Función comentada junto con el callback
# def update_analysis(n_clicks, date_store_trigger, slider_values, date_range_data, discount_percentage, top_groups):
#     """
#     Callback principal que usa la arquitectura de medidas
#     """
#
#     try:
#         pharmacy_id = get_current_pharmacy_id()
#         api_client = APIClient()
#
#         # Construir parámetros desde el RangeSlider
#         params = {}
#         if slider_values and date_range_data:
#             try:
#                 start_value, end_value = slider_values
#                 # Convertir valores del slider a fechas usando epoch (consistente con inicialización)
#                 epoch_date = datetime(1970, 1, 1)
#                 start_date = epoch_date + timedelta(days=int(start_value))
#                 end_date = epoch_date + timedelta(days=int(end_value))
#
#                 params["start_date"] = start_date.strftime('%Y-%m-%d')
#                 params["end_date"] = end_date.strftime('%Y-%m-%d')
#             except Exception as e:
#                 logger.warning(f"Error convirtiendo fechas del slider: {e}")
#                 # Usar fechas por defecto si hay error
#                 pass
#
#         # Llamar al dashboard usando medidas
#         logger.info(f"Calling API with pharmacy_id: {pharmacy_id}, params: {params}")
#         dashboard_data = api_client.get(f"/api/generic-analysis/dashboard/{pharmacy_id}", params=params)
#         logger.info(f"API returned: {type(dashboard_data)}, content: {dashboard_data if isinstance(dashboard_data, dict) and len(str(dashboard_data)) < 200 else 'Data too long'}")
#
#         if not dashboard_data or dashboard_data is None:
#             logger.warning(f"Dashboard data is empty or None: {dashboard_data}")
#             return (
#                 create_kpi_placeholder_row(),
#                 dbc.Alert("No se pudieron cargar los datos del dashboard", color="warning")
#             )
#
#         # TEMPORALMENTE COMENTADO: Simplificar para diagnosticar timeouts
#         # También obtener datos específicos para conjuntos homogéneos con parámetros
#         # groups_params = {
#         #     **params,
#         #     "discount_percentage": discount_percentage or 25,
#         #     "top_groups": top_groups or 20
#         # }
#         # groups_data = api_client.get(f"/api/generic-analysis/homogeneous-groups/{pharmacy_id}", params=groups_params)
#         #
#         # # Integrar datos de grupos en el dashboard
#         # if groups_data and isinstance(groups_data, dict) and 'conjuntos_homogeneos' in groups_data:
#         #     dashboard_data['conjuntos_homogeneos'] = groups_data['conjuntos_homogeneos']
#
#         # Crear KPIs actualizados
#         updated_kpis = create_updated_kpis_row(dashboard_data)
#
#         # Crear secciones del dashboard
#         content = create_dashboard_sections(dashboard_data)
#
#         if not content:
#             content = [dbc.Alert("No hay datos suficientes para generar el análisis", color="info")]
#
#         return updated_kpis, content
#
#     except Exception as e:
#         logger.error(f"Error en análisis: {str(e)}")
#         return (
#             create_kpi_placeholder_row(),
#             dbc.Alert([
#                 html.H4("Error al cargar análisis", className="alert-heading"),
#                 html.P(f"Error: {str(e)}")
#             ], color="danger")
#         )


def create_updated_kpis_row(dashboard_data: Dict) -> dbc.Row:
    """Crear fila de KPIs actualizada con datos reales"""

    if not dashboard_data or dashboard_data is None:
        logger.error("create_updated_kpis_row received None or empty data")
        return create_kpi_placeholder_row()

    try:
        kpis = dashboard_data.get("kpis_generales", {}) or {}
        analysis = dashboard_data.get("analisis_genericos", {}) or {}
        partners = dashboard_data.get("partners", {}) or {}

        # Convertir a números de manera robusta
        def safe_float(value, default=0):
            try:
                if value is None:
                    return default
                # Si ya es número
                if isinstance(value, (int, float)):
                    return float(value)
                # Si es string, limpiar y convertir
                cleaned = str(value).replace("%", "").replace(",", ".").strip()
                return float(cleaned) if cleaned else default
            except:
                return default

        total_ventas = safe_float(kpis.get("total_ventas", 0))

        generic_ratio = analysis.get("ratio_genericos_marca", {}) or {}
        generic_sales = generic_ratio.get("generic_sales", {}) or {}
        generic_percentage = safe_float(generic_sales.get("percentage_amount", 0))

        partner_sales_data = partners.get("ventas_partners", {}) or {}
        partner_percentage = safe_float(partner_sales_data.get("partner_percentage", 0))

        savings_data = analysis.get("oportunidades_ahorro", {}) or {}
        potential_savings = safe_float(savings_data.get("total_potential_savings", 0))
    except Exception as e:
        logger.error(f"Error processing KPI data: {str(e)}")
        return create_kpi_placeholder_row()

    return dbc.Row(
        [
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-euro-sign me-2 text-success"),
                                            format_currency(total_ventas, decimals=0),
                                        ],
                                        className="text-success",
                                    ),
                                    html.P("Total Ventas", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-pills me-2 text-info"),
                                            f"{generic_percentage:.1f}%".replace(".", ","),
                                        ],
                                        className="text-info",
                                    ),
                                    html.P("% Genéricos", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-handshake me-2 text-warning"),
                                            f"{partner_percentage:.1f}%".replace(".", ","),
                                        ],
                                        className="text-warning",
                                    ),
                                    html.P("% Partners", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
            dbc.Col(
                [
                    dbc.Card(
                        [
                            dbc.CardBody(
                                [
                                    html.H4(
                                        [
                                            html.I(className="fas fa-piggy-bank me-2 text-danger"),
                                            format_currency(potential_savings, decimals=0),
                                        ],
                                        className="text-danger",
                                    ),
                                    html.P("Ahorro Potencial", className="mb-0"),
                                ]
                            )
                        ],
                        className="text-center shadow-sm",
                    )
                ],
                width=3,
            ),
        ],
        className="g-3 mb-3",
    )
